<?php
session_start();
require_once 'config/database.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

require_once 'includes/header.php';
require_once 'includes/sidebar.php';
?>

<div class="main-container">
    <div class="header">
        <h1>Nova Troca de Cheques</h1>
        <div class="user-info">
            <i class="fas fa-user"></i> Bem-vindo, <?php echo isset($_SESSION['user_name']) ? $_SESSION['user_name'] : 'Usuário'; ?>
        </div>
    </div>

    <div class="main-content">
        <!-- Formulário de Nova Troca -->
        <div class="card">
            <form id="formCheque" method="post">
                <div class="form-grid">
                    <div class="form-group">
                        <label for="cliente">Nome do Cliente</label>
                        <input type="text" id="cliente" name="cliente" required>
                    </div>
                    <div class="form-group">
                        <label for="numero_cheque">Número do Cheque</label>
                        <input type="text" id="numero_cheque" name="numero_cheque">
                    </div>
                    <div class="form-group">
                        <label for="banco">Banco</label>
                        <input type="text" id="banco" name="banco">
                    </div>
                    <div class="form-group">
                        <label for="valor">Valor do Cheque</label>
                        <input type="text" id="valor" name="valor" class="money" required>
                    </div>
                    <div class="form-group">
                        <label for="taxaJuros">Taxa de Juros (%)</label>
                        <input type="text" id="taxaJuros" name="taxaJuros" class="form-control taxa-juros" placeholder="Ex: 3,5" required>
                    </div>
                    <div class="form-group">
                        <label for="data_troca">Data da Troca</label>
                        <input type="date" id="data_troca" name="data_troca" value="<?php echo date('Y-m-d'); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="data_vencimento">Data de Vencimento</label>
                        <input type="date" id="data_vencimento" name="data_vencimento" required>
                    </div>
                </div>

                <div class="form-actions">
                    <button type="button" class="btn-calcular">
                        <i class="fas fa-calculator"></i> Calcular
                    </button>
                    <button type="button" class="btn-salvar" onclick="salvarCheque()" style="display: none;">
                        <i class="fas fa-save"></i> Confirmar e Salvar
                    </button>
                </div>

                <div id="resultado" class="resultado-calculo" style="display: none;">
                    <h3>Resultado do Cálculo</h3>
                    <table>
                        <tr>
                            <td>Valor do Cheque:</td>
                            <td id="valorChequeResult">R$ 0,00</td>
                        </tr>
                        <tr>
                            <td>Prazo:</td>
                            <td id="diasResult">0 dias</td>
                        </tr>
                        <tr>
                            <td>Juros:</td>
                            <td id="jurosResult">R$ 0,00</td>
                        </tr>
                        <tr>
                            <td>Valor Líquido:</td>
                            <td id="valorLiquidoResult">R$ 0,00</td>
                        </tr>
                    </table>
                </div>
            </form>
        </div>

        <!-- Lista de Cheques Recentes -->
        <div class="card">
            <h2>Cheques Recentes</h2>
            <?php
                // Configuração da paginação
                $registrosPorPagina = 10;
                $paginaAtual = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
                $offset = ($paginaAtual - 1) * $registrosPorPagina;

                // Conta total de registros
                $database = new Database();
                $pdo = $database->getConnection();
                $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM cheques WHERE user_id = :user_id");
                $stmt->execute(['user_id' => $_SESSION['user_id']]);
                $totalRegistros = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
                $totalPaginas = ceil($totalRegistros / $registrosPorPagina);
            ?>
            <div class="table-responsive">
                <table id="chequesTable">
                    <thead>
                        <tr>
                            <th>Cliente</th>
                            <th>Nº Cheque</th>
                            <th>Banco</th>
                            <th>Valor</th>
                            <th>Data Troca</th>
                            <th>Vencimento</th>
                            <th>Dias</th>
                            <th>Taxa Juros</th>
                            <th>Juros</th>
                            <th>Valor Líquido</th>
                            <th>Status</th>
                            <th>Ações</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php
                    try {
                        $stmt = $pdo->prepare("
                            SELECT * FROM cheques 
                            WHERE user_id = :user_id 
                            ORDER BY data_troca DESC, id DESC
                            LIMIT :offset, :limit
                        ");
                        $stmt->bindValue(':user_id', $_SESSION['user_id'], PDO::PARAM_INT);
                        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
                        $stmt->bindValue(':limit', $registrosPorPagina, PDO::PARAM_INT);
                        $stmt->execute();
                        $cheques = $stmt->fetchAll(PDO::FETCH_ASSOC);

                        foreach($cheques as $cheque) {
                            $statusClass = $cheque['status_pagamento'] === 'pago' ? 'status-pago' : 'status-pendente';
                            echo "<tr>";
                            echo "<td>{$cheque['cliente']}</td>";
                            echo "<td>{$cheque['numero_cheque']}</td>";
                            echo "<td>{$cheque['banco']}</td>";
                            echo "<td class='text-end'>R$ " . number_format($cheque['valor'], 2, ',', '.') . "</td>";
                            echo "<td>" . date('d/m/Y', strtotime($cheque['data_troca'])) . "</td>";
                            echo "<td>" . date('d/m/Y', strtotime($cheque['data_vencimento'])) . "</td>";
                            echo "<td>{$cheque['dias']}</td>";
                            echo "<td class='text-end'>R$ " . number_format($cheque['taxa_juros'], 1, ',', '.') . "%" . "</td>";
                            echo "<td class='text-end'>R$ " . number_format($cheque['juros'], 2, ',', '.') . "</td>";
                            echo "<td class='text-end'>R$ " . number_format($cheque['valor_liquido'], 2, ',', '.') . "</td>";
                            echo "<td class='{$statusClass}'>" . ucfirst($cheque['status_pagamento']) . "</td>";
                            echo "<td class='acoes'>";
                            if ($cheque['status_pagamento'] === 'pendente') {
                                echo "<button class='btn-pagar' onclick='quitarCheque({$cheque['id']})'>
                                    <i class='fas fa-check'></i> <span>Quitar</span>
                                </button>";
                                echo "<button class='btn-excluir' onclick='excluirCheque({$cheque['id']})'>
                                    <i class='fas fa-trash'></i> <span>Excluir</span>
                                </button>";
                            } else {
                                echo "<button class='btn-pagar disabled' disabled>
                                    <i class='fas fa-check'></i> <span>Quitado</span>
                                </button>";
                            }
                            echo "</td>";
                            echo "</tr>";
                        }
                    } catch (PDOException $e) {
                        echo "<tr><td colspan='10'>Erro ao carregar os cheques: " . $e->getMessage() . "</td></tr>";
                    }
                    ?>
                    </tbody>
                </table>

                <!-- Paginação -->
                <?php if ($totalPaginas > 1): ?>
                <div class="paginacao">
                    <?php if ($paginaAtual > 1): ?>
                        <a href="?pagina=1" class="btn-pagina">&laquo; Primeira</a>
                        <a href="?pagina=<?php echo $paginaAtual - 1; ?>" class="btn-pagina">&lsaquo; Anterior</a>
                    <?php endif; ?>

                    <?php
                    // Mostra números das páginas
                    for ($i = max(1, $paginaAtual - 2); $i <= min($totalPaginas, $paginaAtual + 2); $i++): ?>
                        <a href="?pagina=<?php echo $i; ?>" 
                           class="btn-pagina <?php echo $i == $paginaAtual ? 'ativa' : ''; ?>">
                            <?php echo $i; ?>
                        </a>
                    <?php endfor; ?>

                    <?php if ($paginaAtual < $totalPaginas): ?>
                        <a href="?pagina=<?php echo $paginaAtual + 1; ?>" class="btn-pagina">Próxima &rsaquo;</a>
                        <a href="?pagina=<?php echo $totalPaginas; ?>" class="btn-pagina">Última &raquo;</a>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
console.log('Script carregado');

let calculoAtual = null;

$(document).ready(function() {
    $('.money').mask('#.##0,00', {reverse: true});
    
    // Atualiza a máscara da taxa de juros para aceitar decimais
    $('.taxa-juros').on('input', function(e) {
        // Remove tudo que não for número, ponto ou vírgula
        let value = this.value.replace(/[^\d.,]/g, '');
        
        // Substitui múltiplos pontos ou vírgulas por uma única vírgula
        value = value.replace(/[.,]/g, function(match, offset, string) {
            return string.indexOf('.') === offset || string.indexOf(',') === offset ? ',' : '';
        });
        
        // Limita a dois decimais
        if (value.includes(',')) {
            let parts = value.split(',');
            if (parts[1] && parts[1].length > 2) {
                parts[1] = parts[1].substring(0, 2);
                value = parts.join(',');
            }
        }
        
        this.value = value;
    });
    
    $('#data_troca, #data_vencimento').change(function() {
        const dataVencimento = new Date($('#data_vencimento').val());
        const dataTroca = new Date($('#data_troca').val());
        
        if (dataVencimento < dataTroca) {
            alert('A data de vencimento não pode ser anterior à data da troca!');
            $(this).val('');
        }
    });

    $('.btn-calcular').click(function() {
        console.log('Botão calcular clicado');
        calcularJurosCustomizado();
    });
});

function excluirCheque(chequeId) {
    if (confirm('Tem certeza que deseja excluir este cheque?')) {
        $.ajax({
            url: 'excluir_cheque.php',
            type: 'POST',
            data: { cheque_id: chequeId },
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    alert(response.message);
                    window.location.reload();
                } else {
                    alert('Erro: ' + response.message);
                }
            },
            error: function(xhr, status, error) {
                console.error('Erro na requisição:', error);
                console.error('Status:', status);
                console.error('Resposta:', xhr.responseText);
                alert('Erro ao tentar excluir o cheque. Verifique o console para mais detalhes.');
            }
        });
    }
}

function calcularJurosCustomizado() {
    try {
        // Pega os valores reais dos campos
        const valorString = document.getElementById('valor').value;
        const taxaString = document.getElementById('taxaJuros').value;
        const dataVencimento = new Date(document.getElementById('data_vencimento').value);
        const dataTroca = new Date(document.getElementById('data_troca').value);

        // Validações
        if (!valorString || !taxaString || !dataVencimento || !dataTroca) {
            alert('Por favor, preencha todos os campos');
            return;
        }

        // Conversões
        const valor = parseFloat(valorString.replace(/\./g, '').replace(',', '.'));
        const taxa = parseFloat(taxaString.replace(',', '.')) / 100;
        
        // Calcula dias
        const diffTime = Math.abs(dataVencimento - dataTroca);
        const dias = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

        // Faz o cálculo
        const valorJuros = (valor * taxa * dias) / 30;
        const valorLiquido = valor - valorJuros;

        // Log dos valores
        console.log('=== CÁLCULO CUSTOMIZADO ===');
        console.log('Valor:', valor);
        console.log('Taxa:', taxa * 100 + '%');
        console.log('Dias:', dias);
        console.log('Juros calculado:', valorJuros);
        console.log('Valor líquido:', valorLiquido);
        console.log('========================');

        // Atualiza a interface
        document.getElementById('valorChequeResult').textContent = `R$ ${valor.toFixed(2).replace('.', ',')}`;
        document.getElementById('diasResult').textContent = `${dias} dias`;
        document.getElementById('jurosResult').textContent = `R$ ${valorJuros.toFixed(2).replace('.', ',')}`;
        document.getElementById('valorLiquidoResult').textContent = `R$ ${valorLiquido.toFixed(2).replace('.', ',')}`;

        document.getElementById('resultado').style.display = 'block';
        document.querySelector('.btn-salvar').style.display = 'inline-block';

        calculoAtual = {
            valor: valor,
            dias: dias,
            juros: valorJuros,
            valorLiquido: valorLiquido
        };

    } catch (error) {
        console.log('Erro na função customizada:', error);
    }
}

function salvarCheque() {
    try {
        if (!calculoAtual) {
            alert('Por favor, calcule os juros antes de salvar');
            return;
        }

        const formData = {
            cliente: document.getElementById('cliente').value,
            numero_cheque: document.getElementById('numero_cheque').value,
            banco: document.getElementById('banco').value,
            valor: calculoAtual.valor,
            taxa_juros: parseFloat(document.getElementById('taxaJuros').value.replace(',', '.')),
            data_troca: document.getElementById('data_troca').value,
            data_vencimento: document.getElementById('data_vencimento').value,
            dias: calculoAtual.dias,
            juros: calculoAtual.juros,
            valor_liquido: calculoAtual.valorLiquido
        };

        // Validações básicas
        if (!formData.cliente || !formData.valor || !formData.data_troca || !formData.data_vencimento) {
            alert('Por favor, preencha todos os campos obrigatórios');
            return;
        }

        $.ajax({
            url: 'salvar_cheque.php',
            type: 'POST',
            data: formData,
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    alert('Cheque salvo com sucesso!');
                    window.location.reload(); // Recarrega a página para atualizar a lista
                } else {
                    alert('Erro ao salvar: ' + response.message);
                }
            },
            error: function(xhr, status, error) {
                console.error('Erro na requisição:', error);
                console.error('Status:', status);
                console.error('Resposta:', xhr.responseText);
                alert('Erro ao salvar o cheque. Verifique o console para mais detalhes.');
            }
        });
    } catch (error) {
        console.error('Erro ao salvar:', error);
        alert('Ocorreu um erro ao tentar salvar o cheque.');
    }
}
</script>

<?php
// Inclua aqui seus outros scripts JavaScript
require_once 'includes/footer.php';
?>
