class TodoApp {
    constructor() {
        this.tasks = [];
        this.currentEditId = null;
        this.init();
    }

    init() {
        this.bindEvents();
        this.loadTasks();
        this.initTheme();
        this.setDefaultDate();
    }

    bindEvents() {
        // Botões principais
        document.getElementById('addTaskBtn').addEventListener('click', () => this.openModal());
        document.getElementById('closeModal').addEventListener('click', () => this.closeModal());
        document.getElementById('cancelBtn').addEventListener('click', () => this.closeModal());
        document.getElementById('themeToggle').addEventListener('click', () => this.toggleTheme());
        
        // Formulário
        document.getElementById('taskForm').addEventListener('submit', (e) => this.handleSubmit(e));
        
        // Busca e filtros
        document.getElementById('searchInput').addEventListener('input', (e) => this.handleSearch(e.target.value));
        document.getElementById('dateFilter').addEventListener('change', (e) => this.handleDateFilter(e.target.value));
        document.getElementById('clearFilters').addEventListener('click', () => this.clearFilters());
        
        // Remover o fechamento por clique fora do modal
        // Remover o fechamento por tecla ESC
    }

    async loadTasks() {
        try {
            this.showLoading();
            const response = await fetch('listar.php');
            const data = await response.json();
            
            if (data.success) {
                this.tasks = data.data;
                this.renderTasks();
            } else {
                Swal.fire({
                    title: 'Erro!',
                    text: 'Erro ao carregar tarefas: ' + data.message,
                    icon: 'error',
                    allowOutsideClick: false
                });
            }
        } catch (error) {
            Swal.fire({
                title: 'Erro de conexão!',
                text: error.message,
                icon: 'error',
                allowOutsideClick: false
            });
        } finally {
            this.hideLoading();
        }
    }

    async saveTask(taskData) {
        try {
            const url = this.currentEditId ? 'editar.php' : 'adicionar.php';
            const method = this.currentEditId ? 'PUT' : 'POST';
            
            if (this.currentEditId) {
                taskData.id = this.currentEditId;
            }
            
            const response = await fetch(url, {
                method: method,
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(taskData)
            });
            
            const data = await response.json();
            
            if (data.success) {
                Swal.fire({
                    title: 'Sucesso!',
                    text: data.message,
                    icon: 'success',
                    timer: 2000,
                    showConfirmButton: false,
                    allowOutsideClick: false
                });
                this.closeModal();
                this.loadTasks();
            } else {
                Swal.fire({
                    title: 'Erro!',
                    text: data.message,
                    icon: 'error',
                    allowOutsideClick: false
                });
            }
        } catch (error) {
            Swal.fire({
                title: 'Erro de conexão!',
                text: error.message,
                icon: 'error',
                allowOutsideClick: false
            });
        }
    }

    async deleteTask(id) {
        const result = await Swal.fire({
            title: 'Tem certeza?',
            text: 'Você não poderá reverter esta ação!',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'Sim, excluir!',
            cancelButtonText: 'Cancelar',
            allowOutsideClick: false,
            allowEscapeKey: false
        });
        
        if (!result.isConfirmed) return;
        
        try {
            const response = await fetch('excluir.php', {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ id: id })
            });
            
            const data = await response.json();
            
            if (data.success) {
                Swal.fire({
                    title: 'Excluído!',
                    text: data.message,
                    icon: 'success',
                    timer: 2000,
                    showConfirmButton: false,
                    allowOutsideClick: false
                });
                this.loadTasks();
            } else {
                Swal.fire({
                    title: 'Erro!',
                    text: data.message,
                    icon: 'error',
                    allowOutsideClick: false
                });
            }
        } catch (error) {
            Swal.fire({
                title: 'Erro de conexão!',
                text: error.message,
                icon: 'error',
                allowOutsideClick: false
            });
        }
    }

    renderTasks(tasksToRender = this.tasks) {
        const container = document.getElementById('tasksContainer');
        const emptyState = document.getElementById('emptyState');
        
        if (tasksToRender.length === 0) {
            container.innerHTML = '';
            emptyState.style.display = 'block';
            return;
        }
        
        emptyState.style.display = 'none';
        container.innerHTML = tasksToRender.map(task => `
            <div class="task-card" data-id="${task.id}">
                <div class="task-header">
                    <h3>${this.escapeHtml(task.titulo)}</h3>
                    <div class="task-actions">
                        <button onclick="todoApp.editTask(${task.id})" class="btn-edit" title="Editar">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button onclick="todoApp.deleteTask(${task.id})" class="btn-delete" title="Excluir">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </div>
                <div class="task-date">
                    <i class="fas fa-calendar"></i>
                    ${this.formatDate(task.data)}
                </div>
                ${task.descricao ? `<div class="task-description">${this.escapeHtml(task.descricao)}</div>` : ''}
            </div>
        `).join('');
    }

    openModal(task = null) {
        const modal = document.getElementById('taskModal');
        const title = document.getElementById('modalTitle');
        const form = document.getElementById('taskForm');
        
        if (task) {
            this.currentEditId = task.id;
            title.textContent = 'Editar Tarefa';
            document.getElementById('taskTitle').value = task.titulo;
            document.getElementById('taskDate').value = task.data;
            document.getElementById('taskDescription').value = task.descricao || '';
        } else {
            this.currentEditId = null;
            title.textContent = 'Nova Tarefa';
            form.reset();
            this.setDefaultDate();
        }
        
        modal.style.display = 'flex';
        document.getElementById('taskTitle').focus();
    }

    closeModal() {
        document.getElementById('taskModal').style.display = 'none';
        document.getElementById('taskForm').reset();
        this.currentEditId = null;
    }

    editTask(id) {
        const task = this.tasks.find(t => t.id == id);
        if (task) {
            this.openModal(task);
        }
    }

    handleSubmit(e) {
        e.preventDefault();
        
        const taskData = {
            titulo: document.getElementById('taskTitle').value.trim(),
            data: document.getElementById('taskDate').value,
            descricao: document.getElementById('taskDescription').value.trim()
        };
        
        if (!taskData.titulo || !taskData.data) {
            Swal.fire({
                title: 'Campos obrigatórios!',
                text: 'Por favor, preencha o título e a data.',
                icon: 'warning',
                allowOutsideClick: false
            });
            return;
        }
        
        this.saveTask(taskData);
    }

    handleSearch(query) {
        const filtered = this.tasks.filter(task => 
            task.titulo.toLowerCase().includes(query.toLowerCase()) ||
            (task.descricao && task.descricao.toLowerCase().includes(query.toLowerCase()))
        );
        this.renderTasks(filtered);
    }

    handleDateFilter(date) {
        if (!date) {
            this.renderTasks();
            return;
        }
        const filtered = this.tasks.filter(task => task.data === date);
        this.renderTasks(filtered);
    }

    clearFilters() {
        document.getElementById('searchInput').value = '';
        document.getElementById('dateFilter').value = '';
        this.renderTasks();
    }

    toggleTheme() {
        const currentTheme = document.documentElement.getAttribute('data-theme');
        const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
        
        document.documentElement.setAttribute('data-theme', newTheme);
        localStorage.setItem('darkTheme', newTheme === 'dark');
        
        const icon = document.querySelector('#themeToggle i');
        icon.className = newTheme === 'dark' ? 'fas fa-sun' : 'fas fa-moon';
    }

    initTheme() {
        const isDark = localStorage.getItem('darkTheme') === 'true';
        const theme = isDark ? 'dark' : 'light';
        
        document.documentElement.setAttribute('data-theme', theme);
        
        const icon = document.querySelector('#themeToggle i');
        if (icon) {
            icon.className = isDark ? 'fas fa-sun' : 'fas fa-moon';
        }
    }

    setDefaultDate() {
        const today = new Date().toISOString().split('T')[0];
        document.getElementById('taskDate').value = today;
        document.getElementById('dateFilter').value = today;
    }

    showToast(message, type = 'info') {
        let icon = 'info';
        if (type === 'success') icon = 'success';
        if (type === 'error') icon = 'error';
        if (type === 'warning') icon = 'warning';
        
        Swal.fire({
            title: message,
            icon: icon,
            toast: true,
            position: 'top-end',
            showConfirmButton: false,
            timer: 3000,
            timerProgressBar: true
        });
    }

    showLoading() {
        // Implementar loading se necessário
    }

    hideLoading() {
        // Implementar loading se necessário
    }

    formatDate(dateString) {
        const date = new Date(dateString + 'T00:00:00');
        return date.toLocaleDateString('pt-BR', {
            day: '2-digit',
            month: '2-digit',
            year: 'numeric'
        });
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
}

// Inicializar a aplicação
const todoApp = new TodoApp();

// Service Worker (opcional)
if ('serviceWorker' in navigator) {
    window.addEventListener('load', () => {
        navigator.serviceWorker.register('/sw.js')
            .then(registration => console.log('SW registered'))
            .catch(error => console.log('SW registration failed'));
    });
}