import sharp from 'sharp';
import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const inputDir = path.join(__dirname, '../public/imgs');
const outputDir = path.join(__dirname, '../public/imgs/optimized');

// Criar diretório de saída se não existir
if (!fs.existsSync(outputDir)) {
    fs.mkdirSync(outputDir, { recursive: true });
}

// Configurações de otimização
const webpOptions = {
    quality: 80,
    lossless: false,
    effort: 6,
};

const avifOptions = {
    quality: 80,
    effort: 9,
};

// Processar imagens
async function optimizeImages() {
    const files = fs.readdirSync(inputDir);
    
    for (const file of files) {
        if (!['.jpg', '.jpeg', '.png'].includes(path.extname(file).toLowerCase())) continue;
        
        const inputPath = path.join(inputDir, file);
        const filename = path.parse(file).name;
        
        try {
            // Processar para WebP
            await sharp(inputPath)
                .resize(1920, null, { withoutEnlargement: true })
                .webp(webpOptions)
                .toFile(path.join(outputDir, `${filename}.webp`));
                
            // Processar para AVIF
            await sharp(inputPath)
                .resize(1920, null, { withoutEnlargement: true })
                .avif(avifOptions)
                .toFile(path.join(outputDir, `${filename}.avif`));
                
            // Otimizar original
            await sharp(inputPath)
                .resize(1920, null, { withoutEnlargement: true })
                .jpeg({ quality: 85, progressive: true })
                .toFile(path.join(outputDir, file));
                
            console.log(`✓ Otimizado: ${file}`);
        } catch (error) {
            console.error(`Erro ao processar ${file}:`, error);
        }
    }
}

optimizeImages().catch(console.error); 