/**
 * Script para melhorar o comportamento do menu em dispositivos móveis
 * Inclui suporte para backdrop e gestos de deslize
 */
document.addEventListener('DOMContentLoaded', function() {
    // Elementos principais
    const body = document.body;
    const toggleButton = document.querySelector('.button-toggle-menu');
    const leftSideMenu = document.querySelector('.leftside-menu');
    const contentPage = document.querySelector('.content-page');
    const backdrop = document.querySelector('.menu-backdrop');
    
    // Se não houver backdrop, criar um
    if (!backdrop) {
        const newBackdrop = document.createElement('div');
        newBackdrop.classList.add('menu-backdrop');
        document.querySelector('.wrapper').appendChild(newBackdrop);
    }
    
    // Função para ativar/desativar o menu lateral
    function toggleSidebar() {
        body.classList.toggle('sidebar-enable');
        
        // Em telas maiores, alternar o tamanho do menu em vez de mostrar/esconder
        if (window.innerWidth >= 768) {
            const currentSize = document.documentElement.getAttribute('data-sidenav-size');
            if (currentSize === 'condensed') {
                document.documentElement.setAttribute('data-sidenav-size', 'default');
            } else {
                document.documentElement.setAttribute('data-sidenav-size', 'condensed');
            }
        }
    }
    
    // Adicionar listener para o botão de toggle
    if (toggleButton) {
        toggleButton.addEventListener('click', function(e) {
            e.preventDefault();
            toggleSidebar();
        });
    }
    
    // Adicionar listener para o backdrop (fecha o menu quando clicado)
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('menu-backdrop')) {
            body.classList.remove('sidebar-enable');
        }
    });
    
    // Adicionar suporte para gestos de deslize em dispositivos móveis
    let touchStartX = 0;
    let touchEndX = 0;
    
    // Detectar início do toque
    document.addEventListener('touchstart', function(e) {
        touchStartX = e.changedTouches[0].screenX;
    }, false);
    
    // Detectar fim do toque e verificar se foi um deslize
    document.addEventListener('touchend', function(e) {
        touchEndX = e.changedTouches[0].screenX;
        handleSwipe();
    }, false);
    
    // Processar o gesto de deslize
    function handleSwipe() {
        const swipeThreshold = 100; // Mínimo de pixels para considerar um deslize
        
        // Deslize da esquerda para a direita - abrir menu
        if (touchEndX - touchStartX > swipeThreshold && touchStartX < 50) {
            if (!body.classList.contains('sidebar-enable')) {
                body.classList.add('sidebar-enable');
            }
        }
        
        // Deslize da direita para a esquerda - fechar menu
        if (touchStartX - touchEndX > swipeThreshold && body.classList.contains('sidebar-enable')) {
            body.classList.remove('sidebar-enable');
        }
    }
    
    // Ajustar comportamento ao redimensionar a janela
    window.addEventListener('resize', function() {
        if (window.innerWidth >= 768 && body.classList.contains('sidebar-enable')) {
            body.classList.remove('sidebar-enable');
        }
    });
    
    // Fechar o menu ao pressionar a tecla ESC
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && body.classList.contains('sidebar-enable')) {
            body.classList.remove('sidebar-enable');
        }
    });
    
    // Otimização para dispositivos de toque - evitar delay de clique
    // Isso melhora a experiência em dispositivos móveis
    if ('ontouchstart' in document.documentElement) {
        for (const el of document.querySelectorAll('.side-nav-link, .dropdown-item, .btn')) {
            el.style.touchAction = 'manipulation';
        }
    }
}); 