<?php

namespace App\Utils;

use Illuminate\Support\Str;
use App\Models\Estoque;
use App\Models\Produto;
use App\Models\Localizacao;
use App\Models\MovimentacaoProduto;
use Illuminate\Support\Facades\Auth;

class EstoqueUtil
{
    public function incrementaEstoque($produto_id, $quantidade, $produto_variacao_id, $local_id = null)
    {
        // Sempre consolidar no registro principal (sem variação e sem local)
        $produto_variacao_id = null;
        $local_id = null;
        $item = Estoque::where('produto_id', $produto_id)
            ->whereNull('produto_variacao_id')
            ->whereNull('local_id')
            ->first();
        if ($item) {
            $estoqueAtual = (float)$item->quantidade;
            $item->quantidade = $estoqueAtual + $quantidade;
            $item->save();
        } else {
            $item = Estoque::create([
                'produto_id' => $produto_id,
                'produto_variacao_id' => null,
                'local_id' => null,
                'quantidade' => $quantidade
            ]);
        }
    }

    public function reduzEstoque($produto_id, $quantidade, $produto_variacao_id, $local_id = null)
    {
        // Sempre consolidar no registro principal (sem variação e sem local)
        $produto_variacao_id = null;
        $local_id = null;
        $item = Estoque::where('produto_id', $produto_id)
            ->whereNull('produto_variacao_id')
            ->whereNull('local_id')
            ->first();
        if(!$item){
            $item = Estoque::create([
                'produto_id' => $produto_id,
                'produto_variacao_id' => null,
                'local_id' => null,
                'quantidade' => 0
            ]);
        }
        $item->load('produto');
        $produto = $item->produto;
        if (!$produto) {
            throw new \Exception("Relacionamento produto não carregado no estoque");
        }
        $quantidade = (float)$quantidade;
        $estoqueAtual = (float)$item->quantidade;
        if ($produto->unidadeDecimal()) {
            $quantidade = (float)$quantidade;
            $estoqueAtual = (float)$item->quantidade;
        }
        if($estoqueAtual - $quantidade < 0){
            throw new \Exception("Estoque insuficiente para esta variação!");
        }
        $item->quantidade = $estoqueAtual - $quantidade;
        $item->save();
    }

    public function reduzComposicao($produto_id, $quantidade, $produto_variacao_id = null)
    {
        $produto = Produto::findOrFail($produto_id);
        foreach ($produto->composicao as $item) {
            $this->reduzEstoque($item->ingrediente_id, ($item->quantidade * $quantidade), $produto_variacao_id);
        }
        $this->incrementaEstoque($produto_id, $quantidade, $produto_variacao_id);
    }

    public function verificaEstoqueComposicao($produto_id, $quantidade, $produto_variacao_id = null)
    {
        $produto = Produto::findOrFail($produto_id);
        $mensagem = "";
        foreach ($produto->composicao as $item) {
            $qtd = $item->quantidade * $quantidade;

            if($item->ingrediente->estoque){
                if($qtd > $item->ingrediente->estoque->quantidade){
                    $mensagem .= $item->ingrediente->nome . " com estoque insuficiente | ";
                }
            }else{
                $mensagem .= $item->ingrediente->nome . " sem nenhum estoque cadastrado | ";
            }
        }
        $mensagem = substr($mensagem, 0, strlen($mensagem)-2);
        return $mensagem;

    }

    public function verificaEstoqueCombo($produto, $quantidade)
    {
        $mensagem = "";
        
        // Verificar estoque do produto principal (combo)
        if($produto->gerenciar_estoque) {
            if($produto->estoque) {
                if($quantidade > $produto->estoque->quantidade) {
                    $mensagem .= $produto->nome . " (combo principal) com estoque insuficiente | ";
                }
            } else {
                $mensagem .= $produto->nome . " (combo principal) sem estoque cadastrado | ";
            }
        }
        
        // Verificar estoque de cada item do combo
        foreach ($produto->itensDoCombo as $item) {
            $qtd = $item->quantidade * $quantidade;
            if($item->produtoDoCombo->gerenciar_estoque){
                if($item->produtoDoCombo->estoque){
                    if($qtd > $item->produtoDoCombo->estoque->quantidade){
                        $mensagem .= $item->produtoDoCombo->nome . " com estoque insuficiente | ";
                    }
                }else{
                    $mensagem .= $item->produtoDoCombo->nome . " sem nenhum estoque cadastrado | ";
                }
            }
        }
        
        $mensagem = substr($mensagem, 0, strlen($mensagem)-2);
        return $mensagem;
    }

    public function movimentacaoProduto($produto_id, $quantidade, $tipo, $codigo_transacao, $tipo_transacao, $user_id,
        $produto_variacao_id = null){

        $produto = Produto::findOrFail($produto_id);
        
        // Registrar movimentação para o produto principal
        $estoque = Estoque::where('produto_id', $produto_id)->first();
        MovimentacaoProduto::create([
            'produto_id' => $produto_id,
            'quantidade' => $quantidade,
            'tipo' => $tipo,
            'codigo_transacao' => $codigo_transacao,
            'tipo_transacao' => $tipo_transacao,
            'produto_variacao_id' => $produto_variacao_id,
            'user_id' => $user_id,
            'estoque_atual' => $estoque ? $estoque->quantidade : 0
        ]);
        
        // Se for um combo, registrar movimentação para cada item do combo
        if($produto->combo) {
            foreach($produto->itensDoCombo as $item) {
                $item_estoque = Estoque::where('produto_id', $item->item_id)->first();
                MovimentacaoProduto::create([
                    'produto_id' => $item->item_id,
                    'quantidade' => $item->quantidade * $quantidade,
                    'tipo' => $tipo,
                    'codigo_transacao' => $codigo_transacao,
                    'tipo_transacao' => $tipo_transacao . '_combo',
                    'produto_variacao_id' => $produto_variacao_id,
                    'user_id' => $user_id,
                    'estoque_atual' => $item_estoque ? $item_estoque->quantidade : 0
                ]);
            }
        }
    }

    public function verificaEstoque($produto_id, $quantidade, $produto_variacao_id = null, $local_id = null)
    {
        // Sempre buscar com local_id = null
        $local_id = null;
        // Tratar produto_variacao_id = 0, '', 'null' como null real
        if (empty($produto_variacao_id) || $produto_variacao_id === 'null' || $produto_variacao_id === 0) {
            $produto_variacao_id = null;
        }
        $produto = Produto::findOrFail($produto_id);
        if($produto->combo) {
            return $this->verificaEstoqueCombo($produto, $quantidade);
        }
        $item = Estoque::where('produto_id', $produto_id)
            ->when($produto_variacao_id !== null, function ($q) use ($produto_variacao_id) {
                return $q->where('produto_variacao_id', $produto_variacao_id);
            }, function ($q) {
                return $q->whereNull('produto_variacao_id');
            })
            ->whereNull('local_id')
            ->first();
        if ($item == null) {
            return "Produto sem estoque cadastrado";
        }
        if ($produto->unidadeDecimal()) {
            $qtdEstoque = (float)$item->quantidade;
            $qtdSolicitada = (float)$quantidade;
            if (round($qtdEstoque, 3) < round($qtdSolicitada, 3)) {
                return "Estoque insuficiente";
            }
        } else {
            if ($item->quantidade < $quantidade) {
                return "Estoque insuficiente";
            }
        }
        return "";
    }

    public static function salvarEstoque($produto_id, $quantidade, $produto_variacao_id = null, $local_id = null)
    {
        // Se não trabalha com múltiplos locais, sempre forçar local_id = null
        $local_id = null;
        // ... restante do código ...
        $item = Estoque::where('produto_id', $produto_id)
            ->where('produto_variacao_id', $produto_variacao_id)
            ->whereNull('local_id')
            ->first();
        if ($item) {
            $item->quantidade = $quantidade;
            $item->save();
        } else {
            $item = Estoque::create([
                'produto_id' => $produto_id,
                'produto_variacao_id' => $produto_variacao_id,
                'local_id' => null,
                'quantidade' => $quantidade
            ]);
        }
        return $item;
    }

}
