@extends('layouts.app', ['title' => 'Troca de Venda'])

@section('css')
<link href="{{ asset('css/devolucao-venda.css') }}" rel="stylesheet">
<style>
    .valor-positivo {
        color: #28a745;
        font-weight: bold;
    }
    .valor-negativo {
        color: #dc3545;
        font-weight: bold;
    }
    .card-difference {
        transition: all 0.3s ease;
    }
    .card-difference-positive {
        border-color: #28a745;
    }
    .card-difference-negative {
        border-color: #dc3545;
    }
    .difference-info {
        font-size: 0.9rem;
        margin-top: 5px;
    }
</style>
@endsection

@section('content')

<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header bg-primary text-white">
                <h4 class="mb-0 text-white">Dados da Venda {{ strtoupper($tipo) }} #{{ $venda->numero_sequencial }}</h4>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <p><strong>Cliente:</strong> {{ $venda->cliente ? $venda->cliente->razao_social : 'Consumidor Final' }}</p>
                        <p><strong>Data:</strong> {{ date('d/m/Y H:i', strtotime($venda->created_at)) }}</p>
                    </div>
                    <div class="col-md-6 text-right">
                        <p><strong>Valor Total:</strong> R$ {{ __moeda($venda->total) }}</p>
                        <p><strong>Forma de Pagamento:</strong> {{ $tipo == 'nfce' ? \App\Models\Nfce::getTipoPagamento($venda->tipo_pagamento) : \App\Models\Nfe::getTipoPagamento($venda->tipo_pagamento) }}</p>
                    </div>
                </div>
            </div>
        </div>

        @if($temCrediario)
        <div class="card mb-3">
            <div class="card-header bg-warning text-dark">
                <h5 class="mb-0"><i class="ri-information-line"></i> Venda com Crediário</h5>
            </div>
            <div class="card-body">
                <p><strong>Esta venda foi paga com crediário no valor de R$ {{ __moeda($valorCrediario) }}.</strong></p>
                <p>O reembolso será feito através do abatimento nas parcelas do crediário (contas a receber).</p>
                
                @if($contasReceber->count() > 0)
                <div class="mt-3">
                    <h6>Parcelas do Crediário Pendentes:</h6>
                    <div class="table-responsive">
                        <table class="table table-sm table-bordered">
                            <thead>
                                <tr>
                                    <th>Parcela</th>
                                    <th>Vencimento</th>
                                    <th>Valor Pendente</th>
                                    <th>Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($contasReceber as $conta)
                                <tr>
                                    <td>{{ $conta->descricao ?? 'Parcela ' . $loop->iteration }}</td>
                                    <td>{{ __data_pt($conta->data_vencimento) }}</td>
                                    <td>R$ {{ __moeda($conta->valor_integral - $conta->valor_recebido) }}</td>
                                    <td>
                                        @if($conta->status == 0)
                                            <span class="badge bg-warning">Pendente</span>
                                        @elseif($conta->status == 1)
                                            <span class="badge bg-success">Recebido</span>
                                        @elseif($conta->status == 2)
                                            <span class="badge bg-danger">Vencido</span>
                                        @endif
                                    </td>
                                </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                </div>
                @endif
            </div>
        </div>
        @endif

        <form action="{{ route('devolucao_venda.store_troca') }}" method="POST">
            @csrf
            <input type="hidden" name="tipo_venda" value="{{ $tipo }}">
            <input type="hidden" name="venda_id" value="{{ $venda->id }}">
            <input type="hidden" name="empresa_id" value="{{ $venda->empresa_id }}">

            <div class="card">
                <div class="card-header">
                    <h4>Selecione os Itens para Troca</h4>
                </div>
                <div class="card-body">
                    <table class="table table-bordered">
                        <thead>
                            <tr>
                                <th width="50">Selecionar</th>
                                <th>Produto</th>
                                <th>Quantidade Vendida</th>
                                <th>Valor Unitário</th>
                                <th>Subtotal</th>
                                <th>Quantidade a Trocar</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($venda->itens as $item)
                            <tr>
                                <td class="text-center">
                                    <div class="form-check">
                                        <input class="form-check-input check-item" type="checkbox" name="itens_ids[]" value="{{ $item->id }}" id="check_{{ $item->id }}">
                                    </div>
                                </td>
                                <td>
                                    <div class="d-flex align-items-center">
                                        @if($item->produto && $item->produto->img)
                                        <img src="{{ $item->produto->img }}" width="40" height="40" class="rounded me-2">
                                        @endif
                                        <div>
                                            {{ $item->produto ? $item->produto->nome : 'Produto não encontrado' }}
                                            @if(isset($item->variacao_id) && $item->variacao_id)
                                            <br><small class="text-muted">{{ $item->produtoVariacao ? $item->produtoVariacao->descricao : '' }}</small>
                                            @endif
                                            @if($item->ja_devolvido)
                                            <br><small class="text-danger">Parcialmente devolvido anteriormente</small>
                                            @endif
                                        </div>
                                    </div>
                                </td>
                                <td>{{ number_format($item->quantidade_disponivel, 3, ',', '.') }}
                                    @if($item->ja_devolvido)
                                    <br><small class="text-muted">(Original: {{ number_format($item->quantidade_original, 3, ',', '.') }})</small>
                                    @endif
                                </td>
                                <td>R$ {{ __moeda($item->valor_unitario) }}</td>
                                <td>R$ {{ __moeda($item->quantidade_disponivel * $item->valor_unitario) }}</td>
                                <td>
                                    <input type="text" name="quantidades[]" class="form-control quantidade-item" data-id="{{ $item->id }}" value="{{ number_format($item->quantidade_disponivel, 3, ',', '.') }}" data-mask="##0,000" data-mask-reverse="true" disabled>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <h4>Produtos para Troca</h4>
                </div>
                <div class="card-body">
                    <div class="row mb-3">
                        <div class="col-12">
                            <button type="button" class="btn btn-success btn-sm" id="btn-add-produto">
                                <i class="ri-add-line"></i> Adicionar Produto
                            </button>
                        </div>
                    </div>
                    <div id="produtos-troca-container">
                        <!-- Aqui serão adicionados os produtos para troca via JavaScript -->
                    </div>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <h4>Resumo da Troca</h4>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-4">
                            <div class="form-group">
                                <label>Valor Total Devolvido</label>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <span class="input-group-text">R$</span>
                                    </div>
                                    <input type="text" class="form-control" id="valor-total-devolvido" readonly>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label>Valor Total dos Novos Produtos</label>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <span class="input-group-text">R$</span>
                                    </div>
                                    <input type="text" class="form-control" id="valor-total-novos" readonly>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label>Diferença</label>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <span class="input-group-text">R$</span>
                                    </div>
                                    <input type="text" class="form-control" id="valor-diferenca" readonly>
                                </div>
                                <small class="form-text" id="texto-diferenca">Diferença a ser calculada com base nos valores.</small>
                            </div>
                        </div>
                    </div>

                    <!-- Resumo visual detalhado da troca -->
                    <div class="row mt-4">
                        <div class="col-md-6">
                            <div class="card border-secondary">
                                <div class="card-header bg-secondary text-white">
                                    <h5 class="mb-0">Produtos a Devolver</h5>
                                </div>
                                <div class="card-body p-0">
                                    <div id="resumo-produtos-devolvidos" class="list-group list-group-flush">
                                        <div class="list-group-item text-center text-muted">
                                            Selecione os produtos para visualizar o resumo
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="card border-primary">
                                <div class="card-header bg-primary text-white">
                                    <h5 class="mb-0">Novos Produtos</h5>
                                </div>
                                <div class="card-body p-0">
                                    <div id="resumo-produtos-novos" class="list-group list-group-flush">
                                        <div class="list-group-item text-center text-muted">
                                            Adicione produtos para visualizar o resumo
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="row mt-3" id="div-metodo-pagamento" style="display: none;">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="tipo_pagamento">Método de Pagamento/Recebimento</label>
                                <select name="tipo_pagamento" id="tipo_pagamento" class="form-control" required {{ $temCrediario ? 'disabled' : '' }}>
                                    <option value="">Selecione</option>
                                    @if($temCrediario)
                                        <option value="06" selected>Crediário (Abatimento nas parcelas)</option>
                                    @else
                                        <!-- Os métodos de pagamento serão preenchidos via JavaScript -->
                                    @endif
                                </select>
                                @if($temCrediario)
                                    <input type="hidden" name="tipo_pagamento" value="06">
                                    <small class="form-text text-muted">Método automaticamente definido para crediário</small>
                                @endif
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="alert alert-info mt-4">
                                <p><i class="ri-information-line"></i> <strong>Importante:</strong></p>
                                <ul class="mb-0">
                                    @if($temCrediario)
                                        <li><strong>Crediário:</strong> O valor será abatido das parcelas pendentes do crediário.</li>
                                    @else
                                        <li id="texto-devolucao"><strong>Crédito em Loja:</strong> O valor será adicionado como crédito para o cliente.</li>
                                        <li id="texto-pagamento" style="display: none;"><strong>Pagamento:</strong> O cliente deverá efetuar o pagamento do valor total dos novos produtos. Será registrado o valor completo dos novos produtos (R$ <span id="valor-total-produtos-novos">0,00</span>).</li>
                                        <li id="texto-pagamento-dinheiro" style="display: none;"><strong>Pagamento em Dinheiro:</strong> Será registrado o valor total dos novos produtos (R$ <span id="valor-total-produtos-novos-dinheiro">0,00</span>), e não apenas a diferença.</li>
                                    @endif
                                </ul>
                            </div>
                        </div>
                    </div>

                    <!-- Campo de seleção de cliente para caso de crédito em loja -->
                    <div class="form-group mt-3" id="div-cliente" style="display: none;">
                        <label for="cliente_id">Cliente <span class="text-danger">*</span></label>
                        <div class="input-group">
                            <select name="cliente_id" id="cliente_id" class="form-control select2">
                                <option value="">Selecione um cliente</option>
                                @foreach(\App\Models\Cliente::where('empresa_id', $venda->empresa_id)->orderBy('razao_social')->get() as $cliente)
                                <option value="{{ $cliente->id }}" {{ $venda->cliente_id == $cliente->id ? 'selected' : '' }}>{{ $cliente->razao_social }}</option>
                                @endforeach
                            </select>
                            <div class="input-group-append">
                                <a href="{{ route('clientes.create') }}" target="_blank" class="btn btn-outline-secondary">
                                    <i class="ri-add-line"></i> Novo
                                </a>
                            </div>
                        </div>
                        <small class="form-text text-danger">Obrigatório para crédito em loja</small>
                    </div>
                </div>
                <div class="card-footer text-right">
                    <a href="{{ route('devolucao_venda.index') }}" class="btn btn-secondary">Cancelar</a>
                    <button type="submit" class="btn btn-primary" id="btn-trocar">Concluir Troca</button>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Template para adição de produto -->
<template id="template-produto-troca">
    <div class="row produto-troca-item mb-3">
        <div class="col-md-5">
            <div class="form-group">
                <label>Produto</label>
                <select name="produtos_troca[]" class="form-control select-produto-troca" required>
                    <option value="">Selecione o produto</option>
                    @foreach(\App\Models\Produto::where('empresa_id', $venda->empresa_id)->orderBy('nome')->get() as $p)
                    <option value="{{ $p->id }}" data-valor="{{ $p->valor_venda }}">{{ $p->nome }}</option>
                    @endforeach
                </select>
            </div>
        </div>
        <div class="col-md-3">
            <div class="form-group">
                <label>Quantidade</label>
                <input type="text" name="quantidades_troca[]" class="form-control quantidade-produto-troca" data-mask="##0,000" data-mask-reverse="true" value="1,000" required>
            </div>
        </div>
        <div class="col-md-3">
            <div class="form-group">
                <label>Valor Unitário</label>
                <div class="input-group">
                    <div class="input-group-prepend">
                        <span class="input-group-text">R$</span>
                    </div>
                    <input type="text" name="valores_unitarios_troca[]" class="form-control valor-unitario-produto-troca" data-mask="#.##0,00" data-mask-reverse="true" value="0,00" required>
                </div>
            </div>
        </div>
        <div class="col-md-1">
            <div class="form-group">
                <label>&nbsp;</label>
                <button type="button" class="btn btn-danger form-control btn-remover-produto">
                    <i class="ri-delete-bin-line"></i>
                </button>
            </div>
        </div>
    </div>
</template>

@endsection

@section('js')
<script>
    $(document).ready(function() {
        // Variáveis para armazenar os valores de devolução, novos produtos e diferença
        let valorDevolucao = 0;
        let valorNovos = 0;
        let diferenca = 0;

        // Verificar se há crediário
        const temCrediario = {{ $temCrediario ? 'true' : 'false' }};
        
        // Se tiver crediário, não mostrar o campo de cliente
        if (temCrediario) {
            $('#div-cliente').hide();
            $('#cliente_id').prop('required', false);
        }

        // Formas de pagamento específicas para devolução ao cliente
        const tiposPagamentoCliente = {
            @foreach($tiposPagamentoCliente as $key => $tipo)
            '{{ $key }}': '{{ $tipo }}',
            @endforeach
        };

        // Todas as formas de pagamento
        const todosTiposPagamento = {
            @foreach($todosTiposPagamento as $key => $tipo)
            '{{ $key }}': '{{ $tipo }}',
            @endforeach
        };

        // Função para atualizar os métodos de pagamento com base na diferença
        function atualizarMetodosPagamento() {
            const $selectTipoPagamento = $('#tipo_pagamento');
            
            // Se tiver crediário, não atualizar os métodos
            if (temCrediario) {
                return;
            }
            
            $selectTipoPagamento.empty();
            $selectTipoPagamento.append('<option value="">Selecione</option>');

            if (diferenca > 0) {
                // Cliente recebe: mostrar apenas métodos específicos
                Object.entries(tiposPagamentoCliente).forEach(([key, value]) => {
                    $selectTipoPagamento.append(`<option value="${key}">${value}</option>`);
                });
            } else {
                // Cliente paga: mostrar todos os métodos
                Object.entries(todosTiposPagamento).forEach(([key, value]) => {
                    $selectTipoPagamento.append(`<option value="${key}">${value}</option>`);
                });
            }
        }

        // Função para calcular e atualizar os totais
        function calcularTotais() {
            valorDevolucao = 0;
            valorNovos = 0;

            console.log('=== CÁLCULO DE TOTAIS ===');

            // Calcular valor total devolvido
            $('.check-item:checked').each(function() {
                const id = $(this).val();
                const quantidade = parseFloat($(`input[data-id="${id}"]`).val().replace('.', '').replace(',', '.'));
                // Corrigir: usar td:eq(3) para valor unitário (coluna 4)
                const valorUnitario = parseFloat($(this).closest('tr').find('td:eq(3)').text().replace('R$ ', '').replace('.', '').replace(',', '.'));
                
                console.log(`Item ${id}: Quantidade=${quantidade}, Valor Unitário=${valorUnitario}, Subtotal=${quantidade * valorUnitario}`);
                
                if (!isNaN(quantidade) && !isNaN(valorUnitario)) {
                    valorDevolucao += quantidade * valorUnitario;
                }
            });

            console.log(`Valor total devolvido: ${valorDevolucao}`);

            // Calcular valor total dos novos produtos
            $('.produto-troca').each(function() {
                const quantidade = parseFloat($(this).find('.quantidade-produto-troca').val().replace('.', '').replace(',', '.'));
                const valorUnitario = parseFloat($(this).find('.valor-unitario-produto-troca').val().replace('.', '').replace(',', '.'));
                
                console.log(`Produto novo: Quantidade=${quantidade}, Valor Unitário=${valorUnitario}, Subtotal=${quantidade * valorUnitario}`);
                
                if (!isNaN(quantidade) && !isNaN(valorUnitario)) {
                    valorNovos += quantidade * valorUnitario;
                }
            });

            console.log(`Valor total novos produtos: ${valorNovos}`);

            // Calcular diferença
            diferenca = valorDevolucao - valorNovos;
            console.log(`Diferença: ${diferenca}`);

            // Formatar valores para exibição
            $('#valor-total-devolvido').val(valorDevolucao.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }));
            $('#valor-total-novos').val(valorNovos.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }));
            $('#valor-diferenca').val(diferenca.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }));
            
            // Atualizar texto da diferença e estilo
            if (diferenca > 0) {
                $('#texto-diferenca').html(`<span class="text-success">Cliente recebe R$ ${diferenca.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}</span>`);
                $('#valor-diferenca').addClass('valor-positivo').removeClass('valor-negativo');
            } else if (diferenca < 0) {
                $('#texto-diferenca').html(`<span class="text-danger">Cliente paga R$ ${Math.abs(diferenca).toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}</span>`);
                $('#valor-diferenca').addClass('valor-negativo').removeClass('valor-positivo');
            } else {
                $('#texto-diferenca').html('Troca sem diferença de valores');
                $('#valor-diferenca').removeClass('valor-positivo valor-negativo');
            }

            // Atualizar visibilidade dos campos de pagamento
            if (valorDevolucao > 0 || valorNovos > 0) {
                $('#div-metodo-pagamento').show();
                
                if (diferenca > 0) {
                    // Cliente recebe
                    $('#texto-devolucao').show();
                    $('#texto-pagamento, #texto-pagamento-dinheiro').hide();
                } else if (diferenca < 0) {
                    // Cliente paga
                    $('#texto-devolucao').hide();
                    $('#texto-pagamento').show();
                    $('#valor-total-produtos-novos').text(valorNovos.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }));
                    $('#valor-total-produtos-novos-dinheiro').text(valorNovos.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }));
                } else {
                    // Sem diferença
                    $('#texto-devolucao, #texto-pagamento, #texto-pagamento-dinheiro').hide();
                }
            } else {
                $('#div-metodo-pagamento').hide();
            }

            // Atualizar os métodos de pagamento com base na diferença
            atualizarMetodosPagamento();
        }

        // Função para atualizar o resumo visual dos produtos
        function atualizarResumoProdutos() {
            // Atualizar resumo dos produtos devolvidos
            const $resumoProdutosDevolvidos = $('#resumo-produtos-devolvidos');
            $resumoProdutosDevolvidos.empty();
            
            let temProdutosDevolvidos = false;
            
            $('.check-item:checked').each(function() {
                temProdutosDevolvidos = true;
                const id = $(this).val();
                const nomeProduto = $(this).closest('tr').find('td:eq(1)').text().trim();
                const quantidade = $(`input[data-id="${id}"]`).val();
                const valorUnitario = $(this).closest('tr').find('td:eq(3)').text().trim();
                
                const valorTotal = parseFloat(quantidade.replace('.', '').replace(',', '.')) * parseFloat(valorUnitario.replace('R$ ', '').replace('.', '').replace(',', '.'));
                
                $resumoProdutosDevolvidos.append(
                    '<div class="list-group-item d-flex justify-content-between align-items-center">' +
                    '<div>' +
                    '<strong>' + nomeProduto + '</strong>' +
                    '<br>' +
                    '<small>' + quantidade + ' x ' + valorUnitario + '</small>' +
                    '</div>' +
                    '<span class="badge bg-secondary">R$ ' + valorTotal.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + '</span>' +
                    '</div>'
                );
            });
            
            if (!temProdutosDevolvidos) {
                $resumoProdutosDevolvidos.html(
                    '<div class="list-group-item text-center text-muted">' +
                    'Selecione os produtos para visualizar o resumo' +
                    '</div>'
                );
            }
            
            // Atualizar resumo dos novos produtos
            const $resumoProdutosNovos = $('#resumo-produtos-novos');
            $resumoProdutosNovos.empty();
            
            let temProdutosNovos = false;
            
            $('.produto-troca').each(function() {
                temProdutosNovos = true;
                const selectProduto = $(this).find('.select-produto-troca');
                const nomeProduto = selectProduto.find('option:selected').text();
                const quantidade = $(this).find('.quantidade-produto-troca').val();
                const valorUnitario = $(this).find('.valor-unitario-produto-troca').val();
                
                if (selectProduto.val()) {
                    const valorTotal = parseFloat(quantidade.replace('.', '').replace(',', '.')) * parseFloat(valorUnitario.replace('.', '').replace(',', '.'));
                    
                    $resumoProdutosNovos.append(
                        '<div class="list-group-item d-flex justify-content-between align-items-center">' +
                        '<div>' +
                        '<strong>' + nomeProduto + '</strong>' +
                        '<br>' +
                        '<small>' + quantidade + ' x R$ ' + valorUnitario + '</small>' +
                        '</div>' +
                        '<span class="badge bg-primary">R$ ' + valorTotal.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + '</span>' +
                        '</div>'
                    );
                }
            });
            
            if (!temProdutosNovos) {
                $resumoProdutosNovos.html(
                    '<div class="list-group-item text-center text-muted">' +
                    'Adicione produtos para visualizar o resumo' +
                    '</div>'
                );
            }
            
            // Atualizar estilo do card de diferença
            const $cardDifference = $('.card-difference');
            if (diferenca > 0) {
                $cardDifference.addClass('card-difference-positive').removeClass('card-difference-negative');
            } else if (diferenca < 0) {
                $cardDifference.addClass('card-difference-negative').removeClass('card-difference-positive');
            } else {
                $cardDifference.removeClass('card-difference-positive card-difference-negative');
            }
        }

        // Inicializar Select2
        $('.select2').select2();

        // Event listeners
        $('#btn-add-produto').click(function() {
            adicionarCampoProduto();
        });

        // Código para verificar seleção de itens e habilitar campos de quantidade
        $('.check-item').change(function() {
            const checked = $(this).is(':checked');
            const id = $(this).val();
            const input = $(`input[data-id="${id}"]`);
            
            console.log(`Item ${id} ${checked ? 'selecionado' : 'desmarcado'}`);
            
            input.prop('disabled', !checked);
            
            if (checked) {
                input.focus();
            }
            
            calcularTotais();
            atualizarResumoProdutos();
        });

        // Atualizar totais quando a quantidade for alterada
        $(document).on('input', '.quantidade-item, .quantidade-produto-troca, .valor-unitario-produto-troca', function() {
            console.log('Campo alterado, recalculando totais...');
            calcularTotais();
            atualizarResumoProdutos();
        });

        // Monitorar alterações nos select de produtos
        $(document).on('change', '.select-produto-troca', function() {
            const produtoId = $(this).val();
            const container = $(this).closest('.produto-troca');
            
            console.log(`Produto selecionado: ${produtoId}`);
            
            if (produtoId) {
                // Obter o valor diretamente do atributo data-valor da opção selecionada
                const valorUnitario = $(this).find('option:selected').attr('data-valor') || 0;
                
                console.log(`Valor unitário do produto: ${valorUnitario}`);
                
                // Formatar e exibir o valor unitário
                container.find('.valor-unitario-produto-troca').val(
                    parseFloat(valorUnitario).toLocaleString('pt-BR', { 
                        minimumFractionDigits: 2, 
                        maximumFractionDigits: 2 
                    }).replace('.', ',')
                );
                
                // Atualizar cálculos
                calcularTotais();
                atualizarResumoProdutos();
            } else {
                container.find('.valor-unitario-produto-troca').val('0,00');
                calcularTotais();
                atualizarResumoProdutos();
            }
        });

        // Monitorar alterações no tipo de pagamento
        $('#tipo_pagamento').change(function() {
            const tipoPagamento = $(this).val();
            
            // Mostrar campo de cliente apenas para vale crédito
            if (tipoPagamento === '00') {
                $('#div-cliente').show();
            } else {
                $('#div-cliente').hide();
            }
            
            // Mostrar informações específicas para pagamento em dinheiro
            if (tipoPagamento === '01' && diferenca < 0) {
                $('#texto-pagamento').hide();
                $('#texto-pagamento-dinheiro').show();
            } else {
                $('#texto-pagamento-dinheiro').hide();
                if (diferenca < 0) {
                    $('#texto-pagamento').show();
                }
            }
        });

        // Função para adicionar campo de produto
        function adicionarCampoProduto() {
            console.log('Adicionando novo campo de produto...');
            const index = $('.produto-troca').length;
            
            // Criar elemento com o HTML necessário
            const $produtoTroca = $('<div>', {
                'class': 'produto-troca mb-3 border rounded p-3 bg-light'
            });
            
            // Estrutura do HTML interno
            const htmlInterno = `
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label>Produto</label>
                            <select name="produtos_troca[]" class="form-control select-produto-troca" required>
                                <option value="">Selecione um produto</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label>Quantidade</label>
                            <input type="text" name="quantidades_troca[]" class="form-control quantidade-produto-troca" value="1,000" data-mask="##0,000" data-mask-reverse="true" required>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label>Valor Unitário</label>
                            <div class="input-group">
                                <div class="input-group-prepend">
                                    <span class="input-group-text">R$</span>
                                </div>
                                <input type="text" name="valores_unitarios_troca[]" class="form-control valor-unitario-produto-troca" value="0,00" data-mask="#.##0,00" data-mask-reverse="true" required>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-1">
                        <div class="form-group">
                            <label>&nbsp;</label>
                            <button type="button" class="btn btn-danger btn-block btn-sm btn-remover-produto">
                                <i class="ri-delete-bin-line"></i>
                            </button>
                        </div>
                    </div>
                </div>
            `;
            
            // Adicionar HTML interno ao elemento
            $produtoTroca.html(htmlInterno);
            
            // Adicionar elemento ao container
            $('#produtos-troca-container').append($produtoTroca);
            
            // Preencher o select com os produtos disponíveis
            const $select = $produtoTroca.find('.select-produto-troca');
            
            // Adicionando as opções de produtos do PHP sem usar string com diretiva Blade
            @foreach(\App\Models\Produto::where('empresa_id', $venda->empresa_id)->where('status', 1)->orderBy('nome')->get() as $produto)
                $select.append($('<option>', {
                    value: "{{ $produto->id }}",
                    'data-valor': "{{ $produto->valor_venda ?? $produto->valor_unitario }}",
                    text: "{{ $produto->nome }}"
                }));
            @endforeach
            
            // Inicializar o Select2 para o seletor de produtos
            $select.select2();
            
            console.log('Campo de produto adicionado com sucesso');
        }

        // Função para remover campo de produto
        $(document).on('click', '.btn-remover-produto', function() {
            console.log('Removendo produto da troca...');
            $(this).closest('.produto-troca').remove();
            console.log('Produto removido, recalculando totais...');
            calcularTotais();
            atualizarResumoProdutos();
            console.log('Totais recalculados após remoção');
        });

        // Validação do formulário antes de enviar
        $('form').submit(function(e) {
            let isValid = true;
            let message = '';
            
            // Verificar se pelo menos um item foi selecionado
            if ($('.check-item:checked').length === 0) {
                isValid = false;
                message = 'Selecione pelo menos um item para devolução.';
            }
            
            // Verificar se pelo menos um produto foi adicionado para troca
            if ($('.produto-troca').length === 0) {
                isValid = false;
                message = 'Adicione pelo menos um produto para troca.';
            }
            
            // Verificar se todos os produtos têm um valor selecionado
            $('.select-produto-troca').each(function() {
                if ($(this).val() === '') {
                    isValid = false;
                    message = 'Selecione um produto para cada item de troca.';
                }
            });
            
            // Verificar se foi selecionado um tipo de pagamento
            if ($('#tipo_pagamento').val() === '') {
                isValid = false;
                message = 'Selecione um método de pagamento.';
            }
            
            // Verificar se foi selecionado um cliente quando o pagamento for vale crédito
            if ($('#tipo_pagamento').val() === '00' && $('#cliente_id').val() === '') {
                isValid = false;
                message = 'Selecione um cliente para o crédito em loja.';
            }
            
            // Para crediário, não precisamos validar cliente
            if (temCrediario && $('#tipo_pagamento').val() === '06') {
                // Crediário é válido, não precisa de validação adicional
            } else if (!$temCrediario && $('#tipo_pagamento').val() === '06') {
                isValid = false;
                message = 'Crediário não é um método válido para esta venda.';
            }
            
            if (!isValid) {
                e.preventDefault();
                alert(message);
            }
        });
    });
</script>
@endsection 