<?php

namespace App\Http\Controllers;

use App\Models\Caixa;
use App\Models\CategoriaProduto;
use App\Models\Marca;
use App\Models\Empresa;
use App\Models\Nfce;
use App\Models\ConfigGeral;
use App\Models\Produto;
use App\Models\VendaSuspensa;
use App\Models\TefMultiPlusCard;
use App\Models\User;
use App\Models\Contigencia;
use App\Models\UsuarioEmissao;
use App\Models\UsuarioEmpresa;
use App\Models\Funcionario;
use Illuminate\Http\Request;
use App\Services\CupomNaoFiscal;
use App\Services\CupomNaoFiscalPosprint;
use App\Utils\EstoqueUtil;
use Illuminate\Support\Facades\Auth;
use App\Models\ComissaoVenda;
use App\Models\ItemNfce;
use App\Models\Devolucao;
use App\Models\Estoque;
use App\Models\ItemListaPreco;

use Barryvdh\DomPDF\Facade\Pdf;

class FrontBoxController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function __construct(EstoqueUtil $util)
    {
        $this->util = $util;
        $this->middleware('permission:pdv_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:pdv_edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:pdv_view', ['only' => ['show', 'index']]);
        $this->middleware('permission:pdv_delete', ['only' => ['destroy']]);
    }

    private function setNumeroSequencial(){
        $docs = Nfce::where('empresa_id', request()->empresa_id)
        ->where('numero_sequencial', null)
        ->get();

        $last = Nfce::where('empresa_id', request()->empresa_id)
        ->orderBy('numero_sequencial', 'desc')
        ->where('numero_sequencial', '>', 0)->first();
        $numero = $last != null ? $last->numero_sequencial : 0;
        $numero++;

        foreach($docs as $d){
            $d->numero_sequencial = $numero;
            $d->save();
            $numero++;
        }
    }

    private function getContigencia($empresa_id){
        $active = Contigencia::
        where('empresa_id', $empresa_id)
        ->where('status', 1)
        ->where('documento', 'NFCe')
        ->first();
        return $active;
    }

    private function corrigeNumeros($empresa_id){

        $item = UsuarioEmissao::where('usuario_empresas.empresa_id', request()->empresa_id)
        ->join('usuario_empresas', 'usuario_empresas.usuario_id', '=', 'usuario_emissaos.usuario_id')
        ->select('usuario_emissaos.*')
        ->where('usuario_emissaos.usuario_id', get_id_user())
        ->first();

        if($item != null){
            return;
        }
        
        $empresa = Empresa::findOrFail($empresa_id);
        if($empresa->ambiente == 1){
            $numero = $empresa->numero_ultima_nfce_producao;
        }else{
            $numero = $empresa->numero_ultima_nfce_homologacao;
        }
        
        if($numero){
            Nfce::where('estado', 'novo')
            ->where('empresa_id', $empresa_id)
            ->update(['numero' => $numero+1]);
        }
    }

    public function index(Request $request)
    {
        $locais = __getLocaisAtivoUsuario();
        $locais = $locais->pluck(['id']);
        
        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $cliente_id = $request->get('cliente_id');
        $estado = $request->get('estado');
        $local_id = $request->get('local_id');
        $tem_devolucao = $request->get('tem_devolucao');

        $this->setNumeroSequencial();
        $this->corrigeNumeros($request->empresa_id);

        $data = Nfce::with(['cliente', 'itens'])
        ->where('empresa_id', request()->empresa_id)
        ->when(!empty($start_date), function ($query) use ($start_date) {
            return $query->whereDate('created_at', '>=', $start_date);
        })
        ->when(!empty($end_date), function ($query) use ($end_date) {
            return $query->whereDate('created_at', '<=', $end_date);
        })
        ->when(!empty($cliente_id), function ($query) use ($cliente_id) {
            return $query->where('cliente_id', $cliente_id);
        })
        ->when($estado != "", function ($query) use ($estado) {
            return $query->where('estado', $estado);
        })
        ->when($local_id, function ($query) use ($local_id) {
            return $query->where('local_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->whereIn('local_id', $locais);
        })
        ->when($tem_devolucao == '1', function ($query) {
            return $query->where('tem_devolucao', 1);
        });

        $somaValores = (clone $data)->whereIn('estado', ['novo', 'aprovado', 'rejeitado'])->with('itens')->get()->sum(function($item){
            return $item->itens->sum('sub_total');
        });

        $somaDescontos = (clone $data)->whereIn('estado', ['novo', 'aprovado', 'rejeitado'])->sum('desconto');
        $somaCancelamentos = (clone $data)->where('estado', 'cancelado')->sum('total');

        $recebimentosQuery = (clone $data)->whereIn('estado', ['novo', 'aprovado', 'rejeitado']);
        $somaRecebimentos = $recebimentosQuery->with('fatura')->get()->sum(function($item) {
            $troco = $item->troco > 0 ? $item->troco : 0;
            return $item->fatura->sum('valor') - $troco;
        });

        $vendasQuery = (clone $data)->whereIn('estado', ['novo', 'aprovado', 'rejeitado']);
        $somaTrocos = $vendasQuery->with(['fatura', 'itens'])->get()->sum(function($item) {
            if ($item->tipo_pagamento == '01') {
                $valorRecebido = $item->fatura->sum('valor');
                $valorVenda = $item->itens->sum('sub_total');
                $troco = $valorRecebido - $valorVenda;
                return $troco > 0 ? $troco : 0;
            }
            return 0;
        });

        $somaRejeitados = (clone $data)->where('estado', 'rejeitado')->with('itens')->get()->sum(function($item){
            return $item->itens->sum('sub_total');
        });

        $data = $data->orderBy('created_at', 'desc')
        ->paginate(env("PAGINAZIONE"));
        
        $contigencia = $this->getContigencia(request()->empresa_id);

        return view('front_box.index', compact('data', 'contigencia', 'somaValores', 'somaDescontos', 'somaCancelamentos', 'somaRecebimentos', 'somaTrocos', 'somaRejeitados'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        // Define cabeçalhos para prevenir cache no PDV
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Cache-Control: post-check=0, pre-check=0', false);
        header('Pragma: no-cache');
        header('Expires: Sun, 19 Nov 1978 05:00:00 GMT');
        
        if (!__isCaixaAberto()) {
            session()->flash("flash_warning", "Abrir caixa antes de continuar!");
            return redirect()->route('caixa.create');
        }
        $caixa = __isCaixaAberto();
        $categorias = CategoriaProduto::where('empresa_id', request()->empresa_id)->get();

        $abertura = Caixa::where('empresa_id', request()->empresa_id)
        ->where('usuario_id', get_id_user())
        ->where('status', 1)
        ->first();

        $config = Empresa::findOrFail(request()->empresa_id);
        if($config == null){
            session()->flash("flash_warning", "Configure antes de continuar!");
            return redirect()->route('config.index');
        }

        if($config->natureza_id_pdv == null){
            session()->flash("flash_warning", "Configure a natureza de operação padrão para continuar!");
            return redirect()->route('config.index');
        }

        $funcionarios = Funcionario::where('empresa_id', request()->empresa_id)->get();

        $config = ConfigGeral::where('empresa_id', request()->empresa_id)->first();
        $tiposPagamento = Nfce::tiposPagamento();
        // dd($tiposPagamento);
        if($config != null && $config->tipos_pagamento_pdv){
            $tipos_pagamento_pdv = json_decode($config->tipos_pagamento_pdv, true);
            if(is_array($tipos_pagamento_pdv) && sizeof($tipos_pagamento_pdv) > 0){
                $temp = [];
                foreach($tiposPagamento as $key => $t){
                    if(in_array($t, $tipos_pagamento_pdv)){
                        $temp[$key] = $t;
                    }
                }
                $tiposPagamento = $temp;
            }
        }

        $item = null;
        $isVendaSuspensa = 0;
        $title = 'Nova Venda - PDV';

        if(isset($request->venda_suspensa)){
            $item = VendaSuspensa::findOrfail($request->venda_suspensa);
            $isVendaSuspensa = 1;
            $title = 'Venda Suspensa';
        }

        $configTef = TefMultiPlusCard::where('empresa_id', request()->empresa_id)
        ->where('status', 1)
        ->where('usuario_id', Auth::user()->id)
        ->first();

        $view = 'front_box.create';
        $produtos = [];
        $marcas = [];
        if($config != null && $config->modelo == 'compact'){
            $view = 'front_box.create2';
            $categorias = CategoriaProduto::where('empresa_id', request()->empresa_id)
            ->where('categoria_id', null)
            ->orderBy('nome', 'asc')
            ->paginate(4);

            $marcas = Marca::where('empresa_id', request()->empresa_id)
            ->orderBy('nome', 'asc')
            ->paginate(4);

            $produtos = Produto::select('produtos.*', \DB::raw('sum(quantidade) as quantidade'))
            ->where('empresa_id', request()->empresa_id)
            ->where('produtos.status', 1)
            ->where('status', 1)
            ->leftJoin('item_nfces', 'item_nfces.produto_id', '=', 'produtos.id')
            ->groupBy('produtos.id')
            ->orderBy('quantidade', 'desc')
            ->join('produto_localizacaos', 'produto_localizacaos.produto_id', '=', 'produtos.id')
            ->where('produto_localizacaos.localizacao_id', $caixa->localizacao->id)
            ->paginate(12);
        }
        // Usar o primeiro local ativo (LOJA) em vez do local do caixa
        $primeiroLocal = __getLocalAtivo();
        $local_id = $primeiroLocal ? $primeiroLocal->id : $caixa->local_id;
        
        // Buscar locais disponíveis para seleção no PDV
        $locais = __getLocaisAtivos();

        $itensConsignacao = null;
        if ($request->has('consignacao_id')) {
            \Log::debug('PDV - Recebido consignacao_id:', ['consignacao_id' => $request->consignacao_id]);
            $consignacao = \App\Models\Consignacao::with(['itens.produto'])->where('empresa_id', request()->empresa_id)->find($request->consignacao_id);
            \Log::debug('PDV - Consignacao encontrada:', ['consignacao' => $consignacao]);
            if ($consignacao) {
                $itensConsignacao = $consignacao->itens->map(function($item) {
                    return [
                        'produto_id' => $item->produto_id,
                        'produto_variacao_id' => $item->produto_variacao_id,
                        'quantidade' => $item->quantidade - $item->quantidade_devolvida,
                        'nome' => $item->produto->nome ?? $item->descricao,
                        'valor_unitario' => $item->valor_unitario,
                        'codigo_barras' => $item->produto->codigo_barras ?? null,
                    ];
                })->filter(function($item) {
                    return $item['quantidade'] > 0;
                })->values()->toArray();
                \Log::debug('PDV - Itens consignacao gerados:', ['itensConsignacao' => $itensConsignacao]);
            } else {
                \Log::warning('PDV - Consignacao não encontrada ou sem itens.', ['consignacao_id' => $request->consignacao_id]);
            }
        }
        if(empty($itensConsignacao)) {
            \Log::warning('PDV - Nenhum item de consignação enviado para a view.', ['consignacao_id' => $request->consignacao_id ?? null]);
        }

        return view($view, compact('categorias', 'abertura', 
            'funcionarios', 'caixa', 'config', 'tiposPagamento', 'item', 'isVendaSuspensa', 'title', 
            'configTef', 'marcas', 'produtos', 'local_id', 'locais', 'itensConsignacao'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            // Log para debug
            \Log::info('FrontBoxController store - Verificando pré-venda:', [
                'pre_venda_id' => $request->pre_venda_id,
                'request_all' => $request->all()
            ]);
            
            // Processar a venda normalmente (delegar para o VendaController)
            $vendaController = new \App\Http\Controllers\API\PDV\VendaController($this->util, app(\App\Utils\ContaEmpresaUtil::class));
            $response = $vendaController->store($request);
            
            // Se a venda foi criada com sucesso e tem pre_venda_id, marcar como finalizada
            if ($response->getStatusCode() == 200) {
                $vendaData = json_decode($response->getContent(), true);
                \Log::info('FrontBoxController store - Venda criada:', [
                    'venda_data' => $vendaData,
                    'pre_venda_id' => $request->pre_venda_id,
                    'response_status' => $response->getStatusCode()
                ]);
                
                if ($request->pre_venda_id && $vendaData && isset($vendaData['id'])) {
                    \Log::info('FrontBoxController store - Chamando marcarPreVendaComoFinalizada:', [
                        'pre_venda_id' => $request->pre_venda_id,
                        'venda_id' => $vendaData['id']
                    ]);
                    $this->marcarPreVendaComoFinalizada($request->pre_venda_id, $vendaData['id']);
                } else {
                    \Log::warning('FrontBoxController store - Não foi possível marcar pré-venda:', [
                        'pre_venda_id' => $request->pre_venda_id,
                        'venda_data' => $vendaData,
                        'has_pre_venda_id' => $request->pre_venda_id ? 'sim' : 'não',
                        'has_venda_data' => $vendaData ? 'sim' : 'não',
                        'has_venda_id' => $vendaData && isset($vendaData['id']) ? 'sim' : 'não'
                    ]);
                }
            } else {
                \Log::error('FrontBoxController store - Erro na criação da venda:', [
                    'response_status' => $response->getStatusCode(),
                    'response_content' => $response->getContent()
                ]);
            }
            
            return $response;
            
        } catch (\Exception $e) {
            \Log::error('FrontBoxController store - Erro:', [
                'error' => $e->getMessage(),
                'pre_venda_id' => $request->pre_venda_id ?? null
            ]);
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $data = Nfce::with(['itens.produto', 'cliente'])->findOrFail($id);
        
        // Carregar informações sobre devoluções
        $devolucoes = Devolucao::where('venda_id', $id)
            ->where('tipo_venda', 'nfce')
            ->with(['itens.produto'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Processar os itens para mostrar informações sobre devoluções
        foreach ($data->itens as $item) {
            // Adicionar informação se o item foi devolvido ou trocado
            if ($item->status == 1) {
                $item->status_devolucao = 'Devolvido';
                $item->classe_status = 'text-danger';
            } else if ($item->status == 2) {
                $item->status_devolucao = 'Trocado';
                $item->classe_status = 'text-warning';
                
                // Buscar o item substituto
                if ($item->item_substituto_id) {
                    $item->substituto = ItemNfce::with('produto')->find($item->item_substituto_id);
                }
            } else {
                if ($item->quantidade_devolvida > 0) {
                    $item->status_devolucao = 'Parcialmente Devolvido';
                    $item->classe_status = 'text-warning';
                    $item->quantidade_restante = $item->quantidade - $item->quantidade_devolvida;
                } else {
                    $item->status_devolucao = 'Normal';
                    $item->classe_status = 'text-success';
                }
            }
        }
        
        return view('front_box.show', compact('data', 'devolucoes'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $item = Nfce::
        with(['itens', 'cliente'])
        ->findOrFail($id);
        if (!__isCaixaAberto()) {
            session()->flash("flash_warning", "Abrir caixa antes de continuar!");
            return redirect()->route('caixa.create');
        }
        $categorias = CategoriaProduto::where('empresa_id', request()->empresa_id)->get();

        $abertura = Caixa::where('usuario_id', get_id_user())
        ->where('status', 1)
        ->first();

        $config = Empresa::findOrFail(request()->empresa_id);
        if($config == null){
            session()->flash("flash_warning", "Configure antes de continuar!");
            return redirect()->route('config.index');
        }

        if($config->natureza_id_pdv == null){
            session()->flash("flash_warning", "Configure a natureza de operação padrão para continuar!");
            return redirect()->route('config.index');
        }

        $tiposPagamento = Nfce::tiposPagamento();
        // dd($tiposPagamento);
        if($config != null && $config->tipos_pagamento_pdv){
            $tipos_pagamento_pdv = json_decode($config->tipos_pagamento_pdv, true);
            if(is_array($tipos_pagamento_pdv) && sizeof($tipos_pagamento_pdv) > 0){
                $temp = [];
                foreach($tiposPagamento as $key => $t){
                    if(in_array($t, $tipos_pagamento_pdv)){
                        $temp[$key] = $t;
                    }
                }
                $tiposPagamento = $temp;
            }
        }

        $funcionarios = Funcionario::where('empresa_id', request()->empresa_id)->get();
        $cliente = $item->cliente;
        $funcionario = $item->funcionario;
        $caixa = __isCaixaAberto();
        $isVendaSuspensa = 0;
        $config = Empresa::findOrFail(request()->empresa_id);

        $view = 'front_box.edit';
        $produtos = [];
        $marcas = [];
        $config = ConfigGeral::where('empresa_id', request()->empresa_id)->first();
        
        if($config != null && $config->modelo == 'compact'){

            $view = 'front_box.edit2';
            $categorias = CategoriaProduto::where('empresa_id', request()->empresa_id)
            ->where('categoria_id', null)
            ->orderBy('nome', 'asc')
            ->paginate(4);

            $marcas = Marca::where('empresa_id', request()->empresa_id)
            ->orderBy('nome', 'asc')
            ->paginate(4);

            $produtos = Produto::select('produtos.*', \DB::raw('sum(quantidade) as quantidade'))
            ->where('empresa_id', request()->empresa_id)
            ->where('produtos.status', 1)
            ->where('status', 1)
            ->leftJoin('item_nfces', 'item_nfces.produto_id', '=', 'produtos.id')
            ->groupBy('produtos.id')
            ->orderBy('quantidade', 'desc')
            ->join('produto_localizacaos', 'produto_localizacaos.produto_id', '=', 'produtos.id')
            ->where('produto_localizacaos.localizacao_id', $caixa->localizacao->id)
            ->paginate(12);
        }

        $local_id = $caixa->local_id;

        return view($view, compact('categorias', 'abertura', 'funcionarios', 'item', 'cliente', 'funcionario', 
            'caixa', 'isVendaSuspensa', 'tiposPagamento', 'config', 'produtos', 'categorias', 'marcas', 'local_id'));
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $item = Nfce::findOrFail($id);
        try {
            $descricaoLog = "#$item->numero_sequencial - R$ " . __moeda($item->total);
            // Devolve estoque
            foreach ($item->itens as $i) {
                if ($i->produto && $i->produto->gerenciar_estoque) {
                    $this->util->incrementaEstoque($i->produto_id, $i->quantidade, $i->variacao_id, $item->local_id);
                    
                    // Registrar movimentação de estoque (histórico)
                    $this->util->movimentacaoProduto(
                        $i->produto_id, 
                        $i->quantidade, 
                        'incremento', 
                        $item->id, 
                        'cancelamento_nfce_pdv', 
                        \Auth::user()->id,
                        $i->variacao_id
                    );
                }
            }
            // Marca como cancelado
            $item->estado = 'cancelado';
            $item->save();
            // Marca faturas como canceladas
            foreach ($item->fatura as $fatura) {
                $fatura->observacao = ($fatura->observacao ? $fatura->observacao . ' | ' : '') . 'cancelada';
                $fatura->save();
            }
            // Marca contas a receber como canceladas
            foreach ($item->contaReceber as $conta) {
                $conta->status = 99; // status 99 = cancelada
                $conta->observacao = ($conta->observacao ? $conta->observacao . ' | ' : '') . 'cancelada';
                $conta->save();
            }
            // Remove comissão
            $comissao = \App\Models\ComissaoVenda::where('empresa_id', $item->empresa_id)
                ->where('nfce_id', $item->id)->first();
            if($comissao){
                $comissao->delete();
            }
            __createLog(request()->empresa_id, 'PDV', 'cancelar', $descricaoLog);
            session()->flash("flash_success", "Venda cancelada!");
        } catch (\Exception $e) {
            __createLog(request()->empresa_id, 'PDV', 'erro', $e->getMessage());
            session()->flash("flash_error", 'Algo deu errado: '. $e->getMessage());
        }
        return redirect()->route('frontbox.index');
    }

    public function destroySuspensa(string $id)
    {
        $item = VendaSuspensa::findOrFail($id);
        try {

            $item->itens()->delete();
            $item->delete();
            session()->flash("flash_success", "Registro removido!");
        } catch (\Exception $e) {
            // echo $e->getMessage() . '<br>' . $e->getLine();
            // die;
            session()->flash("flash_error", 'Algo deu errado.', $e->getMessage());
        }
        return redirect()->back();
    }

    // public function imprimirNaoFiscal($id)
    // {
    //     $item = Nfce::with('itens.produto')->findOrFail($id);
    //     $config = Empresa::where('id', $item->empresa_id)
    //     ->first();

    //     // Carrega as configurações de impressão
    //     $config_impressao = ConfigGeral::where('empresa_id', $item->empresa_id)->first();

    //     $config = __objetoParaEmissao($config, $item->local_id);
        
    //     $usuario = UsuarioEmpresa::find(get_id_user());
        
    //     // Verificar se os itens têm valores corretos
    //     $valorTotalItens = 0;
        
    //     // Filtra itens com quantidade zero ou negativa
    //     $itensValidos = collect();
    //     foreach($item->itens as $itemVenda) {
    //         // Pula itens com quantidade zero ou negativa
    //         if ($itemVenda->quantidade <= 0) {
    //             continue;
    //         }
            
    //         // Adicionamos uma altura padrão para os itens
    //         $itemVenda->height = 5;
            
    //         // Verificar se temos valores válidos para valor_unitario
    //         if (empty($itemVenda->valor_unitario) || $itemVenda->valor_unitario <= 0) {
    //             if (!empty($itemVenda->sub_total) && $itemVenda->sub_total > 0 && $itemVenda->quantidade > 0) {
    //                 // Calcula o valor unitário a partir do subtotal
    //                 $itemVenda->valor_unitario = $itemVenda->sub_total / $itemVenda->quantidade;
    //             } else {
    //                 // Tenta usar o campo valor, se existir
    //                 $itemVenda->valor_unitario = $itemVenda->valor ?? 0;
    //             }
    //         }
            
    //         // Garantir que o subtotal esteja correto
    //         if (empty($itemVenda->sub_total) || $itemVenda->sub_total <= 0) {
    //             $itemVenda->sub_total = $itemVenda->valor_unitario * $itemVenda->quantidade;
    //         }
            
    //         $valorTotalItens += $itemVenda->sub_total;
    //         $itensValidos->push($itemVenda);
    //     }
        
    //     // Substitui a coleção original de itens pela coleção filtrada
    //     $item->itens = $itensValidos;
        
    //     // Filtra métodos de pagamento com valor zero
    //     if (sizeof($item->fatura) > 0) {
    //         $faturasValidas = collect();
    //         foreach ($item->fatura as $fatura) {
    //             if ($fatura->valor > 0) {
    //                 $faturasValidas->push($fatura);
    //             }
    //         }
    //         $item->fatura = $faturasValidas;
    //     }
        
    //     // Se o valor total calculado dos itens for diferente do valor total na NFCe
    //     // e se tivermos acréscimo ou desconto, ajustamos o total
    //     if (abs($item->total - ($valorTotalItens + $item->acrescimo - $item->desconto)) > 0.01) {
    //         $item->total = $valorTotalItens + $item->acrescimo - $item->desconto;
    //     }
        
    //     // Modificação: garantir que valor_recebido e troco estejam definidos no objeto
    //     if ($item->dinheiro_recebido > 0) {
    //         // Se tiver dinheiro_recebido, recalcula o troco
    //         $item->troco = max(0, $item->dinheiro_recebido - $item->total);
    //     }
        
    //     // Tenta impressão térmica (posprint). Se não disponível, usa PDF (Dompdf)
    //     $posprint = new CupomNaoFiscalPosprint();
    //     $escpos = $posprint->buildEscPos($item, $config, 0);
    //     if ($escpos !== null) {
    //         // Retorna ESC/POS como texto plano para ser enviado ao spooler/cliente
    //         return response($escpos)
    //             ->header('Content-Type', 'text/plain');
    //     }

    //     $service = new CupomNaoFiscal();
    //     $pdf = $service->render($item, $config, 0);
    //     header("Content-Disposition: ; filename=CUPOM.pdf");
    //     return response($pdf)
    //     ->header('Content-Type', 'application/pdf');
    // }

    public function imprimirNaoFiscal($id)
    {
        $item = Nfce::with('itens.produto')->findOrFail($id);
        $config = Empresa::where('id', $item->empresa_id)->first();
        $config = __objetoParaEmissao($config, $item->local_id);
        $usuario = UsuarioEmpresa::find(get_id_user());
    
        // ---------- CABEÇALHO ----------
        $enderecoCompleto = trim("{$config->rua}, {$config->numero} - {$config->bairro}");
        $cidadeUF = isset($config->cidade) ? "{$config->cidade->nome} ({$config->cidade->uf})" : "";
    
        $cabecalho = [
            'nome' => $config->nome_fantasia ?? $config->nome,
            'cpf_cnpj' => $config->cpf_cnpj,
            'endereco' => $enderecoCompleto,
            'cidade_uf' => $cidadeUF,
        ];
    
        // ---------- ITENS ----------
        $valorTotalItens = 0;
        $itensValidos = collect();
        foreach($item->itens as $itemVenda) {
            if ($itemVenda->quantidade <= 0) continue;
    
            $itemVenda->height = 5;
    
            if (empty($itemVenda->valor_unitario) || $itemVenda->valor_unitario <= 0) {
                if (!empty($itemVenda->sub_total) && $itemVenda->sub_total > 0 && $itemVenda->quantidade > 0) {
                    $itemVenda->valor_unitario = $itemVenda->sub_total / $itemVenda->quantidade;
                } else {
                    $itemVenda->valor_unitario = $itemVenda->valor ?? 0;
                }
            }
    
            if (empty($itemVenda->sub_total) || $itemVenda->sub_total <= 0) {
                $itemVenda->sub_total = $itemVenda->valor_unitario * $itemVenda->quantidade;
            }
    
            $valorTotalItens += $itemVenda->sub_total;
            $itensValidos->push($itemVenda);
        }
        $item->itens = $itensValidos;
    
        // ---------- FATURAS ----------
        if (sizeof($item->fatura) > 0) {
            $faturasValidas = collect();
            foreach ($item->fatura as $fatura) {
                if ($fatura->valor > 0) $faturasValidas->push($fatura);
            }
            $item->fatura = $faturasValidas;
        }
    
        // ---------- AJUSTE DO TOTAL ----------
        if (abs($item->total - ($valorTotalItens + $item->acrescimo - $item->desconto)) > 0.01) {
            $item->total = $valorTotalItens + $item->acrescimo - $item->desconto;
        }
    
        // ---------- TROCO ----------
        if ($item->dinheiro_recebido > 0) {
            $item->troco = max(0, $item->dinheiro_recebido - $item->total);
        }
    
        // ---------- GERAR PDF 80MM ----------
        $pdf = Pdf::loadView('front_box.nao_fiscal_80mm', [
            'item' => $item,
            'cabecalho' => $cabecalho,
            'usuario' => $usuario,
            'config' => $config,
        ])->setPaper([0, 0, 226.77, 1000], 'portrait'); // 226.77 pts ≈ 80mm largura
    
        return $pdf->stream('cupom_80mm.pdf');
    }

    public function linhaProdutoVendaAdd(Request $request)
    {
        $product = Produto::findOrFail($request->id);
        $lista_id = $request->lista_id;
        $local_id = $request->local_id;

        if($product->variacao_modelo_id){
            return response("produto com variação", 402);
        }
        $qtd = __convert_value_bd($request->qtd);
        if ($qtd <= 0) {
            return response()->json("Quantidade inválida!", 401);
        }
        try{
            $qtd = (float)$qtd+1;
        }catch(\Exception $e){
            // Ignorar
        }
        try {
            $product = __tributacaoProdutoLocalVenda($product, $local_id);

            if($lista_id){
                $itemLista = ItemListaPreco::where('lista_id', $lista_id)
                ->where('produto_id', $product->id)
                ->first();
                if($itemLista != null){
                    $product->valor_unitario = $itemLista->valor;
                }
            }

            // Usar o valor unitário do formulário se fornecido, senão usar o valor padrão do produto
            $value_unit = $request->value_unit ? __convert_value_bd($request->value_unit) : $product->valor_unitario;
            $sub_total = $value_unit * $qtd;
            $variacao_id = $request->variacao_id;
            $product_id = $product->id;

            if ($product->gerenciar_estoque == true && $product->combo) {
                $estoqueMsg = $this->util->verificaEstoqueCombo($product, (float)$qtd);
                if($estoqueMsg != ""){
                    return response()->json($estoqueMsg, 401);
                }
            }
            // Ajuste: tratar local_id e variacao_id como null se vierem vazios, 0 ou 'null'
            if (empty($local_id) || $local_id === 'null' || $local_id === 0 || $local_id === '0') {
                $local_id = null;
            }
            if (empty($variacao_id) || $variacao_id === 'null' || $variacao_id === 0 || $variacao_id === '0') {
                $variacao_id = null;
            }

            if ($product->gerenciar_estoque == true && !$product->combo) {
                $estoque = Estoque::where('produto_id', $product->id)
                    ->when($variacao_id !== null, function ($q) use ($variacao_id) {
                        return $q->where('produto_variacao_id', $variacao_id);
                    })
                    ->when($variacao_id === null, function ($q) {
                        return $q->whereNull('produto_variacao_id');
                    })
                    ->when($local_id !== null, function ($q) use ($local_id) {
                        return $q->where('local_id', $local_id);
                    })
                    ->when($local_id === null, function ($q) {
                        return $q->whereNull('local_id');
                    })
                    ->first();
                if ($estoque == null) {
                    return response()->json("Produto sem estoque", 401);
                }
                
                // Verificar se há estoque suficiente
                if ($estoque->quantidade < $qtd) {
                    return response()->json("Estoque insuficiente! Disponível: " . $estoque->quantidade, 401);
                }
                
                // Fazer a baixa de estoque imediatamente
                $this->util->reduzEstoque($product->id, $qtd, $variacao_id, $local_id);
            }
            $variacao = null;
            $qtd = __moeda($qtd);
            return view('front_box.partials.row_frontBox', 
                compact('product', 'qtd', 'value_unit', 'sub_total', 'variacao_id', 'variacao'));
        } catch (\Exception $e) {
            return response()->json($e->getMessage(), 401);
        }
    }

    /**
     * Debita o estoque de um produto (usado pelo PDV em tempo real)
     */
    public function debitarEstoque(Request $request)
    {
        try {
            $produto_id = $request->produto_id;
            $quantidade = $request->quantidade;
            $variacao_id = $request->variacao_id ?? null;
            $local_id = $request->local_id ?? null;

            \Log::info('=== DEBITAR ESTOQUE FRONTEND ===');
            \Log::info('Request completo: ' . json_encode($request->all()));
            \Log::info('Local ID do request: ' . ($local_id ?? 'NULL'));
            \Log::info('Produto ID: ' . $produto_id);
            \Log::info('Quantidade: ' . $quantidade);

            $this->util->reduzEstoque($produto_id, $quantidade, $variacao_id, $local_id);

            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 400);
        }
    }

    /**
     * Restaura o estoque de um produto (usado pelo PDV em tempo real)
     */
    public function restaurarEstoque(Request $request)
    {
        try {
            $produto_id = $request->produto_id;
            $quantidade = $request->quantidade;
            $variacao_id = $request->variacao_id ?? null;
            $local_id = $request->local_id ?? null;

            $this->util->incrementaEstoque($produto_id, $quantidade, $variacao_id, $local_id);

            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 400);
        }
    }
    
    /**
     * Marca a pré-venda como finalizada
     */
    private function marcarPreVendaComoFinalizada($preVendaId, $vendaId = null)
    {
        try {
            \Log::info('FrontBoxController - Tentando marcar pré-venda como finalizada:', [
                'pre_venda_id' => $preVendaId,
                'venda_id' => $vendaId
            ]);
            
            $preVenda = \App\Models\PreVenda::find($preVendaId);
            
            if ($preVenda) {
                $preVenda->status = 0; // Marca como finalizada
                $preVenda->venda_id = $vendaId; // Associa com a venda criada
                $preVenda->save();
                
                \Log::info('FrontBoxController - Pré-venda finalizada com sucesso:', [
                    'pre_venda_id' => $preVendaId,
                    'venda_id' => $vendaId,
                    'status' => 0,
                    'pre_venda_original_status' => $preVenda->getOriginal('status')
                ]);
            } else {
                \Log::warning('FrontBoxController - Pré-venda não encontrada:', [
                    'pre_venda_id' => $preVendaId
                ]);
            }
        } catch (\Exception $e) {
            \Log::error('FrontBoxController - Erro ao marcar pré-venda como finalizada:', [
                'pre_venda_id' => $preVendaId,
                'venda_id' => $vendaId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    public function buscarProdutos(Request $request)
    {
        $pesquisa = $request->input('pesquisa');
        $produtos = Produto::where('empresa_id', request()->empresa_id)
            ->where('nome', 'LIKE', "%$pesquisa%")
            ->get();
        return response()->json($produtos, 200);
    }

    public function buscarConsignacao(Request $request)
    {
        $id = $request->codigo;
        $empresa_id = $request->empresa_id;
        $consignacao = \App\Models\Consignacao::with(['cliente', 'funcionario', 'itens.produto'])
            ->where('empresa_id', $empresa_id)
            ->find($id);
        if (!$consignacao) {
            return response()->json('Consignação não encontrada', 404);
        }
        $itens = [];
        foreach ($consignacao->itens as $item) {
            $itens[] = [
                'produto_id' => $item->produto_id,
                'variacao_id' => $item->produto_variacao_id,
                'produto_nome' => $item->produto->nome ?? $item->descricao,
                'quantidade' => (float)$item->quantidade - (float)$item->quantidade_devolvida,
                'valor_unitario' => (float)$item->valor_unitario,
                'sub_total' => (float)$item->sub_total,
                'de_consignacao' => true,
            ];
        }
        return response()->json([
            'id' => $consignacao->id,
            'codigo' => $consignacao->id,
            'cliente_id' => $consignacao->cliente_id,
            'cliente_nome' => $consignacao->cliente->nome_fantasia ?? $consignacao->cliente->razao_social ?? '',
            'cliente_cpf_cnpj' => $consignacao->cliente->cpf_cnpj ?? '',
            'funcionario_id' => $consignacao->funcionario_id,
            'funcionario_nome' => $consignacao->funcionario->nome ?? '',
            'observacao' => $consignacao->observacao,
            'desconto' => 0,
            'acrescimo' => 0,
            'itens' => $itens,
            'valor_total' => (float)$consignacao->valor_total,
        ]);
    }
}
