<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\FinanceiroPlano;
use App\Models\Empresa;
use App\Models\Revenda;
use App\Models\RecebimentoPlano;
use App\Models\Plano;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class FinanceiroPlanoController extends Controller
{
    public function index(Request $request){

        $empresa = $request->get('empresa');
        $revenda = $request->get('revenda');
        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $status_pagamento = $request->get('status_pagamento');
        $pagamento_start_date = $request->get('pagamento_start_date');
        $pagamento_end_date = $request->get('pagamento_end_date');
        
        // Função para aplicar os mesmos filtros em todas as consultas
        $applyBaseFilters = function($query) use ($empresa, $revenda, $start_date, $end_date) {
            if (!empty($empresa)) {
                $query->where('empresa_id', $empresa);
            }
            
            if (!empty($revenda)) {
                $query->where('revenda_id', $revenda);
            }
            
            if (!empty($start_date)) {
                $query->whereDate('created_at', '>=', $start_date);
            }
            
            if (!empty($end_date)) {
                $query->whereDate('created_at', '<=', $end_date);
            }
            
            return $query;
        };
        
        // Função para aplicar filtro de data de pagamento
        $applyPaymentDateFilter = function($query) use ($pagamento_start_date, $pagamento_end_date) {
            if (!empty($pagamento_start_date) || !empty($pagamento_end_date)) {
                $query->whereHas('recebimentos', function($q) use ($pagamento_start_date, $pagamento_end_date) {
                    if (!empty($pagamento_start_date)) {
                        $q->whereDate('data_pagamento', '>=', $pagamento_start_date);
                    }
                    if (!empty($pagamento_end_date)) {
                        $q->whereDate('data_pagamento', '<=', $pagamento_end_date);
                    }
                });
            }
            return $query;
        };
        
        $query = FinanceiroPlano::with(['empresa', 'plano', 'revenda', 'recebimentos']);
        $query = $applyBaseFilters($query);
        
        if (!empty($status_pagamento)) {
            $query->where('status_pagamento', $status_pagamento);
        }
        
        // Aplica filtro de data de pagamento
        $query = $applyPaymentDateFilter($query);
        
        $data = $query->orderBy('id', 'desc')->paginate(env("PAGINACAO"));

        // Calcular os totais aplicando os mesmos filtros
        // Para pendentes, aplicar filtros básicos, sem considerar data de pagamento
        $somaPendenteQuery = FinanceiroPlano::query();
        $somaPendenteQuery = $applyBaseFilters($somaPendenteQuery);
        $somaPendente = $somaPendenteQuery->where('status_pagamento', 'pendente')->sum('valor');
        
        // Para valor recebido, considerar filtros de data de pagamento se existirem
        if (!empty($pagamento_start_date) || !empty($pagamento_end_date)) {
            // Se tem filtro de data de pagamento, soma o valor_pago da tabela recebimento_planos
            $somaRecebido = RecebimentoPlano::whereHas('financeiroPlano', function ($query) use ($empresa, $revenda, $start_date, $end_date) {
                // Aplica os mesmos filtros básicos
                if (!empty($empresa)) {
                    $query->where('empresa_id', $empresa);
                }
                
                if (!empty($revenda)) {
                    $query->where('revenda_id', $revenda);
                }
                
                if (!empty($start_date)) {
                    $query->whereDate('created_at', '>=', $start_date);
                }
                
                if (!empty($end_date)) {
                    $query->whereDate('created_at', '<=', $end_date);
                }
            })
            ->when(!empty($pagamento_start_date), function ($query) use ($pagamento_start_date) {
                return $query->whereDate('data_pagamento', '>=', $pagamento_start_date);
            })
            ->when(!empty($pagamento_end_date), function ($query) use ($pagamento_end_date) {
                return $query->whereDate('data_pagamento', '<=', $pagamento_end_date);
            })
            ->sum('valor_pago');
        } else {
            // Caso contrário, usa a lógica original com os filtros básicos
            $somaRecebidoQuery = FinanceiroPlano::query();
            $somaRecebidoQuery = $applyBaseFilters($somaRecebidoQuery);
            $somaRecebido = $somaRecebidoQuery->where('status_pagamento', 'recebido')->sum('valor');
        }
        
        // Para cancelados, aplicar os filtros básicos 
        $somaCanceladoQuery = FinanceiroPlano::query();
        $somaCanceladoQuery = $applyBaseFilters($somaCanceladoQuery);
        $somaCancelado = $somaCanceladoQuery->where('status_pagamento', 'cancelado')->sum('valor');
        
        // Para repasse, considerar todos os filtros e somar os valores da tabela recebimento_planos
        if (!empty($pagamento_start_date) || !empty($pagamento_end_date)) {
            // Se tem filtro de data de pagamento, soma o valor_repasse da tabela recebimento_planos
            $somaRepasse = RecebimentoPlano::whereHas('financeiroPlano', function ($query) use ($empresa, $revenda, $start_date, $end_date) {
                // Aplica os mesmos filtros básicos
                if (!empty($empresa)) {
                    $query->where('empresa_id', $empresa);
                }
                
                if (!empty($revenda)) {
                    $query->where('revenda_id', $revenda);
                }
                
                if (!empty($start_date)) {
                    $query->whereDate('created_at', '>=', $start_date);
                }
                
                if (!empty($end_date)) {
                    $query->whereDate('created_at', '<=', $end_date);
                }
            })
            ->when(!empty($pagamento_start_date), function ($query) use ($pagamento_start_date) {
                return $query->whereDate('data_pagamento', '>=', $pagamento_start_date);
            })
            ->when(!empty($pagamento_end_date), function ($query) use ($pagamento_end_date) {
                return $query->whereDate('data_pagamento', '<=', $pagamento_end_date);
            })
            ->sum('valor_repasse');
        } else {
            // Caso contrário, soma os valores de repasse dos financeiros com status recebido 
            // aplicando os filtros básicos (empresa, revenda, datas)
            $somaRepasseQuery = FinanceiroPlano::query();
            $somaRepasseQuery = $applyBaseFilters($somaRepasseQuery);
            // Somente considerar os registros com status recebido e que possuam valor de repasse
            $somaRepasse = $somaRepasseQuery->where('status_pagamento', 'recebido')
                                          ->whereNotNull('valor_repasse')
                                          ->sum('valor_repasse');
        }

        if($empresa){
            $empresa = Empresa::findOrFail($empresa);
        }
        
        if($revenda){
            $revenda = Revenda::findOrFail($revenda);
        }
        
        return view('financeiro_plano.index', compact('data', 'somaPendente', 'somaRecebido', 'somaCancelado', 'somaRepasse', 'empresa', 'revenda'));
    }

    public function edit($id){
        $item = FinanceiroPlano::with(['empresa', 'plano', 'revenda'])->findOrFail($id);
        return view('financeiro_plano.edit', compact('item'));
    }

    public function update(Request $request, $id){
        $item = FinanceiroPlano::findOrFail($id);
        try {

            $request->merge([
                'valor' => __convert_value_bd($request->valor),
                'valor_repasse' => __convert_value_bd($request->valor_repasse),
            ]);

            $item->fill($request->except(['empresa_id']))->save();
            session()->flash("flash_success", "Registro alterado com sucesso!");
        } catch (\Exception $e) {
            session()->flash("flash_error", 'Algo deu errado: '. $e->getMessage());
        }
        return redirect()->route('financeiro-plano.index');
    }

    public function destroy($id)
    {
        $item = FinanceiroPlano::findOrFail($id);

        try {
            $item->delete();
            session()->flash("flash_success", "Registro removido com sucesso!");
        } catch (\Exception $e) {
            session()->flash("flash_error", 'Algo deu errado: '. $e->getMessage());
        }
        return redirect()->route('financeiro-plano.index');
    }
    
    public function formReceber($id)
    {
        $item = FinanceiroPlano::with(['empresa', 'plano', 'revenda'])->findOrFail($id);
        $formasPagamento = Plano::formasPagamento();
        return view('financeiro_plano.receber', compact('item', 'formasPagamento'));
    }
    
    public function receber(Request $request, $id)
    {
        $item = FinanceiroPlano::findOrFail($id);
        
        try {
            $valorPago = __convert_value_bd($request->valor_pago);
            $valorRepasse = null;
            
            if ($request->has('valor_repasse') && !empty($request->valor_repasse)) {
                $valorRepasse = __convert_value_bd($request->valor_repasse);
            } elseif ($item->valor_repasse) {
                // Se não for informado valor de repasse mas existir no financeiro, usar o mesmo valor
                $valorRepasse = $item->valor_repasse;
            }
            
            // Registrar o recebimento
            RecebimentoPlano::create([
                'financeiro_plano_id' => $item->id,
                'data_pagamento' => $request->data_pagamento,
                'valor_pago' => $valorPago,
                'valor_repasse' => $valorRepasse,
                'forma_pagamento' => $request->forma_pagamento,
                'observacao' => $request->observacao,
                'user_id' => Auth::id()
            ]);
            
            // Atualizar o status do financeiro para recebido
            $item->status_pagamento = 'recebido';
            $item->save();
            
            session()->flash("flash_success", "Pagamento recebido com sucesso!");
        } catch (\Exception $e) {
            session()->flash("flash_error", 'Algo deu errado: ' . $e->getMessage());
        }
        
        return redirect()->route('financeiro-plano.index');
    }
    
    public function historicoRecebimentos(Request $request)
    {
        $empresa = $request->get('empresa');
        $revenda = $request->get('revenda');
        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        
        // Obter todas as revendas para preencher o select
        $revendas = Revenda::where('status', 1)->orderBy('nome')->get();
        
        $query = RecebimentoPlano::with(['financeiroPlano.empresa', 'financeiroPlano.plano', 'financeiroPlano.revenda', 'usuario']);
        
        // Filtro por empresa
        if (!empty($empresa)) {
            $query->whereHas('financeiroPlano', function($q) use ($empresa) {
                $q->where('empresa_id', $empresa);
            });
        }
        
        // Filtro por revenda
        if (!empty($revenda)) {
            $query->whereHas('financeiroPlano', function($q) use ($revenda) {
                $q->where('revenda_id', $revenda);
            });
        }
        
        // Filtros por data
        if (!empty($start_date)) {
            $query->whereDate('data_pagamento', '>=', $start_date);
        }
        
        if (!empty($end_date)) {
            $query->whereDate('data_pagamento', '<=', $end_date);
        }
        
        $recebimentos = $query->orderBy('data_pagamento', 'desc')
            ->paginate(env("PAGINACAO"));
            
        // Cálculo dos totais
        $totalQuery = clone $query;
        $totalRecebido = $totalQuery->sum('valor_pago');
        
        $totalRepasseQuery = clone $query;
        $totalRepasse = $totalRepasseQuery->sum('valor_repasse');
        
        if($empresa){
            $empresa = Empresa::findOrFail($empresa);
        }
        
        if($revenda){
            $revenda = Revenda::findOrFail($revenda);
        }
        
        return view('financeiro_plano.historico', compact('recebimentos', 'totalRecebido', 'totalRepasse', 'empresa', 'revenda', 'revendas'));
    }
    
    /**
     * Sincroniza valores de repasse anteriores para registros da revenda selecionada
     */
    public function sincronizarRepasseRevenda(Request $request)
    {
        $revenda = $request->get('revenda');
        
        if (empty($revenda)) {
            session()->flash("flash_error", "É necessário selecionar uma revenda para sincronizar");
            return redirect()->back();
        }
        
        try {
            // Busca recebimentos sem valor de repasse para a revenda selecionada
            $recebimentosSemRepasse = RecebimentoPlano::whereNull('valor_repasse')
                ->whereHas('financeiroPlano', function ($query) use ($revenda) {
                    $query->where('revenda_id', $revenda);
                })
                ->get();
                
            $contador = 0;
            
            DB::beginTransaction();
            
            foreach ($recebimentosSemRepasse as $recebimento) {
                // Verifica se o financeiro relacionado possui valor de repasse
                if ($recebimento->financeiroPlano && $recebimento->financeiroPlano->valor_repasse) {
                    $recebimento->valor_repasse = $recebimento->financeiroPlano->valor_repasse;
                    $recebimento->save();
                    $contador++;
                }
            }
            
            DB::commit();
            
            if ($contador > 0) {
                session()->flash("flash_success", "{$contador} recebimentos foram atualizados com valores de repasse");
            } else {
                session()->flash("flash_info", "Não foram encontrados recebimentos sem valor de repasse para sincronizar");
            }
        } catch (\Exception $e) {
            DB::rollBack();
            session()->flash("flash_error", "Erro ao sincronizar dados: " . $e->getMessage());
        }
        
        return redirect()->route('financeiro-plano.historico-recebimentos', ['revenda' => $revenda]);
    }
}
