<?php
/**
 * Diagnóstico de hora do servidor para resolver problemas com NFCe
 * Este script verifica a sincronização de hora do servidor para ajudar a resolver 
 * problemas com cancelamento de NFCe
 */

// Definir timezone para Brasil/São Paulo
date_default_timezone_set('America/Sao_Paulo');

echo "====================================================\n";
echo "DIAGNÓSTICO DE HORA DO SERVIDOR PARA PROBLEMAS DE NFCE\n";
echo "====================================================\n\n";

// Verificar timezone configurado
echo "CONFIGURAÇÕES DE TIMEZONE:\n";
echo "- Timezone configurado no PHP: " . date_default_timezone_get() . "\n";
echo "- Timezone do servidor (system): " . exec('date +%Z') . "\n";
echo "- Data/hora atual (PHP): " . date('Y-m-d H:i:s') . "\n";
echo "- Data/hora formatada para XML: " . date('Y-m-d\TH:i:sP') . "\n\n";

// Verificar timestamp do sistema
echo "TIMESTAMP DO SISTEMA:\n";
$system_time = time();
echo "- Timestamp do PHP (time()): $system_time\n";
echo "- Data correspondente: " . date('Y-m-d H:i:s', $system_time) . "\n\n";

// Verificar hora usando serviços externos
echo "COMPARAÇÃO COM SERVIÇOS EXTERNOS:\n";

// Usando worldtimeapi.org
echo "1. World Time API (worldtimeapi.org):\n";
try {
    $ntp_response = @file_get_contents('http://worldtimeapi.org/api/timezone/America/Sao_Paulo');
    if ($ntp_response !== false) {
        $ntp_data = json_decode($ntp_response);
        if ($ntp_data) {
            $ntp_time = $ntp_data->unixtime;
            echo "   - Timestamp NTP: $ntp_time\n";
            echo "   - Data NTP: " . date('Y-m-d H:i:s', $ntp_time) . "\n";
            
            $diff = abs($system_time - $ntp_time);
            echo "   - Diferença: $diff segundos\n";
            if ($diff > 60) {
                echo "   [ALERTA] Diferença maior que 1 minuto detectada!\n";
            }
        } else {
            echo "   - Erro ao decodificar resposta do servidor NTP\n";
        }
    } else {
        echo "   - Não foi possível conectar ao servidor NTP\n";
    }
} catch (Exception $e) {
    echo "   - Erro: " . $e->getMessage() . "\n";
}
echo "\n";

// Usando time.is API
echo "2. Google (via DNS):\n";
try {
    $start = microtime(true);
    $host = 'time.google.com';
    $timeout = 3;
    
    if (function_exists('socket_create')) {
        $socket = socket_create(AF_INET, SOCK_STREAM, SOL_TCP);
        if ($socket !== false) {
            socket_set_option($socket, SOL_SOCKET, SO_RCVTIMEO, ['sec' => $timeout, 'usec' => 0]);
            socket_set_option($socket, SOL_SOCKET, SO_SNDTIMEO, ['sec' => $timeout, 'usec' => 0]);
            
            $result = @socket_connect($socket, $host, 13);
            if ($result !== false) {
                $response = '';
                while ($data = socket_read($socket, 1024)) {
                    $response .= $data;
                }
                echo "   - Resposta do servidor de tempo: $response\n";
            } else {
                echo "   - Não foi possível conectar ao servidor de tempo\n";
            }
            socket_close($socket);
        } else {
            echo "   - Não foi possível criar socket\n";
        }
    } else {
        echo "   - Função socket_create não disponível\n";
    }
    
    // Outro método: DNS
    echo "   - Consulta DNS para time.google.com...\n";
    $dns_start = microtime(true);
    $ip = gethostbyname($host);
    $dns_time = microtime(true) - $dns_start;
    echo "   - IP resolvido: $ip (tempo: " . round($dns_time * 1000, 2) . "ms)\n";
} catch (Exception $e) {
    echo "   - Erro: " . $e->getMessage() . "\n";
}
echo "\n";

// Testes de conexão com a SEFAZ
echo "TESTE DE CONEXÃO COM SEFAZ:\n";
try {
    $sefaz_hosts = [
        'nfe.fazenda.gov.br',
        'www1.nfe.fazenda.gov.br',
        'nfce.fazenda.gov.br',
    ];
    
    foreach ($sefaz_hosts as $host) {
        echo "- Testando $host:\n";
        $ping_start = microtime(true);
        $ip = gethostbyname($host);
        $ping_time = microtime(true) - $ping_start;
        
        if ($ip != $host) {
            echo "  - Resolvido para IP: $ip (tempo: " . round($ping_time * 1000, 2) . "ms)\n";
            
            // Tentar conexão HTTP
            $http_start = microtime(true);
            $port = 443;
            $connection = @fsockopen('ssl://' . $host, $port, $errno, $errstr, 5);
            
            if (is_resource($connection)) {
                $http_time = microtime(true) - $http_start;
                echo "  - Conexão HTTPS estabelecida (tempo: " . round($http_time * 1000, 2) . "ms)\n";
                fclose($connection);
            } else {
                echo "  - Falha na conexão HTTPS: ($errno) $errstr\n";
            }
        } else {
            echo "  - Não foi possível resolver o host\n";
        }
    }
} catch (Exception $e) {
    echo "- Erro ao testar SEFAZ: " . $e->getMessage() . "\n";
}

echo "\n====================================================\n";
echo "SUGESTÕES:\n";
echo "1. Se houver diferença significativa de horário, sincronize o servidor com NTP\n";
echo "2. Verifique se o timezone está configurado corretamente\n";
echo "3. Se a conexão com a SEFAZ estiver lenta, isso pode causar timeouts\n";
echo "4. Execute este script periodicamente para monitorar sincronização\n";
echo "====================================================\n"; 