<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\ComissaoVenda;
use App\Models\Funcionario;
use App\Models\ContaPagar;
use App\Models\ItemContaEmpresa;
use App\Utils\ContaEmpresaUtil;
use Illuminate\Support\Facades\DB;

class ComissaoController extends Controller
{
    public function index(Request $request){
        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $status = $request->get('status');
        $funcionario_id = $request->get('funcionario_id');

        $data = ComissaoVenda::
        when(!empty($start_date), function ($query) use ($start_date) {
            return $query->whereDate('created_at', '>=', $start_date);
        })
        ->when(!empty($end_date), function ($query) use ($end_date,) {
            return $query->whereDate('created_at', '<=', $end_date);
        })
        ->when(!empty($funcionario_id), function ($query) use ($funcionario_id) {
            return $query->where('funcionario_id', $funcionario_id);
        })
        ->when($status != '', function ($query) use ($status) {
            return $query->where('status', $status);
        })
        ->where('empresa_id', $request->empresa_id)
        ->orderBy('created_at', 'desc')->paginate(50);

        $sumComissaoPendente = ComissaoVenda::
        where('status', 0)
        ->where('empresa_id', $request->empresa_id)
        ->sum('valor');

        $sumComissaoPago = ComissaoVenda::
        where('status', 1)
        ->where('empresa_id', $request->empresa_id)
        ->sum('valor');

        $sumVendas = ComissaoVenda::
        where('empresa_id', $request->empresa_id)
        ->sum('valor_venda');

        $funcionario = null;
        if($funcionario_id){
            $funcionario = Funcionario::findOrFail($funcionario_id);
        }

        return view('comissao_venda.index', 
            compact('data', 'sumComissaoPendente', 'sumComissaoPago', 'funcionario', 'sumVendas'));
    }

    public function edit($id){
        $item = ComissaoVenda::findOrfail($id);

        $item->status = 1;
        $item->save();
        session()->flash("flash_success", "Comissão paga!");
        return redirect()->back();
    }

    public function payMultiple(Request $request){
        try {
            DB::transaction(function () use ($request) {
                $valorTotal = 0;
                
                // Marcar comissões como pagas
                for($i=0; $i<sizeof($request->check); $i++){
                    $item = ComissaoVenda::findOrfail($request->check[$i]);
                    $item->status = 1;
                    $item->save();
                    $valorTotal += $item->valor;
                }

                $local_id = null;
                $caixa = __isCaixaAberto();
                if($caixa != null){
                    $local_id = $caixa->local_id;
                }else{
                    $local_id = __getLocalAtivo()->id;
                }

                if($request->gerar_conta){
                    // Criar conta a pagar (não paga)
                    ContaPagar::create([
                        'empresa_id' => $request->empresa_id,
                        'valor_integral' => __convert_value_bd($request->valor_integral),
                        'data_vencimento' => $request->data_vencimento,
                        'observacao' => $request->observacao,
                        'tipo_pagamento' => $request->tipo_pagamento,
                        'descricao' => $request->descricao,
                        'status' => $request->status,
                        'local_id' => $local_id
                    ]);
                    session()->flash("flash_success", "Comissões adicionadas no conta a pagar!");

                }else{
                    // Pagamento direto - criar conta a pagar PAGA para aparecer no caixa
                    ContaPagar::create([
                        'empresa_id' => $request->empresa_id,
                        'valor_integral' => $valorTotal,
                        'data_vencimento' => date('Y-m-d'), // Vencimento hoje
                        'observacao' => $request->observacao ?? 'Pagamento de comissões de vendas',
                        'tipo_pagamento' => $request->tipo_pagamento ?? '01', // Dinheiro por padrão
                        'descricao' => 'Pagamento de comissões de vendas',
                        'status' => 1, // PAGO
                        'local_id' => $local_id,
                        'caixa_id' => $caixa ? $caixa->id : null // Associar ao caixa atual
                    ]);
                    
                    // Registrar log da operação
                    $descricaoLog = "Pagamento de comissões: R$ " . __moeda($valorTotal);
                    __createLog($request->empresa_id, 'Comissão', 'pagamento', $descricaoLog);
                    
                    session()->flash("flash_success", "Comissões pagas! Valor abatido do caixa geral.");
                }
            });
            
        } catch (\Exception $e) {
            session()->flash("flash_error", "Erro ao processar pagamento: " . $e->getMessage());
        }
        
        return redirect()->back();
    }

    public function destroy($id)
    {
        $item = ComissaoVenda::findOrFail($id);
        try {

            $item->delete();
            session()->flash('flash_success', 'Deletado com sucesso');
        } catch (\Exception $e) {
            session()->flash('flash_error', 'Não foi possível deletar' . $e->getMessage());
        }
        return redirect()->back();
    }

}
