<?php

namespace App\Http\Controllers\API\PDV;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Cliente;
use App\Models\ContaReceber;
use Illuminate\Support\Facades\DB;
use App\Models\Caixa;
use Illuminate\Support\Facades\Auth;
use App\Utils\ContaEmpresaUtil;
use App\Models\ItemContaEmpresa;
use Carbon\Carbon;

class ContaReceberController extends Controller
{
    protected $util;

    public function __construct(ContaEmpresaUtil $util = null)
    {
        $this->util = $util ?? new ContaEmpresaUtil();
    }

    /**
     * Busca contas a receber por CPF/CNPJ do cliente
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function buscarPorCpfCnpj(Request $request)
    {
        // Validar entrada
        $request->validate([
            'cpf_cnpj' => 'required|string',
        ]);

        $cpfCnpj = $request->cpf_cnpj;
        
        // Limpar a formatação do CPF/CNPJ
        $cpfCnpjLimpo = preg_replace('/[^0-9]/', '', $cpfCnpj);
        
        // Buscar o cliente pelo CPF/CNPJ usando tanto a versão formatada quanto a limpa
        $cliente = Cliente::where('empresa_id', $request->empresa_id)
            ->where(function($query) use ($cpfCnpj, $cpfCnpjLimpo) {
                // Busca pelo CPF/CNPJ exatamente como foi digitado
                $query->where('cpf_cnpj', $cpfCnpj)
                    // Ou busca pelo CPF/CNPJ sem formatação
                    ->orWhere('cpf_cnpj', $cpfCnpjLimpo)
                    // Ou busca pelo CPF/CNPJ formatado ou não formatado contido no campo
                    ->orWhere('cpf_cnpj', 'like', '%' . $cpfCnpj . '%')
                    ->orWhere('cpf_cnpj', 'like', '%' . $cpfCnpjLimpo . '%');
            })
            ->first();
            
        if (!$cliente) {
            return response()->json([
                'success' => false,
                'message' => 'Cliente não encontrado',
                'data' => []
            ]);
        }
        
        // Buscar contas a receber pendentes do cliente
        $contasReceber = ContaReceber::where('empresa_id', $request->empresa_id)
            ->where('cliente_id', $cliente->id)
            ->where(function($query) {
                $query->where('status', ContaReceber::STATUS_PENDENTE) // Contas pendentes
                      ->orWhere('status', ContaReceber::STATUS_PARCIAL); // Contas recebidas parcialmente
            })
            ->orderBy('data_vencimento', 'asc')
            ->with(['categoria'])
            ->get();
            
        // Calcular saldo disponível de crédito
        $saldoDisponivel = $cliente->valor_credito ?? 0;
        
        // Formatar os dados para retornar
        $contasFormatadas = $contasReceber->map(function ($conta) {
            return [
                'id' => $conta->id,
                'descricao' => $conta->descricao,
                'valor_integral' => $conta->valor_integral,
                'valor_recebido' => $conta->valor_recebido ?? 0,
                'valor_pendente' => $conta->valorPendente(),
                'data_vencimento' => $conta->data_vencimento,
                'data_vencimento_formatada' => __data_pt($conta->data_vencimento),
                'categoria' => $conta->categoria ? $conta->categoria->nome : 'Sem categoria',
                'status_formatado' => $conta->status_formatado,
                'dias_atraso' => $conta->dias_atraso_formatado
            ];
        });
        
        return response()->json([
            'success' => true,
            'message' => 'Contas a receber encontradas',
            'data' => [
                'cliente' => [
                    'id' => $cliente->id,
                    'nome' => $cliente->razao_social,
                    'cpf_cnpj' => $cliente->cpf_cnpj,
                    'email' => $cliente->email,
                    'telefone' => $cliente->telefone,
                    'saldo_disponivel' => $saldoDisponivel
                ],
                'contas' => $contasFormatadas
            ]
        ]);
    }

    /**
     * Exibe formulário para receber múltiplas contas selecionadas
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function prepararRecebimento(Request $request)
    {
        $request->validate([
            'conta_ids' => 'required|array',
            'conta_ids.*' => 'required|integer|exists:conta_recebers,id'
        ]);

        // Buscar contas selecionadas
        $contas = ContaReceber::whereIn('id', $request->conta_ids)
            ->where('empresa_id', $request->empresa_id)
            ->where('status', '<>', ContaReceber::STATUS_RECEBIDO) // Não totalmente recebidas
            ->where('status', '<>', ContaReceber::STATUS_REPARCELADA) // Não reparceladas
            ->orderBy('data_vencimento', 'asc') // Ordenar por data de vencimento (FIFO)
            ->with(['cliente', 'categoria'])
            ->get();

        if ($contas->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'Nenhuma conta válida selecionada para recebimento',
            ]);
        }

        // Calcular valor total pendente
        $valorTotalPendente = $contas->sum(function($conta) {
            return $conta->valorPendente();
        });

        // Formatar as contas selecionadas para exibição
        $contasFormatadas = $contas->map(function($conta) {
            return [
                'id' => $conta->id,
                'descricao' => $conta->descricao,
                'cliente' => $conta->cliente->razao_social,
                'valor_pendente' => $conta->valorPendente(),
                'valor_pendente_formatado' => __moeda($conta->valorPendente()),
                'data_vencimento' => $conta->data_vencimento,
                'data_vencimento_formatada' => __data_pt($conta->data_vencimento),
                'categoria' => $conta->categoria ? $conta->categoria->nome : 'Sem categoria',
                'status_formatado' => $conta->status_formatado,
                'dias_atraso' => $conta->dias_atraso_formatado
            ];
        });

        // Obter tipos de pagamento disponíveis
        $tiposPagamento = ContaReceber::tiposPagamentoRecebimentos();

        return response()->json([
            'success' => true,
            'message' => 'Contas preparadas para recebimento',
            'data' => [
                'contas' => $contasFormatadas,
                'valor_total_pendente' => $valorTotalPendente,
                'valor_total_pendente_formatado' => __moeda($valorTotalPendente),
                'tipos_pagamento' => $tiposPagamento
            ]
        ]);
    }

    /**
     * Processa o recebimento de múltiplas contas 
     * usando a estratégia FIFO (First In, First Out)
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function receberContas(Request $request)
    {
        $request->validate([
            'conta_ids' => 'required|array',
            'conta_ids.*' => 'required|integer|exists:conta_recebers,id',
            'valor_pago' => 'required|string',
            'data_recebimento' => 'required|date',
            'tipo_pagamento' => 'required|string',
            'data_nova_vencimento' => 'required_if:pagamento_parcial,true|date|nullable',
            'conta_empresa_id' => 'nullable|integer|exists:conta_empresas,id'
        ]);

        // Inicializar variáveis
        $valorPago = __convert_value_bd($request->valor_pago);
        $valorRestante = $valorPago;
        $contasProcessadas = [];
        $pagamentoCompleto = true;
        
        // Verificar se o caixa está aberto
        $caixa = Caixa::where('usuario_id', Auth::id())
                      ->where('status', 1)
                      ->first();
                      
        if (!$caixa) {
            return response()->json([
                'success' => false,
                'message' => 'É necessário abrir o caixa antes de realizar recebimentos.',
            ]);
        }

        try {
            DB::beginTransaction();
            
            // Buscar contas selecionadas ordenadas por data de vencimento (FIFO)
            $contas = ContaReceber::whereIn('id', $request->conta_ids)
                ->where('empresa_id', $request->empresa_id)
                ->where('status', '<>', ContaReceber::STATUS_RECEBIDO)
                ->where('status', '<>', ContaReceber::STATUS_REPARCELADA)
                ->orderBy('data_vencimento', 'asc')
                ->get();
                
            if ($contas->isEmpty()) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Nenhuma conta válida selecionada para recebimento',
                ]);
            }
            
            // Verificar valor total pendente
            $valorTotalPendente = $contas->sum(function($conta) {
                return $conta->valorPendente();
            });
            
            // Processar pagamento das contas por ordem de vencimento (FIFO)
            foreach ($contas as $conta) {
                $valorPendenteConta = $conta->valorPendente();
                
                // Se não houver valor pendente, pular para próxima conta
                if ($valorPendenteConta <= 0) {
                    continue;
                }
                
                // Se não tiver mais valor para pagar, sair do loop
                if ($valorRestante <= 0) {
                    break;
                }
                
                // Determinar quanto será pago dessa conta
                $valorASerPago = min($valorRestante, $valorPendenteConta);
                $isPagamentoTotal = ($valorASerPago >= $valorPendenteConta);
                
                // Registrar o pagamento
                $valorAnteriorRecebido = $conta->valor_recebido ?? 0;
                $conta->valor_recebido = $valorAnteriorRecebido + $valorASerPago;
                $conta->data_recebimento = $request->data_recebimento;
                $conta->tipo_pagamento = $request->tipo_pagamento;
                $conta->caixa_id = $caixa->id;
                
                if ($isPagamentoTotal) {
                    // Marca como totalmente recebida
                    $conta->status = ContaReceber::STATUS_RECEBIDO;
                } else {
                    // Pagamento parcial
                    // Salvar data original de vencimento na primeira vez
                    if (!$conta->data_vencimento_original) {
                        $conta->data_vencimento_original = $conta->data_vencimento;
                    }
                    
                    // Atualiza para status parcial
                    $conta->status = ContaReceber::STATUS_PARCIAL;
                    
                    // Atualizar nova data de vencimento
                    if ($request->filled('data_nova_vencimento')) {
                        $conta->data_vencimento = $request->data_nova_vencimento;
                        $conta->data_nova_vencimento = $request->data_nova_vencimento;
                    }
                    
                    // Indicar que o pagamento não foi completo
                    $pagamentoCompleto = false;
                }
                
                $conta->save();
                
                // Registrar histórico do recebimento
                $this->registrarHistorico(
                    $conta->id, 
                    $valorASerPago, 
                    $request->tipo_pagamento, 
                    $isPagamentoTotal
                );
                
                // Atualizar valor restante
                $valorRestante -= $valorASerPago;
                
                // Adicionar à lista de contas processadas
                $contasProcessadas[] = [
                    'id' => $conta->id,
                    'descricao' => $conta->descricao,
                    'valor_pago' => $valorASerPago,
                    'valor_pago_formatado' => __moeda($valorASerPago),
                    'status' => $isPagamentoTotal ? 'Recebido integralmente' : 'Recebido parcialmente',
                    'valor_restante' => $conta->valorPendente(),
                    'valor_restante_formatado' => __moeda($conta->valorPendente())
                ];
            }
            
            // Registrar entrada no caixa/conta se especificado
            if ($request->filled('conta_empresa_id')) {
                $data = [
                    'conta_id' => $request->conta_empresa_id,
                    'descricao' => "Recebimento de " . count($contasProcessadas) . " contas",
                    'tipo_pagamento' => $request->tipo_pagamento,
                    'valor' => $valorPago,
                    'tipo' => 'entrada'
                ];
                $itemContaEmpresa = ItemContaEmpresa::create($data);
                $this->util->atualizaSaldo($itemContaEmpresa);
            }
            
            // Registrar log da operação
            $descricaoLog = "Recebimento de " . count($contasProcessadas) . " contas, valor: R$ " . __moeda($valorPago);
            __createLog($request->empresa_id, 'Conta a Receber', 'recebimento', $descricaoLog);
            
            DB::commit();
            
            // Mensagem de sucesso de acordo com o tipo de pagamento
            $mensagemSucesso = $pagamentoCompleto 
                ? "Todas as contas foram recebidas integralmente!" 
                : "Recebimento parcial realizado com sucesso!";
            
            return response()->json([
                'success' => true,
                'message' => $mensagemSucesso,
                'data' => [
                    'contas_processadas' => $contasProcessadas,
                    'valor_pago' => $valorPago,
                    'valor_pago_formatado' => __moeda($valorPago),
                    'pagamento_completo' => $pagamentoCompleto
                ]
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Erro ao processar recebimento: ' . $e->getMessage(),
            ]);
        }
    }
    
    /**
     * Registrar o histórico de transações da conta a receber
     */
    private function registrarHistorico($contaReceberID, $valorRecebido, $tipoPagamento, $integral = false)
    {
        try {
            $historico = new \App\Models\ContaReceberHistorico();
            $historico->conta_receber_id = $contaReceberID;
            $historico->valor = $valorRecebido;
            $historico->tipo_movimento = $integral ? 'recebimento_integral' : 'recebimento_parcial';
            $historico->tipo_pagamento = $tipoPagamento;
            $historico->data_movimento = date('Y-m-d H:i:s');
            $historico->usuario_id = auth()->id();
            $historico->save();
            
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Busca contas a receber por ID do cliente
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function buscarPorId(Request $request)
    {
        // Validar entrada
        $request->validate([
            'cliente_id' => 'required|integer',
        ]);

        $clienteId = $request->cliente_id;
        
        // Buscar o cliente pelo ID
        $cliente = Cliente::where('empresa_id', $request->empresa_id)
            ->where('id', $clienteId)
            ->first();
            
        if (!$cliente) {
            return response()->json([
                'success' => false,
                'message' => 'Cliente não encontrado',
                'data' => []
            ]);
        }
        
        // Buscar contas a receber pendentes do cliente
        $contasReceber = ContaReceber::where('empresa_id', $request->empresa_id)
            ->where('cliente_id', $cliente->id)
            ->where(function($query) {
                $query->where('status', ContaReceber::STATUS_PENDENTE) // Contas pendentes
                      ->orWhere('status', ContaReceber::STATUS_PARCIAL); // Contas recebidas parcialmente
            })
            ->orderBy('data_vencimento', 'asc')
            ->with(['categoria'])
            ->get();
            
        // Calcular saldo disponível de crédito
        $saldoDisponivel = $cliente->valor_credito ?? 0;
        
        // Formatar os dados para retornar
        $contasFormatadas = $contasReceber->map(function ($conta) {
            return [
                'id' => $conta->id,
                'descricao' => $conta->descricao,
                'valor_integral' => $conta->valor_integral,
                'valor_recebido' => $conta->valor_recebido ?? 0,
                'valor_pendente' => $conta->valorPendente(),
                'data_vencimento' => $conta->data_vencimento,
                'data_vencimento_formatada' => __data_pt($conta->data_vencimento),
                'categoria' => $conta->categoria ? $conta->categoria->nome : 'Sem categoria',
                'status_formatado' => $conta->status_formatado,
                'dias_atraso' => $conta->dias_atraso_formatado
            ];
        });
        
        return response()->json([
            'success' => true,
            'message' => 'Contas a receber encontradas',
            'data' => [
                'cliente' => [
                    'id' => $cliente->id,
                    'nome' => $cliente->razao_social,
                    'cpf_cnpj' => $cliente->cpf_cnpj,
                    'email' => $cliente->email,
                    'telefone' => $cliente->telefone,
                    'saldo_disponivel' => $saldoDisponivel
                ],
                'contas' => $contasFormatadas
            ]
        ]);
    }
} 