<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use App\Mail\ResumoSemanalNotificacao;
use App\Services\NotificacaoService;
use App\Models\Empresa;
use App\Models\Nfe;
use App\Models\Nfce;
use App\Models\User;
use App\Models\Cliente;
use Carbon\Carbon;

class EnviarResumoSemanal extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'notificacoes:resumo-semanal';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Envia um resumo semanal de atividades do sistema por email';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Iniciando geração do resumo semanal...');
        
        // Define o período da semana passada
        $dataInicio = Carbon::now()->subDays(7)->startOfDay();
        $dataFim = Carbon::now()->endOfDay();
        
        // Coleta estatísticas do sistema
        $estatisticas = $this->coletarEstatisticas($dataInicio, $dataFim);
        
        // Obtém lista de destinatários
        $destinatarios = NotificacaoService::getDestinatarios();
        
        $this->info('Enviando resumo semanal para ' . count($destinatarios) . ' destinatários...');
        
        try {
            foreach ($destinatarios as $dest) {
                Mail::to($dest)->send(new ResumoSemanalNotificacao($estatisticas));
                $this->info("Email enviado com sucesso para {$dest}");
                Log::info("Resumo semanal enviado com sucesso para {$dest}");
            }
            
            $this->info('Resumo semanal enviado com sucesso!');
            return 0;
            
        } catch (\Exception $e) {
            $this->error('Erro ao enviar resumo semanal: ' . $e->getMessage());
            Log::error('Erro ao enviar resumo semanal', [
                'erro' => $e->getMessage()
            ]);
            return 1;
        }
    }
    
    /**
     * Coleta estatísticas do sistema para o período especificado
     * 
     * @param Carbon $dataInicio
     * @param Carbon $dataFim
     * @return array
     */
    private function coletarEstatisticas($dataInicio, $dataFim)
    {
        // Novas empresas cadastradas
        $novasEmpresas = Empresa::whereBetween('created_at', [$dataInicio, $dataFim])->count();
        
        // Novas NFes emitidas
        $novasNfes = Nfe::whereBetween('created_at', [$dataInicio, $dataFim])->count();
        
        // Novas NFCes emitidas
        $novasNfces = Nfce::whereBetween('created_at', [$dataInicio, $dataFim])->count();
        
        // Novos usuários
        $novosUsuarios = User::whereBetween('created_at', [$dataInicio, $dataFim])->count();
        
        // Novos clientes
        $novosClientes = Cliente::whereBetween('created_at', [$dataInicio, $dataFim])->count();
        
        // Total de empresas ativas
        $totalEmpresas = Empresa::count();
        
        // As 5 empresas mais ativas (com mais NFes+NFCes)
        $empresasMaisAtivas = Empresa::withCount(['nfes' => function($query) use ($dataInicio, $dataFim) {
                $query->whereBetween('created_at', [$dataInicio, $dataFim]);
            }, 'nfces' => function($query) use ($dataInicio, $dataFim) {
                $query->whereBetween('created_at', [$dataInicio, $dataFim]);
            }])
            ->orderByRaw('nfes_count + nfces_count DESC')
            ->limit(5)
            ->get();
        
        return [
            'periodo' => [
                'inicio' => $dataInicio->format('d/m/Y'),
                'fim' => $dataFim->format('d/m/Y'),
            ],
            'novas_empresas' => $novasEmpresas,
            'novas_nfes' => $novasNfes,
            'novas_nfces' => $novasNfces,
            'novos_usuarios' => $novosUsuarios,
            'novos_clientes' => $novosClientes,
            'total_empresas' => $totalEmpresas,
            'empresas_mais_ativas' => $empresasMaisAtivas,
            'data_geracao' => Carbon::now()->format('d/m/Y H:i:s')
        ];
    }
} 