@extends('layouts.app', ['title' => 'Emissão de Etiquetas'])

@section('css')
<style>
    .produto-item {
        padding: 10px;
        border-bottom: 1px solid #eee;
        display: flex;
        align-items: center;
        justify-content: space-between;
    }
    
    .list-container {
        max-height: 600px;
        overflow-y: auto;
        border: 1px solid #dee2e6;
        border-radius: 5px;
    }
    
    .btn-transfer {
        margin: 5px 0;
    }
    
    .quantidade-input {
        width: 80px;
        text-align: center;
    }
    
    .produto-nome {
        flex: 1;
        margin-left: 10px;
        margin-right: 10px;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
    }
    
    .img-thumbnail {
        max-width: 40px;
        height: auto;
    }
    
    .actions-column {
        min-width: 120px;
        text-align: center;
    }

    .list-header {
        background-color: #f8f9fa;
        font-weight: bold;
        padding: 10px;
        border-bottom: 2px solid #dee2e6;
        display: flex;
        align-items: center;
    }

    .btn-pesquisar-container {
        display: flex;
        gap: 10px;
        margin-bottom: 20px;
    }

    .search-input {
        flex: 1;
    }

    .item-selecionado {
        background-color: #e9f5ff;
    }

    #lista-selecionados-vazia {
        padding: 20px;
        text-align: center;
        color: #6c757d;
    }
    
    .footer-actions {
        margin-top: 20px;
        text-align: center;
    }
    
    .btn-emitir {
        padding: 10px 30px;
        font-size: 16px;
    }
</style>
@endsection

@section('content')
<div class="mt-3">
    <div class="row">
        <div class="card">
            <div class="card-body">
                <h4 class="mb-3">Emissão de Etiquetas</h4>
                <p class="text-muted">Selecione os produtos para gerar etiquetas. A quantidade padrão é o estoque atual, mas você pode alterar conforme necessário.</p>
                
                <div class="row mb-4">
                    <div class="col-md-12">
                        <div class="btn-pesquisar-container">
                            <input type="text" id="pesquisar-produtos" class="form-control search-input" placeholder="Pesquisar produtos...">
                            <button id="btn-limpar-pesquisa" class="btn btn-secondary">
                                <i class="ri-refresh-line"></i> Limpar
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <!-- Lista de Produtos Disponíveis -->
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header bg-primary text-white">
                                <h5 class="mb-0">Produtos Disponíveis</h5>
                            </div>
                            <div class="card-body p-0">
                                <div class="list-header">
                                    <div class="form-check" style="margin-right: 15px;">
                                        <input class="form-check-input" type="checkbox" id="selecionar-todos">
                                    </div>
                                    <span style="flex: 1;">Produto</span>
                                    <span style="width: 100px; text-align: center;">Estoque</span>
                                    <span class="actions-column">Ação</span>
                                </div>
                                <div class="list-container" id="lista-produtos">
                                    @foreach($produtos as $produto)
                                    <div class="produto-item" data-id="{{ $produto->id }}" data-nome="{{ $produto->nome }}" data-estoque="{{ number_format($produto->estoque ? $produto->estoque->quantidade : 0, 2, ',', '.') }}">
                                        <div class="form-check">
                                            <input class="form-check-input produto-checkbox" type="checkbox" value="{{ $produto->id }}">
                                        </div>
                                        <img src="{{ $produto->img }}" class="img-thumbnail">
                                        <span class="produto-nome" title="{{ $produto->nome }}">{{ $produto->nome }}</span>
                                        <input type="number" min="1" value="{{ intval($produto->estoque ? $produto->estoque->quantidade : 1) }}" class="form-control quantidade-input" data-id="{{ $produto->id }}" aria-label="Quantidade">
                                        <div class="actions-column">
                                            <button type="button" class="btn btn-sm btn-primary btn-adicionar" data-id="{{ $produto->id }}">
                                                <i class="ri-arrow-right-line"></i>
                                            </button>
                                        </div>
                                    </div>
                                    @endforeach
                                </div>
                            </div>
                            <div class="card-footer text-end">
                                <button type="button" class="btn btn-primary" id="adicionar-selecionados">
                                    <i class="ri-arrow-right-line"></i> Adicionar Selecionados
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Lista de Produtos Selecionados -->
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header bg-success text-white">
                                <h5 class="mb-0">Produtos Selecionados</h5>
                            </div>
                            <div class="card-body p-0">
                                <div class="list-header">
                                    <span style="flex: 1; margin-left: 25px;">Produto</span>
                                    <span style="width: 100px; text-align: center;">Quantidade</span>
                                    <span class="actions-column">Ação</span>
                                </div>
                                <div class="list-container" id="lista-selecionados">
                                    <div id="lista-selecionados-vazia">
                                        <i class="ri-information-line fs-3 mb-2"></i>
                                        <p>Nenhum produto selecionado</p>
                                    </div>
                                </div>
                            </div>
                            <div class="card-footer text-end">
                                <button type="button" class="btn btn-danger" id="limpar-selecionados">
                                    <i class="ri-delete-bin-line"></i> Limpar Todos
                                </button>
                            </div>
                        </div>
                    </div>
                </div>

                <form action="{{ route('produtos.emissao-etiquetas-store') }}" method="POST" id="form-etiquetas" class="mt-4">
                    @csrf
                    <div id="produtos-selecionados-inputs"></div>
                    
                    <div class="footer-actions">
                        <button type="submit" class="btn btn-success btn-emitir">
                            <i class="ri-printer-line"></i> Emitir Etiquetas
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

@section('js')
<script>
    $(document).ready(function() {
        // Pesquisa de produtos
        $("#pesquisar-produtos").on("keyup", function() {
            var value = $(this).val().toLowerCase();
            $("#lista-produtos .produto-item").filter(function() {
                $(this).toggle($(this).attr('data-nome').toLowerCase().indexOf(value) > -1);
            });
        });

        // Limpar pesquisa
        $("#btn-limpar-pesquisa").on("click", function() {
            $("#pesquisar-produtos").val('');
            $("#lista-produtos .produto-item").show();
        });

        // Selecionar todos os produtos
        $("#selecionar-todos").on("change", function() {
            const isChecked = $(this).prop('checked');
            $(".produto-checkbox").prop('checked', isChecked);
        });

        // Adicionar produto individual
        $(".btn-adicionar").on("click", function() {
            const id = $(this).data('id');
            const item = $(`#lista-produtos .produto-item[data-id="${id}"]`);
            adicionarProduto(
                id,
                item.data('nome'),
                item.find('.quantidade-input').val()
            );
        });

        // Adicionar produtos selecionados
        $("#adicionar-selecionados").on("click", function() {
            $(".produto-checkbox:checked").each(function() {
                const id = $(this).val();
                const item = $(`#lista-produtos .produto-item[data-id="${id}"]`);
                adicionarProduto(
                    id,
                    item.data('nome'),
                    item.find('.quantidade-input').val()
                );
            });
            // Desmarcar todos após adicionar
            $(".produto-checkbox").prop('checked', false);
            $("#selecionar-todos").prop('checked', false);
        });

        // Função para adicionar produto à lista de selecionados
        function adicionarProduto(id, nome, quantidade) {
            // Verificar se o produto já está na lista
            if ($(`#lista-selecionados .produto-item[data-id="${id}"]`).length > 0) {
                // Atualizar apenas a quantidade
                const inputQuantidade = $(`#lista-selecionados .produto-item[data-id="${id}"] .quantidade-input`);
                inputQuantidade.val(quantidade);
                $(`#lista-selecionados .produto-item[data-id="${id}"]`).addClass('item-selecionado');
                
                // Efeito visual temporário
                setTimeout(function() {
                    $(`#lista-selecionados .produto-item[data-id="${id}"]`).removeClass('item-selecionado');
                }, 1000);
                
                // Atualizar o input oculto também
                $(`#quantidade-${id}`).val(quantidade);
                return;
            }

            // Remover mensagem de lista vazia se for o primeiro item
            if ($("#lista-selecionados-vazia").length > 0) {
                $("#lista-selecionados-vazia").remove();
            }

            // Adicionar item à lista de selecionados
            const itemHtml = `
                <div class="produto-item item-selecionado" data-id="${id}">
                    <img src="${$(`#lista-produtos .produto-item[data-id="${id}"] img`).attr('src')}" class="img-thumbnail">
                    <span class="produto-nome" title="${nome}">${nome}</span>
                    <input type="number" min="1" value="${quantidade}" class="form-control quantidade-input" data-id="${id}" aria-label="Quantidade">
                    <div class="actions-column">
                        <button type="button" class="btn btn-sm btn-danger btn-remover" data-id="${id}">
                            <i class="ri-delete-bin-line"></i>
                        </button>
                    </div>
                </div>
            `;
            $("#lista-selecionados").append(itemHtml);

            // Adicionar inputs ocultos para o form
            $("#produtos-selecionados-inputs").append(`
                <input type="hidden" name="produtos_selecionados[]" value="${id}">
                <input type="hidden" name="quantidades[${id}]" id="quantidade-${id}" value="${quantidade}">
            `);

            // Remover o efeito de seleção após um segundo
            setTimeout(function() {
                $(`#lista-selecionados .produto-item[data-id="${id}"]`).removeClass('item-selecionado');
            }, 1000);

            // Configurar eventos para o novo item
            configurarEventos();
        }

        // Limpar todos os selecionados
        $("#limpar-selecionados").on("click", function() {
            $("#lista-selecionados").html(`
                <div id="lista-selecionados-vazia">
                    <i class="ri-information-line fs-3 mb-2"></i>
                    <p>Nenhum produto selecionado</p>
                </div>
            `);
            $("#produtos-selecionados-inputs").html('');
        });

        // Configurar eventos para itens adicionados dinamicamente
        function configurarEventos() {
            // Evento para remover item da lista
            $(".btn-remover").off("click").on("click", function() {
                const id = $(this).data('id');
                $(`#lista-selecionados .produto-item[data-id="${id}"]`).remove();
                $(`#produtos-selecionados-inputs input[name="produtos_selecionados[]"][value="${id}"]`).remove();
                $(`#produtos-selecionados-inputs input[name="quantidades[${id}]"]`).remove();

                // Mostrar mensagem de lista vazia se não houver mais itens
                if ($("#lista-selecionados .produto-item").length === 0) {
                    $("#lista-selecionados").html(`
                        <div id="lista-selecionados-vazia">
                            <i class="ri-information-line fs-3 mb-2"></i>
                            <p>Nenhum produto selecionado</p>
                        </div>
                    `);
                }
            });

            // Evento para atualizar a quantidade
            $("#lista-selecionados .quantidade-input").off("change").on("change", function() {
                const id = $(this).data('id');
                const quantidade = $(this).val();
                $(`#quantidade-${id}`).val(quantidade);
            });
        }

        // Validação do formulário
        $("#form-etiquetas").on("submit", function(e) {
            if ($("#produtos-selecionados-inputs input").length === 0) {
                e.preventDefault();
                alert("Selecione pelo menos um produto para gerar etiquetas.");
                return false;
            }
        });
    });
</script>
@endsection 