/**
 * Script para garantir que todas as tabelas do sistema sejam responsivas
 * Adiciona automaticamente a classe table-responsive a tabelas que não estão
 * dentro de um container .table-responsive
 */
document.addEventListener('DOMContentLoaded', function() {
    // Encontrar todas as tabelas que não estão dentro de um .table-responsive
    const tables = document.querySelectorAll('table:not(.table-responsive):not(.dataTable)');
    
    tables.forEach(function(table) {
        // Verificar se a tabela já está dentro de um container .table-responsive
        const parentElement = table.parentElement;
        if (!parentElement.classList.contains('table-responsive') && 
            !parentElement.classList.contains('dataTables_wrapper')) {
            
            // Criar um novo elemento div com a classe table-responsive
            const wrapper = document.createElement('div');
            wrapper.classList.add('table-responsive');
            
            // Inserir o wrapper antes da tabela no DOM
            parentElement.insertBefore(wrapper, table);
            
            // Mover a tabela para dentro do wrapper
            wrapper.appendChild(table);
            
            // Garantir que a tabela tenha a classe .table para estilos do Bootstrap
            if (!table.classList.contains('table')) {
                table.classList.add('table');
            }
        }
    });

    // Adicionar classe .table-responsive aos containers .dataTables_wrapper que não têm
    const dataTablesWrappers = document.querySelectorAll('.dataTables_wrapper:not(.table-responsive)');
    dataTablesWrappers.forEach(function(wrapper) {
        wrapper.classList.add('table-responsive');
    });

    // Garantir que tabelas com muitas colunas tenham largura mínima adequada
    const allTables = document.querySelectorAll('table.table');
    allTables.forEach(function(table) {
        const columnCount = table.querySelectorAll('thead th, thead td').length;
        
        // Se a tabela tiver muitas colunas, definir uma largura mínima
        if (columnCount > 6) {
            const minWidth = Math.max(100 * columnCount, 800); // Mínimo de 800px ou 100px por coluna
            table.style.minWidth = minWidth + 'px';
        }
    });
    
    // Tratar elementos com classe grid-table ou grid-view
    const gridElements = document.querySelectorAll('.grid-table, .grid-view');
    gridElements.forEach(function(grid) {
        if (!grid.parentElement.classList.contains('table-responsive') &&
            !grid.classList.contains('table-responsive')) {
            
            // Se o elemento não tiver overflow-x definido, adicione
            const style = window.getComputedStyle(grid);
            if (style.overflowX === 'visible' || style.overflowX === 'hidden') {
                grid.style.overflowX = 'auto';
                grid.style.display = 'block';
                grid.style.width = '100%';
                grid.style.WebkitOverflowScrolling = 'touch';
            }
            
            // Adicionar classe para estilo visual consistente
            grid.classList.add('force-horizontal-scroll');
        }
    });
    
    // Procurar divs que podem conter tabelas mas não têm rolagem adequada
    const containers = document.querySelectorAll('.card-body, .content-container, .content, .container, .container-fluid');
    containers.forEach(function(container) {
        // Verificar se tem alguma tabela dentro
        const hasTables = container.querySelectorAll('table').length > 0;
        
        if (hasTables) {
            // Garantir que o container não impeça a rolagem
            container.style.overflowX = 'visible';
            
            // Verificar tabelas dentro do container
            const tablesInContainer = container.querySelectorAll('table');
            tablesInContainer.forEach(function(table) {
                const columnCount = table.querySelectorAll('th, td').length;
                
                // Se a tabela tiver muitas colunas e não estiver em um wrapper responsivo
                if (columnCount > 5 && !table.parentElement.classList.contains('table-responsive')) {
                    // Verificar se algum ancestral já tem table-responsive
                    let hasResponsiveAncestor = false;
                    let parent = table.parentElement;
                    
                    while (parent && parent !== container) {
                        if (parent.classList.contains('table-responsive')) {
                            hasResponsiveAncestor = true;
                            break;
                        }
                        parent = parent.parentElement;
                    }
                    
                    // Se não tiver ancestral responsivo, adicione um wrapper
                    if (!hasResponsiveAncestor) {
                        const wrapper = document.createElement('div');
                        wrapper.classList.add('table-responsive');
                        
                        table.parentNode.insertBefore(wrapper, table);
                        wrapper.appendChild(table);
                    }
                }
            });
        }
    });

    // Verificação para tabelas carregadas dinamicamente via AJAX
    const observeDOM = (function(){
        const MutationObserver = window.MutationObserver || window.WebKitMutationObserver;
        
        return function(obj, callback){
            if(!obj || obj.nodeType !== 1) return;
            
            if(MutationObserver){
                // define um novo observer
                const mutationObserver = new MutationObserver(callback);
                
                // observa mudanças de childList em obj
                mutationObserver.observe(obj, { childList: true, subtree: true });
                return mutationObserver;
            }
            
            // fallback para navegadores mais antigos
            else if(window.addEventListener){
                obj.addEventListener('DOMNodeInserted', callback, false);
                obj.addEventListener('DOMNodeRemoved', callback, false);
            }
        };
    })();
    
    // Observe o body para novos elementos
    observeDOM(document.body, function(mutations) {
        // Verificar se novas tabelas foram adicionadas
        const newTables = document.querySelectorAll('table:not(.table-responsive):not(.dataTable)');
        newTables.forEach(function(table) {
            const parentElement = table.parentElement;
            if (!parentElement.classList.contains('table-responsive') && 
                !parentElement.classList.contains('dataTables_wrapper')) {
                
                const wrapper = document.createElement('div');
                wrapper.classList.add('table-responsive');
                
                parentElement.insertBefore(wrapper, table);
                wrapper.appendChild(table);
                
                if (!table.classList.contains('table')) {
                    table.classList.add('table');
                }
            }
        });
        
        // Verificar novos elementos grid adicionados dinamicamente
        const newGrids = document.querySelectorAll('.grid-table:not(.force-horizontal-scroll), .grid-view:not(.force-horizontal-scroll)');
        newGrids.forEach(function(grid) {
            grid.classList.add('force-horizontal-scroll');
            grid.style.overflowX = 'auto';
        });
    });
}); 