/**
 * Correção para botões em dispositivos móveis
 * 
 * Este script resolve problemas de clique em botões em dispositivos Android e outros dispositivos móveis
 * onde botões como "Fechar Caixa" e "Salvar" não estão respondendo corretamente.
 * 
 * O problema ocorre principalmente por causa de:
 * 1. Sobreposição de elementos ou problemas de z-index
 * 2. Problemas com propagação de eventos touch/click
 * 3. Delay padrão de 300ms em cliques em dispositivos móveis
 */

(function() {
    document.addEventListener('DOMContentLoaded', function() {
        // Detecta se está em um dispositivo móvel
        const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
        
        if (!isMobile) return; // Se não for dispositivo móvel, não aplica as correções
        
        console.log('Aplicando correções para cliques em dispositivos móveis');
        
        // 1. Remover delay de clique para todos os botões
        document.body.style.touchAction = 'manipulation';
        
        // 2. Lidar com botões dentro de modais (corrige problema com data-bs-dismiss="modal")
        document.querySelectorAll('.modal button[type="submit"], .modal .btn-primary').forEach(button => {
            // Remover atributo data-bs-dismiss se existir, pois está causando problemas
            if (button.getAttribute('data-bs-dismiss') === 'modal') {
                button.removeAttribute('data-bs-dismiss');
                
                // Adicionar evento de clique para fechar o modal manualmente após submissão
                button.addEventListener('click', function(e) {
                    // Permitir que o formulário seja enviado primeiro
                    setTimeout(function() {
                        const modal = button.closest('.modal');
                        if (modal) {
                            const modalInstance = bootstrap.Modal.getInstance(modal);
                            if (modalInstance) {
                                modalInstance.hide();
                            }
                        }
                    }, 100);
                });
            }
        });
        
        // 3. Aplicar correção específica aos botões problemáticos
        const problematicButtons = [
            // Botões específicos reportados com problemas
            document.querySelector('a.btn-danger[href*="caixa.fechar-conta"]'), // Botão "Fechar Caixa"
            document.querySelector('#btn-store'), // Botão "Salvar"
            document.querySelector('#btn-fechar-caixa') // Botão "Salvar Fechamento"
        ];
        
        problematicButtons.forEach(button => {
            if (!button) return;
            
            console.log('Aplicando correção ao botão:', button.textContent.trim());
            
            // Aumenta a área clicável do botão
            button.style.padding = '10px 20px';
            
            // Define uma z-index alta para garantir que o botão esteja acima de outros elementos
            button.style.position = 'relative';
            button.style.zIndex = '100';
            
            // Adiciona uma borda visível para debug em ambiente de desenvolvimento
            if (window.location.hostname.includes('localhost') || window.location.hostname.includes('127.0.0.1')) {
                button.style.border = '2px solid red';
            }
            
            // Adiciona evento de toque dedicado
            button.addEventListener('touchstart', function(e) {
                console.log('Evento touchstart capturado no botão:', this.textContent.trim());
                
                // Prevenir evento padrão para evitar atrasos
                e.preventDefault();
                
                // Se for um link, navegar programaticamente
                if (this.tagName === 'A' && this.href) {
                    console.log('Navegando para:', this.href);
                    window.location.href = this.href;
                } else {
                    // Simular o clique
                    console.log('Simulando clique no botão');
                    this.click();
                }
            }, { passive: false });
        });
        
        // 4. Correção específica para o botão de fechamento de caixa no modal
        const fechamentoCaixaModal = document.getElementById('fechamento_caixa');
        if (fechamentoCaixaModal) {
            const form = fechamentoCaixaModal.querySelector('form');
            if (form) {
                form.addEventListener('submit', function(e) {
                    console.log('Formulário de fechamento de caixa submetido');
                    
                    // Fechar o modal manualmente após submissão
                    setTimeout(function() {
                        const modalInstance = bootstrap.Modal.getInstance(fechamentoCaixaModal);
                        if (modalInstance) {
                            modalInstance.hide();
                        }
                    }, 100);
                });
            }
        }
        
        // 5. Corrigir o botão "Salvar" na tela de abertura de caixa
        const saveCaixaButton = document.querySelector('.card form #btn-store');
        if (saveCaixaButton) {
            console.log('Encontrou botão de salvar caixa:', saveCaixaButton.textContent.trim());
            
            saveCaixaButton.addEventListener('touchstart', function(e) {
                console.log('Evento touchstart no botão de salvar caixa');
                e.preventDefault();
                
                // Encontrar o formulário pai
                const form = this.closest('form');
                if (form) {
                    console.log('Enviando formulário');
                    form.submit();
                } else {
                    console.log('Formulário não encontrado, disparando clique');
                    this.click();
                }
            }, { passive: false });
        }
        
        // 6. Correção genérica para todos os botões de formulário
        document.querySelectorAll('button[type="submit"], input[type="submit"]').forEach(button => {
            button.addEventListener('touchstart', function(e) {
                console.log('Touchstart em botão genérico:', this.textContent.trim());
                e.stopPropagation();
                // Não prevenir default aqui para permitir envio normal do formulário
            }, { passive: true });
            
            // Adicionar classe especial para destacar em ambientes de dev
            if (window.location.hostname.includes('localhost') || window.location.hostname.includes('127.0.0.1')) {
                button.classList.add('touch-enhanced');
                
                // Adicionar estilo CSS para a classe
                if (!document.getElementById('mobile-touch-fix-styles')) {
                    const style = document.createElement('style');
                    style.id = 'mobile-touch-fix-styles';
                    style.textContent = `
                        .touch-enhanced {
                            position: relative;
                        }
                        .touch-enhanced::after {
                            content: '';
                            position: absolute;
                            top: -5px;
                            left: -5px;
                            right: -5px;
                            bottom: -5px;
                            border: 2px dashed rgba(255, 0, 0, 0.5);
                            border-radius: 6px;
                            pointer-events: none;
                        }
                    `;
                    document.head.appendChild(style);
                }
            }
        });
        
        // 7. Adicionar um pequeno indicador visual para mostrar que a correção está ativa
        if (window.location.hostname.includes('localhost') || window.location.hostname.includes('127.0.0.1')) {
            const indicator = document.createElement('div');
            indicator.style.cssText = `
                position: fixed;
                bottom: 10px;
                right: 10px;
                background-color: rgba(255, 0, 0, 0.7);
                color: white;
                padding: 5px 10px;
                border-radius: 5px;
                font-size: 12px;
                z-index: 9999;
            `;
            indicator.textContent = 'Correção de toque ativa';
            document.body.appendChild(indicator);
        }
    });
})(); 