<?php

namespace App\Utils;

use Illuminate\Support\Str;
use App\Models\User;
use App\Models\Produto;
use App\Models\Empresa;
use App\Models\Role;
use App\Models\Permission;
use App\Models\Localizacao;
use App\Models\ProdutoLocalizacao;
use App\Models\UsuarioLocalizacao;
use Illuminate\Support\Facades\Artisan;

class EmpresaUtil
{

	public function defaultPermissions($empresa_id){
		$empresa = Empresa::findOrFail($empresa_id);
		$usuarios = $empresa->usuarios;
		
		$roles = Role::where('empresa_id', null)->get();
		Artisan::call('cache:forget spatie.permission.cache ');
		foreach($roles as $role){

			if($role->name != 'gestor_plataforma'){

				foreach($usuarios as $u){
					$user = $u->usuario;
					$r = Role::create([ 
						'name' => $role->description . '#' . $empresa_id,
						'description' => $role->description,
						'empresa_id' => $empresa_id,
						'guard_name' => 'web', 
						'is_default' => 1,
						'type_user' => 2
					]);
					$permissions = [];
					foreach($role->permissions as $p){
						array_push($permissions, 
							[
								'permission_id' => $p->id,
								'role_id' => $r->id,
							]
						);
					}

					$role->permissions()->attach($permissions);
					$user->assignRole($r->name);

				}
			}
		}
	}

	public function getPermissions($empresa_id){
		// Retorna todas as permissões disponíveis no sistema
		return Permission::orderBy('description')->get();
	}

	public function createPermissions(){
		$count = Permission::count();
		if($count == 0){
			$this->createPermissionsDefault();
		}

		$count = Role::count();
		if($count == 0){
			$this->createRolesDefault();
		}

	}

	private function createPermissionsDefault()
	{
        // Seed the default permissions
		$permissions = Permission::defaultPermissions();

		foreach ($permissions as $permission) {
			Permission::updateOrCreate(
				['name' => $permission['name']],
				$permission
			);
		}

	}

	private function createRolesDefault()
	{
		$superadmin = Role::firstOrCreate([
			'name' => 'gestor_plataforma'
		], [
			'description' => 'Gestor Plataforma',
			'type_user' => 1
		]);
		$superadmin->permissions()->sync(Permission::all());

		$admin = Role::firstOrCreate([
			'name' => 'admin',
		], [
			'description' => 'Admin',
			'type_user' => 2
		]);
		$admin->permissions()->sync(Permission::all());

	}

	public function initLocation($empresa){

		$localizacao = Localizacao::where('empresa_id', $empresa->id)->first();
		if(!$localizacao){
			$localizacao = $empresa->toArray();
			$localizacao['descricao'] = 'BL0001';
			$localizacao['empresa_id'] = $empresa->id;

			$localizacao = Localizacao::create($localizacao);

			foreach($empresa->usuarios as $u){
				UsuarioLocalizacao::updateOrCreate([
					'usuario_id' => $u->usuario_id,
					'localizacao_id' => $localizacao->id
				]);
			}
		}

		$this->initProducts($empresa->id);
		$this->initRegisters($empresa->id);
		
		// Cria configurações fiscais padrão
		$this->initFiscalConfigs($empresa->id);
	}

	private function initProducts($empresa_id){
		$produtos = Produto::where('empresa_id', $empresa_id)->get();
		$localizacao = Localizacao::where('empresa_id', $empresa_id)->first();
		if($localizacao){
			foreach($produtos as $p){
				$produtoLocalizacao = ProdutoLocalizacao::where('produto_id', $p->id)->first();
				if($produtoLocalizacao == null){
					ProdutoLocalizacao::updateOrCreate([
						'produto_id' => $p->id,
						'localizacao_id' => $localizacao->id
					]);
				}
			}
		}
	}

	private function initRegisters($empresa_id){
		$localizacao = Localizacao::where('empresa_id', $empresa_id)->first();

		\App\Models\Nfe::where('empresa_id', $empresa_id)->where('local_id', null)
		->update(['local_id' => $localizacao->id]);

		\App\Models\Nfce::where('empresa_id', $empresa_id)->where('local_id', null)
		->update(['local_id' => $localizacao->id]);

		\App\Models\Cte::where('empresa_id', $empresa_id)->where('local_id', null)
		->update(['local_id' => $localizacao->id]);

		\App\Models\Mdfe::where('empresa_id', $empresa_id)->where('local_id', null)
		->update(['local_id' => $localizacao->id]);

		\App\Models\ContaPagar::where('empresa_id', $empresa_id)->where('local_id', null)
		->update(['local_id' => $localizacao->id]);

		\App\Models\ContaReceber::where('empresa_id', $empresa_id)->where('local_id', null)
		->update(['local_id' => $localizacao->id]);
	}

	public function initUserLocations($user){
		if($user->empresa){
			$empresa_id = $user->empresa->empresa_id;
			$localizacao = Localizacao::where('empresa_id', $empresa_id)->first();
			UsuarioLocalizacao::updateOrCreate([
				'usuario_id' => $user->id,
				'localizacao_id' => $localizacao->id
			]);
		}
	}

	private function initFiscalConfigs($empresa_id) {
		// Verificar se já existem configurações fiscais para não duplicar
		$existeNatureza = \App\Models\NaturezaOperacao::where('empresa_id', $empresa_id)->exists();
		$existeTributacao = \App\Models\PadraoTributacaoProduto::where('empresa_id', $empresa_id)->exists();
		
		// Buscar a empresa para verificar seu regime tributário
		$empresa = Empresa::findOrFail($empresa_id);
		$regimeNormal = ($empresa->tributacao == 'Regime Normal');
		
		// Criar Naturezas de Operação padrão se não existirem
		if (!$existeNatureza) {
			// Valores padrão para Simples Nacional
			$cst_csosn = $regimeNormal ? '00' : '102';
			$cst_pis = $regimeNormal ? '01' : '07';
			$cst_cofins = $regimeNormal ? '01' : '07';
			$perc_icms = $regimeNormal ? 17 : 0;
			$perc_pis = $regimeNormal ? 1.65 : 0;
			$perc_cofins = $regimeNormal ? 7.6 : 0;
			
			// Natureza para Venda de Mercadoria
			\App\Models\NaturezaOperacao::create([
				'empresa_id' => $empresa_id,
				'descricao' => 'Venda de Mercadoria',
				'cst_csosn' => $cst_csosn,
				'cst_pis' => $cst_pis,
				'cst_cofins' => $cst_cofins,
				'cst_ipi' => '99',
				'cfop_estadual' => '5102',
				'cfop_outro_estado' => '6102',
				'cfop_entrada_estadual' => '1102',
				'cfop_entrada_outro_estado' => '2102',
				'perc_icms' => $perc_icms,
				'perc_pis' => $perc_pis,
				'perc_cofins' => $perc_cofins,
				'perc_ipi' => 0,
				'padrao' => 1
			]);
			
			// Natureza para Devolução de Mercadorias
			\App\Models\NaturezaOperacao::create([
				'empresa_id' => $empresa_id,
				'descricao' => 'Devolução de Mercadorias',
				'cst_csosn' => $cst_csosn,
				'cst_pis' => $cst_pis,
				'cst_cofins' => $cst_cofins,
				'cst_ipi' => '99',
				'cfop_estadual' => '5202',
				'cfop_outro_estado' => '6202',
				'cfop_entrada_estadual' => '1202',
				'cfop_entrada_outro_estado' => '2202',
				'perc_icms' => $perc_icms,
				'perc_pis' => $perc_pis,
				'perc_cofins' => $perc_cofins,
				'perc_ipi' => 0,
				'padrao' => 0
			]);
			
			// Natureza para Remessa para consumo/industrialização
			\App\Models\NaturezaOperacao::create([
				'empresa_id' => $empresa_id,
				'descricao' => 'Remessa para consumo/industrialização',
				'cst_csosn' => $cst_csosn,
				'cst_pis' => $cst_pis,
				'cst_cofins' => $cst_cofins,
				'cst_ipi' => '99',
				'cfop_estadual' => '5901',
				'cfop_outro_estado' => '6901',
				'cfop_entrada_estadual' => '1901',
				'cfop_entrada_outro_estado' => '2901',
				'perc_icms' => $perc_icms,
				'perc_pis' => $perc_pis,
				'perc_cofins' => $perc_cofins,
				'perc_ipi' => 0,
				'padrao' => 0
			]);
		}
		
		// Criar Padrão de Tributação de Produtos se não existir
		if (!$existeTributacao) {
			// Valores padrão com base no regime tributário
			$descricao = $regimeNormal ? 'Tributação Padrão Regime Normal' : 'Tributação Padrão Simples Nacional';
			$cst_csosn = $regimeNormal ? '00' : '102';
			$cst_pis = $regimeNormal ? '01' : '07';
			$cst_cofins = $regimeNormal ? '01' : '07';
			$perc_icms = $regimeNormal ? 17 : 0;
			$perc_pis = $regimeNormal ? 1.65 : 0;
			$perc_cofins = $regimeNormal ? 7.6 : 0;
			
			\App\Models\PadraoTributacaoProduto::create([
				'empresa_id' => $empresa_id,
				'descricao' => $descricao,
				'perc_icms' => $perc_icms,
				'perc_pis' => $perc_pis,
				'perc_cofins' => $perc_cofins,
				'perc_ipi' => 0,
				'cst_csosn' => $cst_csosn,
				'cst_pis' => $cst_pis,
				'cst_cofins' => $cst_cofins,
				'cst_ipi' => '99',
				'cfop_estadual' => '5102',
				'cfop_outro_estado' => '6102',
				'cfop_entrada_estadual' => '1102',
				'cfop_entrada_outro_estado' => '2102',
				'perc_red_bc' => 0,
				'cEnq' => '999',
				'padrao' => 1
			]);
		}
	}

	/**
	 * Cria configurações fiscais padrão para uma empresa existente
	 * Útil para empresas que já estão cadastradas e não possuem configurações padrão
	 */
	public function createFiscalConfigsForExistingCompany($empresa_id) {
		$this->initFiscalConfigs($empresa_id);
		return [
			'naturezas' => \App\Models\NaturezaOperacao::where('empresa_id', $empresa_id)->count(),
			'tributacao' => \App\Models\PadraoTributacaoProduto::where('empresa_id', $empresa_id)->count()
		];
	}

}