<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ItemConsignacao extends Model
{
    use HasFactory;

    protected $table = 'item_consignacoes';

    protected $fillable = [
        'consignacao_id',
        'produto_id',
        'produto_variacao_id',
        'quantidade',
        'quantidade_vendida',
        'quantidade_devolvida',
        'valor_unitario',
        'sub_total',
        'observacao'
    ];

    protected $casts = [
        'quantidade' => 'decimal:3',
        'quantidade_vendida' => 'decimal:3',
        'quantidade_devolvida' => 'decimal:3',
        'valor_unitario' => 'decimal:2',
        'sub_total' => 'decimal:2'
    ];

    // Relacionamentos
    public function consignacao()
    {
        return $this->belongsTo(Consignacao::class, 'consignacao_id');
    }

    public function produto()
    {
        return $this->belongsTo(Produto::class, 'produto_id');
    }

    public function produtoVariacao()
    {
        return $this->belongsTo(ProdutoVariacao::class, 'produto_variacao_id');
    }

    // Métodos
    public function getQuantidadeDisponivelAttribute()
    {
        return $this->quantidade - $this->quantidade_vendida - $this->quantidade_devolvida;
    }

    public function getQuantidadeDisponivelParaVendaAttribute()
    {
        return $this->quantidade - $this->quantidade_vendida;
    }

    public function getQuantidadeDisponivelParaDevolucaoAttribute()
    {
        return $this->quantidade_vendida;
    }

    public function getDescricaoAttribute()
    {
        if (!$this->produto) {
            return "Produto não encontrado (ID: {$this->produto_id})";
        }
        
        if ($this->produtoVariacao && $this->produtoVariacao->descricao) {
            return $this->produto->nome . ' - ' . $this->produtoVariacao->descricao;
        }
        return $this->produto->nome;
    }

    public function podeVender($quantidade)
    {
        return $this->quantidade_disponivel_para_venda >= $quantidade;
    }

    public function podeDevolver($quantidade)
    {
        return $this->quantidade_disponivel_para_devolucao >= $quantidade;
    }

    public function registrarVenda($quantidade)
    {
        if ($this->podeVender($quantidade)) {
            $this->quantidade_vendida += $quantidade;
            $this->save();
            return true;
        }
        return false;
    }

    public function registrarDevolucao($quantidade)
    {
        if ($this->podeDevolver($quantidade)) {
            $this->quantidade_devolvida += $quantidade;
            $this->quantidade_vendida -= $quantidade;
            $this->save();
            return true;
        }
        return false;
    }

    public function getValorTotalVendidoAttribute()
    {
        return $this->quantidade_vendida * $this->valor_unitario;
    }

    public function getValorTotalDevolvidoAttribute()
    {
        return $this->quantidade_devolvida * $this->valor_unitario;
    }

    public function getValorTotalDisponivelAttribute()
    {
        return $this->quantidade_disponivel_para_venda * $this->valor_unitario;
    }
} 