<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class ContaReceber extends Model
{
    use HasFactory;

    protected $fillable = [
        'empresa_id', 'nfe_id', 'nfce_id', 'cliente_id', 'descricao', 'valor_integral', 'valor_recebido', 'data_vencimento',
        'data_recebimento', 'status', 'observacao', 'tipo_pagamento', 'caixa_id', 'local_id', 'arquivo',
        'categoria_id', 'prioridade', 'recorrente', 'periodicidade', 'total_parcelas', 'parcela_atual', 'parent_id', 
        'centro_custo_id', 'data_vencimento_original', 'data_nova_vencimento', 'juros_aplicados', 'multa_aplicada'
    ];

    protected $appends = [ 'info', 'status_formatado', 'dias_atraso_formatado' ];

    // Constantes para prioridade
    const PRIORIDADE_BAIXA = 'baixa';
    const PRIORIDADE_MEDIA = 'media';
    const PRIORIDADE_ALTA = 'alta';
    const PRIORIDADE_URGENTE = 'urgente';

    // Constantes para periodicidade
    const PERIODICIDADE_DIARIA = 'diaria';
    const PERIODICIDADE_SEMANAL = 'semanal';
    const PERIODICIDADE_QUINZENAL = 'quinzenal';
    const PERIODICIDADE_MENSAL = 'mensal';
    const PERIODICIDADE_BIMESTRAL = 'bimestral';
    const PERIODICIDADE_TRIMESTRAL = 'trimestral';
    const PERIODICIDADE_SEMESTRAL = 'semestral';
    const PERIODICIDADE_ANUAL = 'anual';

    // Adicionar constantes para os status
    const STATUS_PENDENTE = 0;
    const STATUS_RECEBIDO = 1;
    const STATUS_PARCIAL = 2;
    const STATUS_REPARCELADA = 3;

    public function getInfoAttribute()
    {
        return "Cliente: " . ($this->cliente ? $this->cliente->info : '—') . " - valor: R$ " . __moeda($this->valor_integral) . ", vencimento: " . __data_pt($this->data_vencimento, 0);
    }

    public function getStatusFormatadoAttribute()
    {
        if ($this->status == 1) {
            return "Recebido";
        } elseif ($this->status == 3) {
            return "Reparcelada";
        } elseif ($this->isVencida()) {
            return "Vencida";
        } elseif ($this->isParcial()) {
            return "Parcialmente recebido";
        } else {
            return "Pendente";
        }
    }

    public function getDiasAtrasoFormatadoAttribute()
    {
        return $this->diasAtraso();
    }

    public function localizacao()
    {
        return $this->belongsTo(Localizacao::class, 'local_id');
    }

    public function nfce()
    {
        return $this->belongsTo(Nfce::class, 'nfce_id');
    }
    
    public function cliente()
    {
        return $this->belongsTo(Cliente::class, 'cliente_id');
    }

    public function boleto()
    {
        return $this->hasOne(Boleto::class, 'conta_receber_id');
    }

    public function categoria()
    {
        return $this->belongsTo(CategoriaConta::class, 'categoria_id');
    }

    public function centroCusto()
    {
        return $this->belongsTo(CentroCusto::class, 'centro_custo_id');
    }

    public function anexos()
    {
        return $this->hasMany(AnexoConta::class, 'conta_receber_id');
    }

    public function alocacoes()
    {
        return $this->hasMany(AlocacaoCentroCusto::class, 'conta_receber_id');
    }

    public function contasRecorrentes()
    {
        return $this->hasMany(ContaReceber::class, 'parent_id');
    }

    public function contaPai()
    {
        return $this->belongsTo(ContaReceber::class, 'parent_id');
    }

    /**
     * Relacionamento com as faturas de pagamento
     */
    public function faturas()
    {
        return $this->hasMany(FaturaContaReceber::class, 'conta_receber_id');
    }

    public function isVencida()
    {
        return !$this->status && strtotime($this->data_vencimento) < strtotime(date('Y-m-d'));
    }

    public function isParcial()
    {
        return !$this->status && $this->valor_recebido > 0 && $this->valor_recebido < $this->valor_integral;
    }

    public function valorPendente()
    {
        $valorRecebido = $this->valor_recebido ?? 0;
        $valorIntegral = $this->valor_integral ?? 0;
        
        // Garante que o valor pendente não seja negativo
        $pendente = max(0, $valorIntegral - $valorRecebido);
        
        // Se o status for 1 (recebido) ou 3 (reparcelada), retorna 0
        if ($this->status == self::STATUS_RECEBIDO || $this->status == self::STATUS_REPARCELADA) {
            return 0;
        }
        
        return $pendente;
    }

    public function diasAtraso()
    {
        $d = date('Y-m-d');
        $d2 = $this->data_vencimento;
        $dif = strtotime($d2) - strtotime($d);
        $dias = floor($dif / (60 * 60 * 24));

        if($dias == 0){
            return "conta vence hoje";
        }
        
        if($dias > 0){
            return "$dias dia(s) para o vencimento";
        }else{
            return "conta vencida há " . ($dias*-1) . " dia(s)";
        }
    }

    // Método para gerar próxima parcela recorrente
    public function gerarProximaParcela()
    {
        if (!$this->recorrente || ($this->total_parcelas && $this->parcela_atual >= $this->total_parcelas)) {
            return null;
        }
        
        $novaParcela = $this->replicate(['status', 'valor_recebido', 'data_recebimento']);
        
        // Calcular próxima data de vencimento baseado na periodicidade
        switch ($this->periodicidade) {
            case self::PERIODICIDADE_DIARIA:
                $novaData = Carbon::parse($this->data_vencimento)->addDay();
                break;
            case self::PERIODICIDADE_SEMANAL:
                $novaData = Carbon::parse($this->data_vencimento)->addWeek();
                break;
            case self::PERIODICIDADE_QUINZENAL:
                $novaData = Carbon::parse($this->data_vencimento)->addDays(15);
                break;
            case self::PERIODICIDADE_MENSAL:
                $novaData = Carbon::parse($this->data_vencimento)->addMonth();
                break;
            case self::PERIODICIDADE_BIMESTRAL:
                $novaData = Carbon::parse($this->data_vencimento)->addMonths(2);
                break;
            case self::PERIODICIDADE_TRIMESTRAL:
                $novaData = Carbon::parse($this->data_vencimento)->addMonths(3);
                break;
            case self::PERIODICIDADE_SEMESTRAL:
                $novaData = Carbon::parse($this->data_vencimento)->addMonths(6);
                break;
            case self::PERIODICIDADE_ANUAL:
                $novaData = Carbon::parse($this->data_vencimento)->addYear();
                break;
            default:
                $novaData = Carbon::parse($this->data_vencimento)->addMonth();
        }
        
        $novaParcela->data_vencimento = $novaData;
        $novaParcela->parent_id = $this->id;
        $novaParcela->parcela_atual = ($this->parcela_atual ?? 0) + 1;
        
        return $novaParcela;
    }

    public static function periodicidades()
    {
        return [
            self::PERIODICIDADE_DIARIA => 'Diária',
            self::PERIODICIDADE_SEMANAL => 'Semanal',
            self::PERIODICIDADE_QUINZENAL => 'Quinzenal',
            self::PERIODICIDADE_MENSAL => 'Mensal',
            self::PERIODICIDADE_BIMESTRAL => 'Bimestral',
            self::PERIODICIDADE_TRIMESTRAL => 'Trimestral',
            self::PERIODICIDADE_SEMESTRAL => 'Semestral',
            self::PERIODICIDADE_ANUAL => 'Anual',
        ];
    }

    public static function prioridades()
    {
        return [
            self::PRIORIDADE_BAIXA => 'Baixa',
            self::PRIORIDADE_MEDIA => 'Média',
            self::PRIORIDADE_ALTA => 'Alta',
            self::PRIORIDADE_URGENTE => 'Urgente',
        ];
    }

    public static function tiposPagamento()
    {
        return [
            '01' => 'Dinheiro',
            '02' => 'Cheque',
            '03' => 'Cartão de Crédito',
            '04' => 'Cartão de Débito',
            '05' => 'Crédito Loja',
            '06' => 'Crediário',
            '10' => 'Vale Alimentação',
            '11' => 'Vale Refeição',
            '12' => 'Vale Presente',
            '13' => 'Vale Combustível',
            '14' => 'Duplicata Mercantil',
            '15' => 'Boleto Bancário',
            '16' => 'Depósito Bancário',
            '17' => 'Pagamento Instantâneo (PIX)',
            '90' => 'Sem Pagamento',
            '99' => 'Outros',
        ];
    }

    public static function tiposPagamentoRecebimentos()
    {
        return [
            '01' => 'Dinheiro',
            '02' => 'Cheque',
            '03' => 'Cartão de Crédito',
            '04' => 'Cartão de Débito',
            '10' => 'Vale Alimentação',
            '11' => 'Vale Refeição',
            '15' => 'Boleto Bancário',
            '17' => 'Pagamento Instantâneo (PIX)',
            '99' => 'Outros',
        ];
    }
    

    // Adicionar método para obter descrição do status
    public function getStatusDescricao()
    {
        switch ($this->status) {
            case self::STATUS_PENDENTE:
                return 'Pendente';
            case self::STATUS_RECEBIDO:
                return 'Recebido';
            case self::STATUS_PARCIAL:
                return 'Recebido Parcial';
            case self::STATUS_REPARCELADA:
                return 'Reparcelada';
            default:
                return 'Desconhecido';
        }
    }

    // Adicionar método para obter classe CSS do status
    public function getStatusClass()
    {
        switch ($this->status) {
            case self::STATUS_PENDENTE:
                return 'danger';
            case self::STATUS_RECEBIDO:
                return 'success';
            case self::STATUS_PARCIAL:
                return 'warning';
            case self::STATUS_REPARCELADA:
                return 'info';
            default:
                return 'secondary';
        }
    }
}
