<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use App\Models\MenuAccessLog;
use App\Models\UserSession;

class TrackMenuAccess
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $response = $next($request);

        // Só registrar se o usuário estiver autenticado
        if (Auth::check()) {
            $user = Auth::user();
            $routeName = Route::currentRouteName();
            $url = $request->fullUrl();
            
            // Extrair o menu e submenu com base na rota
            list($menu, $submenu) = $this->extractMenuInfo($routeName);
            
            // Registrar acesso ao menu
            MenuAccessLog::create([
                'usuario_id' => $user->id,
                'empresa_id' => $user->empresa ? $user->empresa->empresa_id : null,
                'rota' => $routeName,
                'menu' => $menu,
                'submenu' => $submenu,
                'url' => $url,
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent()
            ]);
            
            // Atualizar o contador de funcionalidades usadas na sessão atual
            $this->updateFeaturesUsed($user->id);
        }

        return $response;
    }
    
    /**
     * Extrai o menu e submenu com base no nome da rota
     */
    private function extractMenuInfo($routeName)
    {
        if (!$routeName) {
            return [null, null];
        }
        
        $parts = explode('.', $routeName);
        
        if (count($parts) == 1) {
            return [$parts[0], null];
        }
        
        if (count($parts) >= 2) {
            return [$parts[0], $parts[1]];
        }
        
        return [null, null];
    }
    
    /**
     * Atualiza o contador de funcionalidades usadas na sessão atual
     */
    private function updateFeaturesUsed($userId)
    {
        // Buscar a sessão atual (ou criar uma nova se não existir)
        $session = UserSession::where('usuario_id', $userId)
            ->whereNull('logout_time')
            ->orderBy('created_at', 'desc')
            ->first();
            
        if ($session) {
            $session->features_used += 1;
            $session->save();
        }
    }
}
