<?php

namespace App\Http\Controllers;

use App\Models\MesaComanda;
use App\Models\CategoriaProduto;
use App\Models\Produto;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class MesaComandaController extends Controller
{
    public function index(Request $request)
    {
        $empresaId = auth()->user()->empresa_id ?? 1;
        $config = DB::table('config_gerals')->where('empresa_id', $empresaId)->first();
        $quantidadeMesas = $config->quantidade_mesas ?? 10;

        // Se não houver mesas, cria de 1 até quantidadeMesas
        $totalMesas = MesaComanda::where('empresa_id', $empresaId)->count();
        if ($totalMesas < $quantidadeMesas) {
            $numerosExistentes = MesaComanda::where('empresa_id', $empresaId)->pluck('numero')->toArray();
            for ($i = 1; $i <= $quantidadeMesas; $i++) {
                if (!in_array($i, $numerosExistentes)) {
                    MesaComanda::create([
                        'empresa_id' => $empresaId,
                        'numero' => $i,
                        'status' => 'livre',
                    ]);
                }
            }
        }

        $mesas = MesaComanda::where('empresa_id', $empresaId)
            ->orderBy('numero', 'asc')
            ->get();

        // Buscar categorias ativas para cardápio
        $categorias = CategoriaProduto::where('empresa_id', $empresaId)
            ->orderBy('nome', 'asc')
            ->get();

        // Buscar produtos ativos para cardápio e com estoque disponível, paginando 18 por página
        $produtos = Produto::where('empresa_id', $empresaId)
            ->where('status', 1)
            ->with(['categoria', 'estoque'])
            ->get()
            ->filter(function($produto) {
                if ($produto->gerenciar_estoque) {
                    return $produto->estoque && $produto->estoque->quantidade > 0;
                }
                return true;
            });
        $produtos = $produtos->values();
        $perPage = 16; // 16 produtos por página
        $currentPage = request()->get('page', 1);
        $currentItems = $produtos->slice(($currentPage - 1) * $perPage, $perPage);
        $paginatedProdutos = new \Illuminate\Pagination\LengthAwarePaginator(
            $currentItems,
            $produtos->count(),
            $perPage,
            $currentPage,
            ['path' => request()->url(), 'query' => request()->query()]
        );
        $produtos = $paginatedProdutos;

        if ($request->wantsJson() || $request->is('api/*')) {
            return response()->json($mesas);
        }
        return view('mesa_comandas.index', compact('mesas', 'quantidadeMesas', 'categorias', 'produtos'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'empresa_id' => 'required|exists:empresas,id',
            'numero' => 'required|integer|unique:mesa_comandas,numero,NULL,id,empresa_id,' . $request->empresa_id,
            'status' => 'required|in:livre,ocupada',
            'nome_cliente' => 'nullable|string',
            'valor_total' => 'nullable|numeric',
            'abertura' => 'nullable|date',
            'fechamento' => 'nullable|date',
        ]);
        $mesa = MesaComanda::create($data);
        return response()->json($mesa, 201);
    }

    public function show($id)
    {
        $mesa = MesaComanda::findOrFail($id);
        return response()->json($mesa);
    }

    public function update(Request $request, $id)
    {
        $mesa = MesaComanda::findOrFail($id);
        $data = $request->validate([
            'status' => 'in:livre,ocupada',
            'nome_cliente' => 'nullable|string',
            'valor_total' => 'nullable|numeric',
            'abertura' => 'nullable|date',
            'fechamento' => 'nullable|date',
        ]);
        $mesa->update($data);
        return response()->json($mesa);
    }

    public function destroy($id)
    {
        $mesa = MesaComanda::findOrFail($id);
        $mesa->delete();
        return response()->json(['message' => 'Mesa/Comanda removida com sucesso.']);
    }

    public function itens($id)
    {
        $mesa = \App\Models\MesaComanda::findOrFail($id);
        $itens = $mesa->itens()->with('produto')->get();
        return response()->json($itens);
    }

    public function deletarItens($id)
    {
        $mesa = \App\Models\MesaComanda::findOrFail($id);
        $mesa->itens()->delete();
        // Atualiza status e total da mesa
        $mesa->status = 'livre';
        $mesa->total = 0;
        $mesa->save();
        return response()->json(['message' => 'Itens removidos com sucesso.']);
    }

    public function imprimirConferencia($id)
    {
        $mesa = \App\Models\MesaComanda::with(['itens.produto', 'itens'])->findOrFail($id);
        $usuario = auth()->user();
        return view('mesa_comandas.imprimir_conferencia', compact('mesa', 'usuario'));
    }
} 