<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PlanoEmpresa;
use App\Models\Empresa;
use App\Models\Plano;
use App\Models\FinanceiroPlano;
use App\Models\Revenda;
use App\Models\RecebimentoPlano;
use Illuminate\Support\Facades\Auth;

class GerenciarPlanoController extends Controller
{
    public function index(Request $request)
    {
        $empresa = $request->empresa;
        $revenda = $request->revenda;
        $planos = Plano::orderBy('nome', 'asc')->get();
        $revendas = Revenda::where('status', 1)->orderBy('nome')->get();
        
        $data = PlanoEmpresa::orderBy('id', 'desc')
        ->when(!empty($empresa), function ($query) use ($empresa) {
            return $query->where('empresa_id', $empresa);
        })
        ->when(!empty($revenda), function ($query) use ($revenda) {
            return $query->where('revenda_id', $revenda);
        })
        ->paginate(env("PAGINACAO"));

        if($empresa){
            $empresa = Empresa::findOrFail($empresa);
        }
        
        if($revenda){
            $revenda = Revenda::findOrFail($revenda);
        }
        
        return view('gerencia_planos.index', compact('data', 'planos', 'empresa', 'revendas', 'revenda'));
    }

    public function store(Request $request)
    {
        try {
            $plano = Plano::findOrfail($request->plano_id);
            $intervalo = $plano->intervalo_dias;
            $exp = date('Y-m-d', strtotime(date('Y-m-d') . "+ $intervalo days"));

            $planoEmpresa = PlanoEmpresa::create([
                'empresa_id' => $request->empresa_atribuir,
                'revenda_id' => $request->revenda_id,
                'plano_id' => $request->plano_id,
                'data_expiracao' => $exp,
                'valor' => __convert_value_bd($request->valor),
                'valor_repasse' => __convert_value_bd($request->valor_repasse),
                'forma_pagamento' => $request->forma_pagamento
            ]);

            $valorPlano = __convert_value_bd($request->valor);
            $valorRepasse = __convert_value_bd($request->valor_repasse);
            
            // Criar registro no financeiro
            $financeiroPlano = FinanceiroPlano::create([
                'empresa_id' => $request->empresa_atribuir,
                'revenda_id' => $request->revenda_id,
                'plano_id' => $request->plano_id,
                'valor' => $valorPlano,
                'valor_repasse' => $valorRepasse,
                'tipo_pagamento' => $request->forma_pagamento,
                'status_pagamento' => $request->status_pagamento,
                'plano_empresa_id' => $planoEmpresa->id
            ]);
            
            // Se o status de pagamento for recebido, registrar também na tabela de recebimentos
            if ($request->status_pagamento === 'recebido') {
                RecebimentoPlano::create([
                    'financeiro_plano_id' => $financeiroPlano->id,
                    'data_pagamento' => $request->has('data_pagamento') ? $request->data_pagamento : date('Y-m-d'),
                    'valor_pago' => $valorPlano,
                    'valor_repasse' => $valorRepasse,
                    'forma_pagamento' => $request->forma_pagamento,
                    'observacao' => 'Registrado automaticamente durante atribuição do plano',
                    'user_id' => Auth::id()
                ]);
            }
            
            session()->flash("flash_success", "Plano atribuído!");
        } catch (\Exception $e) {
            session()->flash("flash_error", "Algo deu errado: " . $e->getMessage());
        }
        return redirect()->back();
    }

    public function destroy($id)
    {
        $item = PlanoEmpresa::findOrFail($id);
        try {
            $financeiro = FinanceiroPlano::where('plano_empresa_id', $item->id)->first();
            if($financeiro){
                $financeiro->delete();
            }
            $item->delete();
            session()->flash("flash_success", "Apagado com sucesso!");
        } catch (\Exception $e) {
            session()->flash("flash_error", 'Algo deu errado.', $e->getMessage());
        }
        return redirect()->back();
    }

    public function edit($id){
        $item = PlanoEmpresa::findOrFail($id);
        return view('gerencia_planos.edit', compact('item'));
    }

    public function update(Request $request, $id){
        $item = PlanoEmpresa::findOrFail($id);
        try{
            $item->data_expiracao = $request->data_expiracao;
            $item->save();
            session()->flash("flash_success", "Data alterada!");
        } catch (\Exception $e) {
            session()->flash("flash_error", "Algo deu errado: " . $e->getMessage());
        }
        return redirect()->route('gerenciar-planos.index');
    }
    
    public function syncFinanceiroPlanos(){
        try {
            $planoEmpresas = PlanoEmpresa::whereNotNull('revenda_id')->orWhereNotNull('valor_repasse')->get();
            $count = 0;
            
            foreach($planoEmpresas as $planoEmpresa){
                $financeiro = FinanceiroPlano::where('plano_empresa_id', $planoEmpresa->id)->first();
                
                if($financeiro){
                    $financeiro->revenda_id = $planoEmpresa->revenda_id;
                    $financeiro->valor_repasse = $planoEmpresa->valor_repasse;
                    $financeiro->save();
                    $count++;
                }
            }
            
            session()->flash("flash_success", "Sincronização concluída! $count registros atualizados.");
        } catch (\Exception $e) {
            session()->flash("flash_error", "Erro na sincronização: " . $e->getMessage());
        }
        
        return redirect()->route('financeiro-plano.index');
    }
}
