<?php

namespace App\Http\Controllers;

use App\Models\Caixa;
use App\Models\Cliente;
use App\Models\ContaReceber;
use App\Models\Frete;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Utils\ContaEmpresaUtil;
use App\Models\ItemContaEmpresa;
use App\Utils\UploadUtil;
use App\Models\AlocacaoCentroCusto;
use App\Models\AnexoConta;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Permission;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Models\ContaReceberHistorico;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;  // Adicionado import do Log
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Schema;
// use App\Models\LogSistema;
// use App\Events\ContaReceberEvent;

class ContaReceberController extends Controller
{
    protected $util;
    protected $uploadUtil;

    public function __construct(ContaEmpresaUtil $util, UploadUtil $uploadUtil){
        $this->util = $util;
        $this->uploadUtil = $uploadUtil;

        $this->middleware('permission:conta_receber_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:conta_receber_edit', ['only' => ['edit', 'update', 'pay', 'payPut', 'reparcelar', 'receberSelecionados']]);
        $this->middleware('permission:conta_receber_view', ['only' => ['show', 'index']]);
        $this->middleware('permission:conta_receber_delete', ['only' => ['destroy']]);
    }

    public function index(Request $request)
    {

        $locais = __getLocaisAtivoUsuario();
        $locais = $locais->pluck(['id']);

        $cliente_id = $request->cliente_id;
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        $status = $request->status;
        $ordem = $request->ordem;
        $local_id = $request->get('local_id');
        $categoria_id = $request->categoria_id;
        $vencidas = $request->vencidas;
        $parcialmente_recebidas = $request->parcialmente_recebidas;

        $data = ContaReceber::where('empresa_id', request()->empresa_id)
        ->when(!empty($cliente_id), function ($query) use ($cliente_id) {
            return $query->where('cliente_id', $cliente_id);
        })
        ->when(!empty($start_date), function ($query) use ($start_date) {
            return $query->whereDate('data_vencimento', '>=', $start_date);
        })
        ->when(!empty($end_date), function ($query) use ($end_date) {
            return $query->whereDate('data_vencimento', '<=', $end_date);
        })
        ->when($local_id, function ($query) use ($local_id) {
            return $query->where('local_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->whereIn('local_id', $locais);
        })
        ->when($status != '', function ($query) use ($status) {
            return $query->where('status', $status);
        })
        ->when($categoria_id, function ($query) use ($categoria_id) {
            return $query->where('categoria_id', $categoria_id);
        })
        ->when($vencidas, function ($query) {
            return $query->where('status', 0)
                         ->whereDate('data_vencimento', '<', now());
        })
        ->when($parcialmente_recebidas, function ($query) {
            return $query->where('status', 0)
                         ->whereNotNull('valor_recebido')
                         ->whereRaw('valor_recebido > 0')
                         ->whereRaw('valor_recebido < valor_integral');
        })
        ->when($ordem != '', function ($query) use ($ordem) {
            return $query->orderBy('data_vencimento', 'asc');
        })
        ->when($ordem == '', function ($query) use ($ordem) {
            return $query->orderBy('created_at', 'asc');
        })
        ->with(['cliente', 'categoria', 'localizacao'])
        ->paginate(env("PAGINACAO"));

        $cliente = null;
        if($cliente_id){
            $cliente = Cliente::findOrFail($cliente_id);
        }
        
        $categorias = \App\Models\CategoriaConta::where('empresa_id', request()->empresa_id)
            ->where(function($query) {
                $query->where('tipo', 'receber')
                      ->orWhere('tipo', 'ambos');
            })
            ->where('ativo', true)
            ->orderBy('nome')
            ->get();
            
        return view('conta-receber.index', compact('data', 'cliente', 'categorias'));
    }

    public function create(Request $request)
    {
        $clientes = Cliente::where('empresa_id', request()->empresa_id)->get();
        
        $categorias = \App\Models\CategoriaConta::where('empresa_id', request()->empresa_id)
            ->where(function($query) {
                $query->where('tipo', 'receber')
                      ->orWhere('tipo', 'ambos');
            })
            ->where('ativo', true)
            ->orderBy('nome')
            ->get();
        
        $centrosCusto = \App\Models\CentroCusto::where('empresa_id', request()->empresa_id)
            ->where('ativo', true)
            ->orderBy('nome')
            ->get();
        
        $periodicidades = ContaReceber::periodicidades();
        $prioridades = ContaReceber::prioridades();

        $item = null;
        $diferenca = null;
        if($request->id){
            $item = ContaReceber::findOrFail($request->id);
            $item->valor_integral = $request->diferenca;
        }

        if($request->diferenca){
            $diferenca = $request->diferenca;
        }

        return view('conta-receber.create', compact('clientes', 'item', 'diferenca', 'categorias', 'centrosCusto', 'periodicidades', 'prioridades'));
    }

    public function store(Request $request)
    {
        $this->__validate($request);
        try {
            $file_name = '';
            if ($request->hasFile('file')) {
                $file_name = $this->uploadUtil->uploadFile($request->file, '/financeiro');
            }

            $request->merge([
                'valor_integral' => __convert_value_bd($request->valor_integral),
                'valor_recebido' => $request->valor_recebido ? __convert_value_bd($request->valor_recebido) : 0,
                'arquivo' => $file_name,
                'recorrente' => $request->has('recorrente'),
            ]);
            
            $conta = ContaReceber::create($request->all());
            
            // Registrar histórico de criação da conta
            \App\Models\ContaReceberHistorico::create([
                'conta_receber_id' => $conta->id,
                'valor' => $conta->valor_integral,
                'usuario_id' => Auth::id(),
                'descricao' => 'Conta criada',
                'tipo_pagamento' => $conta->tipo_pagamento,
                'tipo_movimento' => 'criação'
            ]);
            
            // Salvar alocações de centro de custo se fornecidas
            if ($request->has('centro_custo_valores') && is_array($request->centro_custo_valores)) {
                foreach ($request->centro_custo_valores as $index => $valor) {
                    if (!empty($valor)) {
                        $valorAlocacao = __convert_value_bd($valor);
                        $porcentagem = ($valorAlocacao / $request->valor_integral) * 100;
                        
                        AlocacaoCentroCusto::create([
                            'conta_receber_id' => $conta->id,
                            'centro_custo_id' => $request->centro_custo_ids[$index],
                            'valor' => $valorAlocacao,
                            'porcentagem' => $porcentagem,
                            'observacao' => $request->centro_custo_observacoes[$index] ?? null
                        ]);
                    }
                }
            }
            
            // Processar múltiplos anexos
            if ($request->hasFile('anexos')) {
                foreach ($request->file('anexos') as $anexo) {
                    $nomeArquivo = uniqid() . '.' . $anexo->getClientOriginalExtension();
                    $caminhoArquivo = $anexo->storeAs('financeiro/anexos', $nomeArquivo, 'public');
                    
                    AnexoConta::create([
                        'conta_receber_id' => $conta->id,
                        'nome_arquivo' => $anexo->getClientOriginalName(),
                        'caminho_arquivo' => $caminhoArquivo,
                        'tipo_arquivo' => $anexo->getClientMimeType(),
                        'tamanho_arquivo' => $anexo->getSize(),
                        'usuario_id' => Auth::id()
                    ]);
                }
            }
            
            $descricaoLog = "Vencimento: " . __data_pt($request->data_vencimento, 0) . " R$ " . __moeda($request->valor_integral);
            __createLog($request->empresa_id, 'Conta a Receber', 'cadastrar', $descricaoLog);
            
            if(isset($request->frete_id)){
                $frete = Frete::findOrFail($request->frete_id);
                $frete->conta_receber_id = $conta->id;
                $frete->save();
            }

            // Processar recorrência
            if ($request->recorrente && $request->periodicidade && $request->total_parcelas > 0) {
                $conta->periodicidade = $request->periodicidade;
                $conta->total_parcelas = $request->total_parcelas;
                $conta->parcela_atual = 1;
                $conta->save();
                
                // Gerar parcelas futuras
                for ($i = 2; $i <= $request->total_parcelas; $i++) {
                    $parcela = clone $conta;
                    $parcela->status = 0;
                    $parcela->valor_recebido = 0;
                    $parcela->data_recebimento = null;
                    $parcela->parent_id = $conta->id;
                    $parcela->parcela_atual = $i;
                    
                    // Garantir que a descrição seja mantida
                    $parcela->descricao = $conta->descricao;
                    
                    // Calcular próxima data de vencimento baseado na periodicidade
                    $dataBase = $conta->data_vencimento;
                    switch ($request->periodicidade) {
                        case ContaReceber::PERIODICIDADE_DIARIA:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' days'));
                            break;
                        case ContaReceber::PERIODICIDADE_SEMANAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' weeks'));
                            break;
                        case ContaReceber::PERIODICIDADE_QUINZENAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 15) . ' days'));
                            break;
                        case ContaReceber::PERIODICIDADE_MENSAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' months'));
                            break;
                        case ContaReceber::PERIODICIDADE_BIMESTRAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 2) . ' months'));
                            break;
                        case ContaReceber::PERIODICIDADE_TRIMESTRAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 3) . ' months'));
                            break;
                        case ContaReceber::PERIODICIDADE_SEMESTRAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 6) . ' months'));
                            break;
                        case ContaReceber::PERIODICIDADE_ANUAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' years'));
                            break;
                        default:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' months'));
                    }
                    
                    $parcela->data_vencimento = $dataVencimento;
                    $parcela->save();
                    
                    $descricaoLog = "Vencimento (parcela $i): " . __data_pt($dataVencimento, 0) . " R$ " . __moeda($conta->valor_integral);
                    __createLog($request->empresa_id, 'Conta a Receber', 'cadastrar', $descricaoLog);
                }
            } 
            // Recorrência antiga via dt_recorrencia
            elseif ($request->dt_recorrencia) {
                for ($i = 0; $i < sizeof($request->dt_recorrencia); $i++) {
                    $data = $request->dt_recorrencia[$i];
                    $valor = __convert_value_bd($request->valor_recorrencia[$i]);
                    $data = [
                        'venda_id' => null,
                        'data_vencimento' => $data,
                        'data_pagamento' => $data,
                        'valor_integral' => $valor,
                        'valor_recebido' => $request->status ? $valor : 0,
                        'referencia' => $request->referencia,
                        'categoria_id' => $request->categoria_id,
                        'status' => $request->status,
                        'empresa_id' => $request->empresa_id,
                        'cliente_id' => $request->cliente_id,
                        'tipo_pagamento' => $request->tipo_pagamento,
                        'local_id' => $request->local_id,
                        'descricao' => $request->descricao // Garantir que a descrição seja mantida
                    ];
                    $novaConta = ContaReceber::create($data);
                    
                    $descricaoLog = "Vencimento: " . __data_pt($request->dt_recorrencia[$i], 0) . " R$ " . __moeda($valor);
                    __createLog($request->empresa_id, 'Conta a Receber', 'cadastrar', $descricaoLog);
                }
            }
            
            session()->flash("flash_success", "Conta a receber cadastrada!");
        } catch (\Exception $e) {
            __createLog($request->empresa_id, 'Conta a Receber', 'erro', $e->getMessage());
            session()->flash("flash_error", "Algo deu errado: " . $e->getMessage());
        }

        if(isset($request->redirect)){
            return redirect($request->redirect);
        }
        return redirect()->route('conta-receber.index');
    }

    public function edit($id)
    {
        $item = ContaReceber::findOrFail($id);
        __validaObjetoEmpresa($item);

        // Impedir edição de contas reparceladas
        if ($item->status == ContaReceber::STATUS_REPARCELADA) {
            session()->flash('flash_error', 'Contas reparceladas não podem ser editadas.');
            return redirect()->back();
        }

        $clientes = Cliente::where('empresa_id', request()->empresa_id)->get();

        $categorias = \App\Models\CategoriaConta::where('empresa_id', request()->empresa_id)
            ->where(function($query) {
                $query->where('tipo', 'receber')
                      ->orWhere('tipo', 'ambos');
            })
            ->where('ativo', true)
            ->orderBy('nome')
            ->get();
            
        $centrosCusto = \App\Models\CentroCusto::where('empresa_id', request()->empresa_id)
            ->where('ativo', true)
            ->orderBy('nome')
            ->get();
            
        $alocacoes = AlocacaoCentroCusto::where('conta_receber_id', $id)
            ->with('centroCusto')
            ->get();
            
        $anexos = AnexoConta::where('conta_receber_id', $id)
            ->get();
            
        $periodicidades = ContaReceber::periodicidades();
        $prioridades = ContaReceber::prioridades();
        
        $parcelasGeradas = [];
        if ($item->recorrente) {
            $parcelasGeradas = ContaReceber::where('parent_id', $id)
                ->orderBy('parcela_atual')
                ->get();
        }

        return view('conta-receber.edit', compact(
            'item', 
            'clientes', 
            'categorias', 
            'centrosCusto', 
            'alocacoes', 
            'anexos', 
            'periodicidades', 
            'prioridades', 
            'parcelasGeradas'
        ));
    }

    public function update(Request $request, $id)
    {
        $item = ContaReceber::findOrFail($id);
        __validaObjetoEmpresa($item);

        try {
            $file_name = $item->arquivo;
            if ($request->hasFile('file')) {
                $this->uploadUtil->unlinkImage($item, '/financeiro');
                $file_name = $this->uploadUtil->uploadFile($request->file, '/financeiro');
            }
            
            $request->merge([
                'valor_integral' => __convert_value_bd($request->valor_integral),
                'valor_recebido' => __convert_value_bd($request->valor_recebido) ? __convert_value_bd($request->valor_recebido) : 0,
                'arquivo' => $file_name,
                'recorrente' => $request->has('recorrente')
            ]);
            
            $item->fill($request->all())->save();
            
            // Atualizar alocações de centro de custo
            if ($request->has('centro_custo_valores') && is_array($request->centro_custo_valores)) {
                // Remover alocações existentes
                AlocacaoCentroCusto::where('conta_receber_id', $id)->delete();
                
                // Adicionar novas alocações
                foreach ($request->centro_custo_valores as $index => $valor) {
                    if (!empty($valor)) {
                        $valorAlocacao = __convert_value_bd($valor);
                        $porcentagem = $valorAlocacao / $item->valor_integral * 100;
                        
                        AlocacaoCentroCusto::create([
                            'conta_receber_id' => $item->id,
                            'centro_custo_id' => $request->centro_custo_ids[$index],
                            'valor' => $valorAlocacao,
                            'porcentagem' => $porcentagem,
                            'observacao' => $request->centro_custo_observacoes[$index] ?? null
                        ]);
                    }
                }
            }
            
            // Processar múltiplos anexos
            if ($request->hasFile('anexos')) {
                foreach ($request->file('anexos') as $anexo) {
                    $nomeArquivo = uniqid() . '.' . $anexo->getClientOriginalExtension();
                    $caminhoArquivo = $anexo->storeAs('financeiro/anexos', $nomeArquivo, 'public');
                    
                    AnexoConta::create([
                        'conta_receber_id' => $item->id,
                        'nome_arquivo' => $anexo->getClientOriginalName(),
                        'caminho_arquivo' => $caminhoArquivo,
                        'tipo_arquivo' => $anexo->getClientMimeType(),
                        'tamanho_arquivo' => $anexo->getSize(),
                        'usuario_id' => Auth::id()
                    ]);
                }
            }
            
            // Gerar novas parcelas se recorrência foi ativada e não havia parcelas anteriores
            if ($request->recorrente && 
                $request->periodicidade && 
                $request->total_parcelas > 0 && 
                !$item->contasRecorrentes()->exists()) {
                
                $item->periodicidade = $request->periodicidade;
                $item->total_parcelas = $request->total_parcelas;
                $item->parcela_atual = 1;
                $item->save();
                
                // Gerar parcelas futuras
                $this->gerarParcelasFuturas($item, $request->periodicidade, $request->total_parcelas);
            }

            $descricaoLog = "Vencimento: " . __data_pt($item->data_vencimento) . " R$ " . __moeda($item->valor_integral);
            __createLog($request->empresa_id, 'Conta a Receber', 'editar', $descricaoLog);
            session()->flash("flash_success", "Conta a receber atualizada!");
        } catch (\Exception $e) {
            __createLog($request->empresa_id, 'Conta a Receber', 'erro', $e->getMessage());
            session()->flash("flash_error", "Algo deu errado: " . $e->getMessage());
        }
        
        return redirect()->route('conta-receber.index');
    }

    public function downloadFile($id){
        $item = ContaReceber::findOrFail($id);
        if (file_exists(public_path('uploads/financeiro/') . $item->arquivo)) {
            return response()->download(public_path('uploads/financeiro/') . $item->arquivo);
        } else {
            session()->flash("flash_error", "Arquivo não encontrado");
            return redirect()->back();
        }
    }

    private function __validate(Request $request)
    {
        $rules = [
            'cliente_id' => 'required',
            'valor_integral' => 'required',
            'data_vencimento' => 'required',
            'status' => 'required',
            'tipo_pagamento' => 'required'
        ];
        $messages = [
            'cliente_id.required' => 'Campo obrigatório',
            'valor_integral.required' => 'Campo obrigatório',
            'data_vencimento.required' => 'Campo obrigatório',
            'status.required' => 'Campo obrigatório',
            'tipo_pagamento.required' => 'Campo obrigatório'
        ];
        $this->validate($request, $rules, $messages);
    }

    public function destroy($id)
    {
        $item = ContaReceber::findOrFail($id);
        __validaObjetoEmpresa($item);
        
        try {
            $descricaoLog = "Vencimento: " . __data_pt($item->data_vencimento, 0) . " R$ " . __moeda($item->valor_integral);
            
            // Registrar histórico antes de excluir
            $historico = new ContaReceberHistorico();
            $historico->conta_receber_id = $item->id;
            $historico->valor = $item->valor_integral;
            $historico->tipo_movimento = 'cancelamento';
            $historico->data_movimento = date('Y-m-d H:i:s');
            $historico->usuario_id = auth()->id();
            $historico->observacao = "Conta removida manualmente";
            $historico->save();
            
            $item->delete();
            __createLog(request()->empresa_id, 'Conta a Receber', 'excluir', $descricaoLog);
            session()->flash("flash_success", "Conta removida!");
        } catch (\Exception $e) {
            __createLog(request()->empresa_id, 'Conta a Receber', 'erro', $e->getMessage());
            session()->flash("flash_error", "Algo deu errado: " . $e->getMessage());
        }
        return redirect()->route('conta-receber.index');
    }

    public function destroySelecet(Request $request)
    {
        $removidos = 0;
        $recebidas = 0;
        // dd($request->all());
        for($i=0; $i<sizeof($request->item_delete); $i++){
            $item = ContaReceber::findOrFail($request->item_delete[$i]);
            if($item->boleto){
                session()->flash("flash_error", 'Conta a receber selecionada com boleto vinculado!');
                return redirect()->back();
            }

            if(!$item->status){
                try {
                    $descricaoLog = "Vencimento: " . __data_pt($item->data_vencimento, 0) . " R$ " . __moeda($item->valor_integral);
                    $item->delete();
                    $removidos++;
                    __createLog(request()->empresa_id, 'Conta a Receber', 'excluir', $descricaoLog);
                } catch (\Exception $e) {
                    __createLog(request()->empresa_id, 'Conta a Receber', 'erro', $e->getMessage());
                    session()->flash("flash_error", 'Algo deu errado: '. $e->getMessage());
                    return redirect()->back();
                }
            }else{
                $recebidas++;
            }
        }

        session()->flash("flash_success", "Total de contas removidas: $removidos");
        if($recebidas > 0){
            session()->flash("flash_warning", "Total de contas não removidas: $recebidas");
        }
        return redirect()->back();
    }

    public function receberSelecionados(Request $request)
    {
        // Verificar se o caixa está aberto
        if (!__isCaixaAberto()) {
            session()->flash("flash_warning", "Abrir caixa antes de continuar!");
            return redirect()->route('caixa.create');
        }

        // Se for uma requisição GET sem item_recebe_paga, redirecionar para a lista de contas com mensagem
        if ($request->isMethod('get') && !$request->filled('item_recebe_paga')) {
            session()->flash("flash_warning", "Selecione as contas a receber na listagem principal antes de continuar!");
            return redirect()->route('conta-receber.index');
        }

        // Verificar se a tabela fatura_contas_recebers existe
        try {
            if (!Schema::hasTable('fatura_contas_recebers')) {
                Log::error("Tabela 'fatura_contas_recebers' não existe no banco de dados!");
                session()->flash("flash_error", "Erro de configuração do banco de dados. Entre em contato com o suporte.");
                return redirect()->back();
            }
            Log::info("Tabela 'fatura_contas_recebers' verificada e existe.");
        } catch (\Exception $e) {
            Log::error("Erro ao verificar tabela: " . $e->getMessage());
        }

        // Se for apenas para exibir o formulário de pagamento
        if (!$request->isMethod('post') || !$request->filled('valor_pago')) {
            // Buscar as contas selecionadas
            $contas = ContaReceber::whereIn('id', $request->item_recebe_paga)
                ->where('empresa_id', request()->empresa_id)
                ->where('status', '<>', ContaReceber::STATUS_RECEBIDO)
                ->where('status', '<>', ContaReceber::STATUS_REPARCELADA)
                ->orderBy('data_vencimento', 'asc')
                ->with(['cliente', 'categoria']) // Garantir carregamento do relacionamento cliente
                ->get();

            if ($contas->isEmpty()) {
                session()->flash("flash_warning", "Nenhuma conta válida selecionada para recebimento!");
                return redirect()->back();
            }

            // Calcular valor total pendente
            $valorTotalPendente = $contas->sum(function($conta) {
                return $conta->valorPendente();
            });

            // Obter tipos de pagamento disponíveis
            $tiposPagamento = ContaReceber::tiposPagamento();

            return view('conta-receber.receber-multiplos', compact('contas', 'valorTotalPendente', 'tiposPagamento'));
        }

        // Registrar todos os dados da requisição para debug
        \Illuminate\Support\Facades\Log::info("Dados completos da requisição de recebimento:", [
            'request' => $request->all(),
            'headers' => $request->header()
        ]);

        try {
            // Validação dos dados do formulário
            $validatedData = $request->validate([
                'item_recebe_paga' => 'required|array',
                'valor_pago' => 'required',
                // 'data_recebimento' => 'required|date', // Removido para sempre usar data atual
                'tipos_pagamento' => 'required|array',
                'valores_pagamento' => 'required|array',
                'observacoes_pagamento' => 'nullable|array',
                'data_nova_vencimento' => 'required_if:pagamento_parcial,1|nullable|date',
            ]);

            \Illuminate\Support\Facades\Log::info("Validação dos dados bem-sucedida");

            // Sempre usar a data atual para recebimento
            $dataRecebimento = date('Y-m-d');
            \Illuminate\Support\Facades\Log::info("Usando data atual para recebimento: " . $dataRecebimento);

            $usuario = Auth::user()->id;
            $caixa = Caixa::where('usuario_id', $usuario)->where('status', 1)->first();
            
            if (!$caixa) {
                session()->flash("flash_warning", "É necessário abrir o caixa antes de realizar recebimentos!");
                return redirect()->back();
            }

            \Illuminate\Support\Facades\Log::info("Caixa verificado, ID: " . $caixa->id);

            // Inicializar variáveis
            $valorPago = __convert_value_bd($request->valor_pago);
            \Illuminate\Support\Facades\Log::info("Valor pago convertido: " . $valorPago);
            
            DB::beginTransaction();
            
            // Buscar contas selecionadas ordenadas por data de vencimento (FIFO)
            $contas = ContaReceber::whereIn('id', $request->item_recebe_paga)
                ->where('empresa_id', request()->empresa_id)
                ->where('status', '<>', ContaReceber::STATUS_RECEBIDO)
                ->where('status', '<>', ContaReceber::STATUS_REPARCELADA)
                ->orderBy('data_vencimento', 'asc')
                ->with(['cliente', 'categoria']) // Garantir carregamento do relacionamento cliente
                ->get();
                
            if ($contas->isEmpty()) {
                DB::rollBack();
                session()->flash("flash_warning", "Nenhuma conta válida selecionada para recebimento!");
                return redirect()->back();
            }
            
            \Illuminate\Support\Facades\Log::info("Contas encontradas: " . $contas->count());
            
            // Verificar valor total pendente
            $valorTotalPendente = $contas->sum(function($conta) {
                return $conta->valorPendente();
            });
            
            \Illuminate\Support\Facades\Log::info("Valor total pendente: " . $valorTotalPendente);
            
            // Array para armazenar os pagamentos processados
            $pagamentosProcessados = [];
            
            // Processar pagamentos múltiplos
            if (is_array($request->tipos_pagamento)) {
                foreach ($request->tipos_pagamento as $index => $tipoPagamento) {
                    // Converter o valor para float se necessário
                    $valorPagamentoAtual = floatval($request->valores_pagamento[$index]);
                    $observacaoPagamento = isset($request->observacoes_pagamento[$index]) ? $request->observacoes_pagamento[$index] : '';
                    
                    // Adicionar ao array de pagamentos processados
                    $pagamentosProcessados[] = [
                        'tipo' => $tipoPagamento,
                        'valor' => $valorPagamentoAtual,
                        'observacao' => $observacaoPagamento
                    ];
                    
                    \Illuminate\Support\Facades\Log::info("Adicionado pagamento: tipo={$tipoPagamento}, valor={$valorPagamentoAtual}");
                }
            } else {
                \Illuminate\Support\Facades\Log::error("tipos_pagamento não é um array: " . gettype($request->tipos_pagamento));
                DB::rollBack();
                session()->flash("flash_error", "Erro ao processar os tipos de pagamento. Entre em contato com o suporte.");
                return redirect()->back();
            }
            
            if (empty($pagamentosProcessados)) {
                \Illuminate\Support\Facades\Log::error("Nenhum pagamento processado");
                DB::rollBack();
                session()->flash("flash_error", "Nenhum pagamento foi informado. Adicione pelo menos uma forma de pagamento.");
                return redirect()->back();
            }
            
            // Verificar o valor total pago
            $valorPago = array_sum(array_column($pagamentosProcessados, 'valor'));
            $valorRestante = $valorPago;
            
            \Illuminate\Support\Facades\Log::info("Valor total dos pagamentos: " . $valorPago);
            
            // Verificar se o pagamento é parcial
            $isPagamentoParcial = ($valorPago < $valorTotalPendente);
            
            // Se for pagamento parcial, verificar se a nova data de vencimento foi informada
            if ($isPagamentoParcial && !$request->filled('data_nova_vencimento')) {
                DB::rollBack();
                session()->flash("flash_warning", "Para pagamentos parciais, é necessário informar uma nova data de vencimento!");
                return redirect()->back();
            }
            
            $contasProcessadas = 0;
            $contasParciaisProcessadas = 0;
            
            // Processar pagamento das contas por ordem de vencimento (FIFO)
            foreach ($contas as $conta) {
                $valorPendenteConta = $conta->valorPendente();
                
                // Se não houver valor pendente, pular para próxima conta
                if ($valorPendenteConta <= 0) {
                    \Illuminate\Support\Facades\Log::info("Conta ID {$conta->id} sem valor pendente, pulando");
                    continue;
                }
                
                // Se não tiver mais valor para pagar, sair do loop
                if ($valorRestante <= 0) {
                    \Illuminate\Support\Facades\Log::info("Sem valor restante para pagamento, saindo do loop");
                    break;
                }
                
                // Determinar quanto será pago dessa conta
                $valorASerPago = min($valorRestante, $valorPendenteConta);
                $isPagamentoTotalConta = ($valorASerPago >= $valorPendenteConta);
                
                \Illuminate\Support\Facades\Log::info("Processando conta ID {$conta->id}: valor_pendente={$valorPendenteConta}, valor_a_ser_pago={$valorASerPago}, pagamento_total=" . ($isPagamentoTotalConta ? 'sim' : 'não'));
                
                // Registrar o pagamento
                $valorAnteriorRecebido = $conta->valor_recebido ?? 0;
                $conta->valor_recebido = $valorAnteriorRecebido + $valorASerPago;
                $conta->data_recebimento = $dataRecebimento;
                
                // Usar o tipo de pagamento do primeiro pagamento como referência na conta principal
                $conta->tipo_pagamento = $pagamentosProcessados[0]['tipo'];
                $conta->caixa_id = $caixa->id;
                
                if ($isPagamentoTotalConta) {
                    // Marca como totalmente recebida
                    $conta->status = ContaReceber::STATUS_RECEBIDO;
                    $contasProcessadas++;
                } else {
                    // Pagamento parcial
                    // Salvar data original de vencimento na primeira vez
                    if (!$conta->data_vencimento_original) {
                        $conta->data_vencimento_original = $conta->data_vencimento;
                    }
                    
                    // Atualiza para status parcial
                    $conta->status = ContaReceber::STATUS_PARCIAL;
                    
                    // Atualizar nova data de vencimento
                    if ($request->filled('data_nova_vencimento')) {
                        $conta->data_vencimento = $request->data_nova_vencimento;
                        $conta->data_nova_vencimento = $request->data_nova_vencimento;
                    }
                    
                    $contasParciaisProcessadas++;
                }
                
                \Illuminate\Support\Facades\Log::info("Salvando conta ID {$conta->id}: status={$conta->status}, valor_recebido={$conta->valor_recebido}");
                
                try {
                    $result = $conta->save();
                    \Illuminate\Support\Facades\Log::info("Conta ID {$conta->id} salva com sucesso: " . ($result ? 'sim' : 'não'));
                } catch (\Exception $e) {
                    \Illuminate\Support\Facades\Log::error("Erro ao salvar conta ID {$conta->id}: " . $e->getMessage());
                    throw $e;
                }
                
                // Salvar os registros de cada pagamento na fatura
                // A proporção de cada pagamento deve ser proporcional ao valor total
                $valorRestanteProporcionado = $valorASerPago;
                
                foreach ($pagamentosProcessados as $pagamento) {
                    // Calcular a proporção deste pagamento sobre o valor total
                    $proporcao = $pagamento['valor'] / $valorPago;
                    
                    // Calcular o valor proporcional deste pagamento para esta conta
                    $valorProporcionalPagamento = round($valorASerPago * $proporcao, 2);
                    
                    // Ajustar para garantir que o total seja exato
                    if ($pagamento === end($pagamentosProcessados) && $valorRestanteProporcionado < $valorProporcionalPagamento) {
                        $valorProporcionalPagamento = $valorRestanteProporcionado;
                    }
                    
                    // Não processar se o valor for zero ou negativo
                    if ($valorProporcionalPagamento <= 0) {
                        \Illuminate\Support\Facades\Log::info("Valor proporcional zero ou negativo, pulando");
                        continue;
                    }
                    
                    try {
                        // Criar a fatura
                        $fatura = new \App\Models\FaturaContaReceber();
                        $fatura->conta_receber_id = $conta->id;
                        $fatura->empresa_id = request()->empresa_id;
                        $fatura->tipo_pagamento = $pagamento['tipo'];
                        $fatura->valor = $valorProporcionalPagamento;
                        $fatura->data_vencimento = $dataRecebimento;
                        $fatura->obs = $pagamento['observacao'] ?? '';
                        
                        \Illuminate\Support\Facades\Log::info("Tentando criar fatura: tabela={$fatura->getTable()}, dados=", [
                            'conta_receber_id' => $fatura->conta_receber_id,
                            'empresa_id' => $fatura->empresa_id,
                            'tipo_pagamento' => $fatura->tipo_pagamento,
                            'valor' => $fatura->valor,
                            'data_vencimento' => $fatura->data_vencimento
                        ]);
                        
                        $result = $fatura->save();
                        
                        \Illuminate\Support\Facades\Log::info("Fatura criada com ID {$fatura->id} para conta {$conta->id}: tipo_pagamento={$pagamento['tipo']}, valor={$valorProporcionalPagamento}, resultado={$result}");
                    } catch (\Exception $e) {
                        \Illuminate\Support\Facades\Log::error("Erro ao criar fatura para conta ID {$conta->id}: " . $e->getMessage());
                        \Illuminate\Support\Facades\Log::error("Detalhes da exceção: " . get_class($e) . " - " . $e->getTraceAsString());
                        throw $e;
                    }
                    
                    try {
                        // Registrar histórico para este pagamento
                        $this->registrarHistorico(
                            $conta->id, 
                            $valorProporcionalPagamento, 
                            $pagamento['tipo'],
                            $isPagamentoTotalConta,
                            "Obs: " . ($pagamento['observacao'] ?? '')
                        );
                        
                        \Illuminate\Support\Facades\Log::info("Histórico registrado para pagamento: conta_id={$conta->id}, valor={$valorProporcionalPagamento}");
                    } catch (\Exception $e) {
                        \Illuminate\Support\Facades\Log::error("Erro ao registrar histórico para conta ID {$conta->id}: " . $e->getMessage());
                        throw $e;
                    }
                    
                    $valorRestanteProporcionado -= $valorProporcionalPagamento;
                }
                
                // Atualizar valor restante
                $valorRestante -= $valorASerPago;
            }
            
            // Registrar entrada no caixa/conta se especificado
            if (isset($request->conta_empresa_id)) {
                // Para cada pagamento processado, registrar entrada
                foreach ($pagamentosProcessados as $pagamento) {
                    $descricao = "Recebimento de conta(s)";
                    if (!empty($pagamento['observacao'])) {
                        $descricao .= " - " . $pagamento['observacao'];
                    }
                    
                    try {
                        $data = [
                            'conta_id' => $request->conta_empresa_id,
                            'descricao' => $descricao,
                            'tipo_pagamento' => $pagamento['tipo'],
                            'valor' => $pagamento['valor'],
                            'tipo' => 'entrada',
                            'data_movimento' => $dataRecebimento // Garantir que use a data atual
                        ];
                        
                        $itemContaEmpresa = ItemContaEmpresa::create($data);
                        $this->util->atualizaSaldo($itemContaEmpresa);
                        
                        \Illuminate\Support\Facades\Log::info("Entrada registrada no caixa: conta_id={$request->conta_empresa_id}, valor={$pagamento['valor']}");
                    } catch (\Exception $e) {
                        \Illuminate\Support\Facades\Log::error("Erro ao registrar entrada no caixa: " . $e->getMessage());
                        throw $e;
                    }
                }
            }
            
            // Registrar log da operação
            $descricaoLog = "Recebimento de " . ($contasProcessadas + $contasParciaisProcessadas) . " contas, valor: R$ " . __moeda($valorPago);
            __createLog(request()->empresa_id, 'Conta a Receber', 'recebimento', $descricaoLog);
            
            \Illuminate\Support\Facades\Log::info("Transação finalizada, commitando...");
            DB::commit();
            \Illuminate\Support\Facades\Log::info("Commit realizado com sucesso");
            
            // Mensagem de sucesso
            if ($contasProcessadas > 0 && $contasParciaisProcessadas > 0) {
                $mensagem = "Total de contas recebidas integralmente: {$contasProcessadas}. Total de contas recebidas parcialmente: {$contasParciaisProcessadas}.";
            } elseif ($contasProcessadas > 0) {
                $mensagem = "Total de contas recebidas integralmente: {$contasProcessadas}.";
            } elseif ($contasParciaisProcessadas > 0) {
                $mensagem = "Total de contas recebidas parcialmente: {$contasParciaisProcessadas}.";
            } else {
                $mensagem = "Recebimento processado com sucesso.";
            }
            
            // Registrar mensagem nos logs
            \Illuminate\Support\Facades\Log::info("Recebimento concluído com sucesso: {$mensagem}");
            
            // Sempre gravar a mensagem na sessão flash
            session()->flash("flash_success", $mensagem);
            
            // Verificar se devemos redirecionar para o PDV
            $shouldRedirectToPdv = ($request->has('redirect_to_pdv') && $request->redirect_to_pdv == 1);
            
            \Illuminate\Support\Facades\Log::info("Redirecionamento: redirect_to_pdv=" . ($shouldRedirectToPdv ? 'sim' : 'não'));
            
            // Redirecionar para o PDV se solicitado
            if ($shouldRedirectToPdv) {
                \Illuminate\Support\Facades\Log::info("Redirecionando para PDV após recebimento de contas: {$mensagem}");
                // Cria um cookie para indicar sucesso ao redirecionar
                $cookie = cookie('recebimento_finalizado', 'true', 5);
                
                // Adiciona parâmetros na URL para reforçar o feedback visual
                return redirect()->route('frontbox.create', ['pagamento_sucesso' => 1])
                    ->withCookie($cookie)
                    ->with([
                        'show_success_modal' => true, 
                        'success_message' => $mensagem
                    ]);
            }
            
            \Illuminate\Support\Facades\Log::info("Redirecionando para lista de contas a receber após processamento bem-sucedido");
            return redirect()->route('conta-receber.index')->with('flash_success', $mensagem);
            
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error("ERRO CRÍTICO no processamento do recebimento: " . $e->getMessage());
            \Illuminate\Support\Facades\Log::error("Stack trace: " . $e->getTraceAsString());
            
            if (DB::transactionLevel() > 0) {
                DB::rollBack();
                \Illuminate\Support\Facades\Log::info("Transação abortada (rollback)");
            }
            
            session()->flash("flash_error", "Erro ao processar recebimento: " . $e->getMessage());
            return redirect()->back()->withInput();
        }
    }

    public function pay($id)
    {
        if (!__isCaixaAberto()) {
            session()->flash("flash_warning", "Abrir caixa antes de continuar!");
            return redirect()->route('caixa.create');
        }
        $item = ContaReceber::findOrFail($id);
        __validaObjetoEmpresa($item);
        
        if($item->status == 1) { // Só bloqueia se estiver totalmente recebida
            session()->flash("flash_warning", "Esta conta já está totalmente recebida!");
            return redirect()->route('conta-receber.index');
        }
        
        // Impedir recebimento de contas reparceladas
        if ($item->status == ContaReceber::STATUS_REPARCELADA) {
            session()->flash('flash_error', 'Contas reparceladas não podem ser recebidas.');
            return redirect()->back();
        }
        
        return view('conta-receber.pay', compact('item'));
    }

    public function payPut(Request $request, $id)
    {
        $usuario = Auth::user()->id;
        $caixa = Caixa::where('usuario_id', $usuario)->where('status', 1)->first();
        $item = ContaReceber::findOrFail($id);
        __validaObjetoEmpresa($item);

        try {
            // Converter valores para cálculos
            $valorRecebido = __convert_value_bd($request->valor_pago);
            $valorPendente = $item->valor_integral - ($item->valor_recebido ?? 0);
            
            // Verificar se o valor é menor que o total (pagamento parcial)
            $isPagamentoTotal = ($valorRecebido >= $valorPendente);
            
            // Se for pagamento parcial, verificar se a nova data de vencimento foi informada
            if (!$isPagamentoTotal && !$request->filled('data_nova_vencimento')) {
                session()->flash("flash_warning", "Para pagamentos parciais, é necessário informar uma nova data de vencimento.");
                return redirect()->back();
            }
            
            // Se for pagamento total
            if ($isPagamentoTotal) {
                $item->valor_recebido = ($item->valor_recebido ?? 0) + $valorRecebido;
                $item->status = 1; // marca como completamente recebido
            $item->data_recebimento = $request->data_recebimento;
            $item->tipo_pagamento = $request->tipo_pagamento;
            $item->caixa_id = $caixa->id;
            $item->save();

                $mensagemSucesso = "Conta recebida integralmente!";
            } else {
                // Se for pagamento parcial
                // Salvar data original de vencimento na primeira vez
                if (!$item->data_vencimento_original) {
                    $item->data_vencimento_original = $item->data_vencimento;
                }
                
                // Aplicar juros e multa, se não isento
                $jurosAplicados = 0;
                $multaAplicada = 0;
                
                // Verificar se solicitou isenção de juros/multa
                if ($request->has('isentar_multa')) {
                    // Verificar permissão do usuário ou senha do gerente
                    $podeisentar = false;
                    
                    // Verificar permissão do usuário via DB
                    $userId = auth()->id();
                    $empresaId = request()->empresa_id;
                    
                    // Verificar se tem permissão via roles (simplificado usando query direta)
                    $temPermissao = \Illuminate\Support\Facades\DB::table('model_has_permissions')
                        ->where('model_id', $userId)
                        ->where('model_type', 'App\\Models\\User')
                        ->whereExists(function ($query) {
                            $query->select(\Illuminate\Support\Facades\DB::raw(1))
                                  ->from('permissions')
                                  ->whereRaw('permissions.id = model_has_permissions.permission_id')
                                  ->where('permissions.name', 'financeiro_config');
                        })
                        ->exists();
                        
                    if ($temPermissao) {
                        // Se o usuário tem permissão direta
                        $podeisentar = true;
                    } else if ($request->filled('senha_gerente')) {
                        // Verificar senha do gerente
                        $gerentes = User::whereHas('permissions', function($q) {
                            $q->where('name', 'financeiro_config');
                        })->where('empresa_id', request()->empresa_id)->get();
                        
                        foreach ($gerentes as $gerente) {
                            if (Hash::check($request->senha_gerente, $gerente->password)) {
                                $podeisentar = true;
                                
                                // Registrar log de aprovação
                                $descricaoLog = "Isenção de juros/multa aprovada pelo gerente ID:" . $gerente->id . 
                                               " para a conta ID:" . $item->id;
                                __createLog(request()->empresa_id, 'Conta a Receber', 'isenção', $descricaoLog);
                                
                                break;
                            }
                        }
                        
                        if (!$podeisentar) {
                            session()->flash("flash_warning", "Senha de gerente inválida para isenção de juros/multa!");
                            return redirect()->back();
                        }
                    } else {
                        session()->flash("flash_warning", "É necessário fornecer a senha de gerente para isenção de juros/multa!");
                        return redirect()->back();
                    }
                }
                
                if ($request->filled('data_nova_vencimento')) {
                    // Se não está isento e tem nova data de vencimento
                    $jurosAplicados = $request->filled('juros_aplicados') ? 
                                     __convert_value_bd($request->juros_aplicados) : 0;
                    $multaAplicada = $request->filled('multa_aplicada') ? 
                                     __convert_value_bd($request->multa_aplicada) : 0;
                }
                
                // Registra o pagamento parcial
                $item->valor_recebido = ($item->valor_recebido ?? 0) + $valorRecebido;
                $item->status = 2; // marca como parcialmente pago (2)
                $item->tipo_pagamento = $request->tipo_pagamento;
                $item->caixa_id = $caixa->id;
                
                // Atualiza nova data de vencimento, se fornecida
                if ($request->filled('data_nova_vencimento')) {
                    $item->data_vencimento = $request->data_nova_vencimento;
                    $item->data_nova_vencimento = $request->data_nova_vencimento;
                }
                
                // Atualiza juros e multa aplicados
                $item->juros_aplicados = $jurosAplicados;
                $item->multa_aplicada = $multaAplicada;
                
                $item->save();
                
                $valorRestante = $item->valor_integral - $item->valor_recebido;
                $mensagemSucesso = "Pagamento parcial registrado! Valor restante: R$ " . __moeda($valorRestante);
            }

            // Registrar entrada no caixa/conta se especificado
            if (isset($request->conta_empresa_id)) {
                $data = [
                    'conta_id' => $request->conta_empresa_id,
                    'descricao' => ($isPagamentoTotal ? "Recebimento integral" : "Recebimento parcial") . " da conta " . $item->id,
                    'tipo_pagamento' => $request->tipo_pagamento,
                    'valor' => $valorRecebido,
                    'tipo' => 'entrada'
                ];
                $itemContaEmpresa = ItemContaEmpresa::create($data);
                $this->util->atualizaSaldo($itemContaEmpresa);
            }

            // Registra histórico da transação
            $this->registrarHistorico($item->id, $valorRecebido, $request->tipo_pagamento, $isPagamentoTotal);
            
            // Registra log da operação
            $descricaoLog = "Recebimento " . ($isPagamentoTotal ? "integral" : "parcial") . 
                " de R$ " . __moeda($valorRecebido) . 
                " da conta de cliente: " . $item->cliente->razao_social;
            __createLog(request()->empresa_id, 'Conta a Receber', 'recebimento', $descricaoLog);

            session()->flash("flash_success", $mensagemSucesso);
            return redirect()->route('conta-receber.index');
        } catch (\Exception $e) {
            session()->flash("flash_error", "Erro ao receber: " . $e->getMessage());
            return redirect()->back();
        }
    }

    /**
     * Registrar o histórico de transações da conta a receber
     */
    private function registrarHistorico($contaReceberID, $valorRecebido, $tipoPagamento, $integral = false, $tipoMovimento = null)
    {
        // Log para debug
        \Illuminate\Support\Facades\Log::info("Registrando histórico de pagamento: ", [
            'conta_id' => $contaReceberID,
            'valor' => $valorRecebido,
            'tipo_pagamento' => $tipoPagamento
        ]);
        
        $tipos = ContaReceber::tiposPagamento();
        $textoMovimento = "Recebimento de conta";
        
        if ($tipoMovimento) {
            $textoMovimento .= " - " . $tipoMovimento;
        }
        
        $descricao = $textoMovimento . " - Valor: R$ " . __moeda($valorRecebido);
        
        // Verificar se o tipo de pagamento existe no array de tipos
        if (isset($tipos[$tipoPagamento])) {
            $descricao .= " - Forma: " . $tipos[$tipoPagamento];
        } else {
            \Illuminate\Support\Facades\Log::warning("Tipo de pagamento não encontrado: {$tipoPagamento}");
            $descricao .= " - Forma: Desconhecido ({$tipoPagamento})";
        }
        
        if ($integral) {
            $descricao .= " (Integral)";
        } else {
            $descricao .= " (Parcial)";
        }
        
        $historico = ContaReceberHistorico::create([
            'conta_receber_id' => $contaReceberID,
            'valor' => $valorRecebido,
            'usuario_id' => Auth::id(),
            'descricao' => $descricao,
            'tipo_pagamento' => $tipoPagamento,
            'tipo_movimento' => $tipoMovimento ? $tipoMovimento : 'recebimento'
        ]);
        
        \Illuminate\Support\Facades\Log::info("Histórico registrado com ID {$historico->id}");
        
        return $historico;
    }

    /**
     * Registrar histórico específico de reparcelamento
     */
    private function registrarHistoricoReparcelamento($contaReceberID, $valor, $observacao = null)
    {
        try {
            $historico = new ContaReceberHistorico();
            $historico->conta_receber_id = $contaReceberID;
            $historico->valor = $valor;
            $historico->tipo_movimento = 'reparcelamento';
            $historico->data_movimento = date('Y-m-d H:i:s');
            $historico->usuario_id = auth()->id();
            $historico->observacao = $observacao;
            $historico->save();
            
            return true;
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Erro ao registrar histórico de reparcelamento: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Relatório de projeção de fluxo de caixa
     */
    public function relatorioProjecao(Request $request)
    {
        $empresa_id = request()->empresa_id;
        $start_date = $request->input('start_date', date('Y-m-d'));
        $end_date = $request->input('end_date', date('Y-m-d', strtotime('+30 days')));
        $local_id = $request->get('local_id');
        $hoje = date('Y-m-d');
        
        // Obter locais do usuário
        $locais = __getLocaisAtivoUsuario();
        $locais = $locais->pluck(['id']);
        
        // Log de depuração para verificar os filtros
        \Illuminate\Support\Facades\Log::info('Parâmetros de filtragem do relatório de projeção', [
            'empresa_id' => $empresa_id,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'local_id' => $local_id,
            'locais' => $locais,
        ]);
        
        // Contas a receber projetadas - Vamos verificar se há contas com status 0 (pendentes) para o período
        $contaReceberQuery = ContaReceber::where('empresa_id', $empresa_id)
            ->where('status', 0) // apenas pendentes
            ->whereDate('data_vencimento', '>=', $start_date)
            ->whereDate('data_vencimento', '<=', $end_date);
            
        // Verificar a quantidade de registros sem aplicar filtro de local
        $totalRegistros = (clone $contaReceberQuery)->count();
        \Illuminate\Support\Facades\Log::info('Total de registros sem filtro de local: ' . $totalRegistros);
        
        // Agora aplicar o filtro de local conforme o parâmetro
        $contasReceber = $contaReceberQuery
            ->when($local_id, function ($query) use ($local_id) {
                return $query->where('local_id', $local_id);
            })
            ->when(!$local_id, function ($query) use ($locais) {
                return $query->whereIn('local_id', $locais);
            })
            ->with(['cliente', 'categoria'])
            ->orderBy('data_vencimento')
            ->get();
            
        // Log para verificar as contas recuperadas
        \Illuminate\Support\Facades\Log::info('Contas a receber recuperadas após filtro de local', [
            'quantidade' => $contasReceber->count(),
            'total_valor' => $contasReceber->sum('valor_integral'),
            'total_recebido' => $contasReceber->sum('valor_recebido'),
            'total_pendente' => $contasReceber->sum(function($conta) { 
                return $conta->valorPendente(); 
            }),
        ]);
        
        // Se não encontrou contas com os filtros atuais, vamos verificar se existem contas no banco
        if ($contasReceber->isEmpty()) {
            // Verificar quantas contas a receber existem no banco para a empresa
            $totalContasEmpresa = ContaReceber::where('empresa_id', $empresa_id)->count();
            \Illuminate\Support\Facades\Log::warning('Nenhuma conta encontrada com os filtros aplicados. Total de contas para a empresa: ' . $totalContasEmpresa);
            
            // Verificar se existem contas pendentes
            $totalContasPendentes = ContaReceber::where('empresa_id', $empresa_id)
                ->where('status', 0)
                ->count();
            \Illuminate\Support\Facades\Log::warning('Total de contas pendentes para a empresa: ' . $totalContasPendentes);
            
            // Verificar se existem contas no período especificado
            $totalContasPeriodo = ContaReceber::where('empresa_id', $empresa_id)
                ->whereDate('data_vencimento', '>=', $start_date)
                ->whereDate('data_vencimento', '<=', $end_date)
                ->count();
            \Illuminate\Support\Facades\Log::warning('Total de contas no período especificado: ' . $totalContasPeriodo);
        }
        
        // Contas a pagar projetadas
        $contaPagarQuery = \App\Models\ContaPagar::where('empresa_id', $empresa_id)
            ->where('status', 0) // apenas pendentes
            ->whereDate('data_vencimento', '>=', $start_date)
            ->whereDate('data_vencimento', '<=', $end_date);
            
        // Capturar o SQL para depuração
        $queryClone = clone $contaPagarQuery;
        $sql = $queryClone->toSql();
        $bindings = $queryClone->getBindings();
        \Illuminate\Support\Facades\Log::debug('SQL da consulta de contas a pagar: ' . $sql, $bindings);
            
        // Aplicar filtro de local    
        $contasPagar = $contaPagarQuery
            ->when($local_id, function ($query) use ($local_id) {
                return $query->where('local_id', $local_id);
            })
            ->when(!$local_id, function ($query) use ($locais) {
                return $query->whereIn('local_id', $locais);
            })
            ->with(['fornecedor', 'categoria'])
            ->orderBy('data_vencimento')
            ->get();
        
        // Log para verificar as contas a pagar recuperadas
        \Illuminate\Support\Facades\Log::info('Contas a pagar recuperadas', [
            'quantidade' => $contasPagar->count(),
            'total_valor' => $contasPagar->sum('valor_integral'),
            'total_pago' => $contasPagar->sum('valor_pago'),
            'total_pendente' => $contasPagar->sum(function($conta) { 
                return $conta->valorPendente(); 
            }),
        ]);
        
        // Verificar se estamos usando os locais corretos no filtro
        if ($contasReceber->isEmpty() && !$local_id) {
            // Se não encontrou contas com os locais do usuário, tenta recuperar com todos os locais da empresa
            \Illuminate\Support\Facades\Log::warning('Nenhuma conta encontrada com os locais do usuário, tentando com todos os locais');
            
            $todosLocais = \App\Models\Localizacao::where('empresa_id', $empresa_id)
                ->where('status', 1)
                ->pluck('id');
                
            $contasReceber = ContaReceber::where('empresa_id', $empresa_id)
                ->where('status', 0)
                ->whereDate('data_vencimento', '>=', $start_date)
                ->whereDate('data_vencimento', '<=', $end_date)
                ->whereIn('local_id', $todosLocais)
                ->with(['cliente', 'categoria'])
                ->orderBy('data_vencimento')
                ->get();
                
            \Illuminate\Support\Facades\Log::info('Tentativa com todos os locais', [
                'quantidade' => $contasReceber->count(),
                'total_valor' => $contasReceber->sum('valor_integral'),
            ]);
        }
        
        // Calcular totais e agrupar por data
        $datas = [];
        $totaisReceber = 0;
        $totaisPagar = 0;
        
        
        foreach ($contasReceber as $conta) {
            $data = $conta->data_vencimento;
            if (!isset($datas[$data])) {
                $datas[$data] = [
                    'receber' => 0,
                    'pagar' => 0,
                    'saldo' => 0,
                    'saldo_previsto' => 0,
                    'saldo_realizado' => 0,
                    'receber_realizado' => 0,
                    'pagar_realizado' => 0
                ];
            }
            
            // Verificar o tipo de dado e valor
            $valorIntegral = $conta->valor_integral;
            $valorRecebido = $conta->valor_recebido ?? 0;
            
            // Log para verificar os tipos de dados
            if ($valorIntegral === null || !is_numeric($valorIntegral) || $valorIntegral <= 0) {
                \Illuminate\Support\Facades\Log::warning('Valor integral inválido na conta ID: ' . $conta->id, [
                    'valor_integral' => $valorIntegral,
                    'tipo' => gettype($valorIntegral)
                ]);
            }
            
            // Garantir que o valor pendente seja calculado corretamente
            $valorPendente = max(0, $valorIntegral - $valorRecebido);
            
            // Log do cálculo
            \Illuminate\Support\Facades\Log::debug('Cálculo do valor pendente', [
                'conta_id' => $conta->id,
                'valor_integral' => $valorIntegral,
                'tipo_integral' => gettype($valorIntegral),
                'valor_recebido' => $valorRecebido,
                'tipo_recebido' => gettype($valorRecebido),
                'valorPendente' => $valorPendente,
                'valorPendenteMetodo' => $conta->valorPendente()
            ]);
            
            $datas[$data]['receber'] += $valorPendente;
            $totaisReceber += $valorPendente;
        }
        
        foreach ($contasPagar as $conta) {
            $data = $conta->data_vencimento;
            if (!isset($datas[$data])) {
                $datas[$data] = [
                    'receber' => 0,
                    'pagar' => 0,
                    'saldo' => 0,
                    'saldo_previsto' => 0,
                    'saldo_realizado' => 0,
                    'receber_realizado' => 0,
                    'pagar_realizado' => 0
                ];
            }
            
            // Garantir que o valor pendente seja calculado corretamente
            $valorPendente = max(0, $conta->valor_integral - ($conta->valor_pago ?? 0));
            
            $datas[$data]['pagar'] += $valorPendente;
            $totaisPagar += $valorPendente;
        }
        
        // Log dos totais calculados
        \Illuminate\Support\Facades\Log::info('Totais calculados para o relatório', [
            'totaisReceber' => $totaisReceber,
            'totaisPagar' => $totaisPagar,
            'saldo' => $totaisReceber - $totaisPagar,
            'datas' => count($datas)
        ]);
        
        // Verificar se os totais foram calculados corretamente
        if ($totaisReceber == 0 && count($contasReceber) > 0) {
            \Illuminate\Support\Facades\Log::warning('Inconsistência: Existem contas a receber mas o total está zerado. Recalculando...');
            
            // Recalcular diretamente a partir da coleção
            $totaisReceber = $contasReceber->sum(function($conta) {
                $valorIntegral = (float)$conta->valor_integral;
                $valorRecebido = (float)($conta->valor_recebido ?? 0);
                return max(0, $valorIntegral - $valorRecebido);
            });
            
            \Illuminate\Support\Facades\Log::info('Total recalculado: ' . $totaisReceber);
            
            // Se ainda estiver zerado, tentar uma abordagem diferente
            if ($totaisReceber == 0) {
                \Illuminate\Support\Facades\Log::warning('Ainda zerado após recálculo. Verificando valores manualmente...');
                
                foreach ($contasReceber as $index => $conta) {
                    $valorIntegral = (float)$conta->valor_integral;
                    $valorRecebido = (float)($conta->valor_recebido ?? 0);
                    $pendente = max(0, $valorIntegral - $valorRecebido);
                    
                    \Illuminate\Support\Facades\Log::info("Conta #{$index} - ID: {$conta->id}", [
                        'valor_integral' => $valorIntegral,
                        'valor_recebido' => $valorRecebido,
                        'pendente' => $pendente,
                        'tipo_integral' => gettype($conta->valor_integral)
                    ]);
                    
                    if ($pendente > 0) {
                        $totaisReceber += $pendente;
                    }
                }
                
                \Illuminate\Support\Facades\Log::info('Total após verificação manual: ' . $totaisReceber);
            }
        }
        
        // Mesmo procedimento para contas a pagar
        if ($totaisPagar == 0 && count($contasPagar) > 0) {
            \Illuminate\Support\Facades\Log::warning('Inconsistência: Existem contas a pagar mas o total está zerado. Recalculando...');
            
            // Recalcular diretamente a partir da coleção
            $totaisPagar = $contasPagar->sum(function($conta) {
                $valorIntegral = (float)$conta->valor_integral;
                $valorPago = (float)($conta->valor_pago ?? 0);
                return max(0, $valorIntegral - $valorPago);
            });
            
            \Illuminate\Support\Facades\Log::info('Total recalculado: ' . $totaisPagar);
        }
        
        // Calcular saldo para cada data
        $saldoAcumulado = 0;
        $saldoPrevistoAcumulado = 0;
        $saldoRealizadoAcumulado = 0;
        
        // Buscar contas realizadas (já recebidas/pagas) no período
        $contasRecebidasQuery = ContaReceber::where('empresa_id', $empresa_id)
            ->where('status', 1) // apenas recebidas
            ->whereDate('data_recebimento', '>=', $start_date)
            ->whereDate('data_recebimento', '<=', $end_date);
            
        $contasRecebidas = $contasRecebidasQuery
            ->when($local_id, function ($query) use ($local_id) {
                return $query->where('local_id', $local_id);
            })
            ->when(!$local_id, function ($query) use ($locais) {
                return $query->whereIn('local_id', $locais);
            })
            ->get();
            
        $contasPagasQuery = \App\Models\ContaPagar::where('empresa_id', $empresa_id)
            ->where('status', 1) // apenas pagas
            ->whereDate('data_pagamento', '>=', $start_date)
            ->whereDate('data_pagamento', '<=', $end_date);
            
        $contasPagas = $contasPagasQuery
            ->when($local_id, function ($query) use ($local_id) {
                return $query->where('local_id', $local_id);
            })
            ->when(!$local_id, function ($query) use ($locais) {
                return $query->whereIn('local_id', $locais);
            })
            ->get();
        
        // Adicionar as contas recebidas ao array de datas
        foreach ($contasRecebidas as $conta) {
            $data = $conta->data_recebimento;
            if (!isset($datas[$data])) {
                $datas[$data] = [
                    'receber' => 0,
                    'pagar' => 0,
                    'saldo' => 0,
                    'saldo_previsto' => 0,
                    'saldo_realizado' => 0,
                    'receber_realizado' => 0,
                    'pagar_realizado' => 0
                ];
            }
            
            $valorRecebido = (float)$conta->valor_recebido;
            $datas[$data]['receber_realizado'] += $valorRecebido;
        }
        
        // Adicionar as contas pagas ao array de datas
        foreach ($contasPagas as $conta) {
            $data = $conta->data_pagamento;
            if (!isset($datas[$data])) {
                $datas[$data] = [
                    'receber' => 0,
                    'pagar' => 0,
                    'saldo' => 0,
                    'saldo_previsto' => 0,
                    'saldo_realizado' => 0,
                    'receber_realizado' => 0,
                    'pagar_realizado' => 0
                ];
            }
            
            $valorPago = (float)$conta->valor_pago;
            $datas[$data]['pagar_realizado'] += $valorPago;
        }
        
        // Buscar contas previstas (todas, independente do status)
        $contasPrevistasReceber = ContaReceber::where('empresa_id', $empresa_id)
            ->whereDate('data_vencimento', '>=', $start_date)
            ->whereDate('data_vencimento', '<=', $end_date)
            ->when($local_id, function ($query) use ($local_id) {
                return $query->where('local_id', $local_id);
            })
            ->when(!$local_id, function ($query) use ($locais) {
                return $query->whereIn('local_id', $locais);
            })
            ->get();
            
        $contasPrevistasPagar = \App\Models\ContaPagar::where('empresa_id', $empresa_id)
            ->whereDate('data_vencimento', '>=', $start_date)
            ->whereDate('data_vencimento', '<=', $end_date)
            ->when($local_id, function ($query) use ($local_id) {
                return $query->where('local_id', $local_id);
            })
            ->when(!$local_id, function ($query) use ($locais) {
                return $query->whereIn('local_id', $locais);
            })
            ->get();
        
        // Calcular saldo previsto para cada data
        $saldoPrevistoTotal = 0;
        $saldoRealizadoTotal = 0;
        
        ksort($datas); // ordenar por data
        
        foreach ($datas as $data => &$valores) {
            // Saldo pendente (original)
            $saldoAcumulado += $valores['receber'] - $valores['pagar'];
            $valores['saldo'] = $saldoAcumulado;
            
            // Saldo realizado
            $saldoRealizadoAcumulado += $valores['receber_realizado'] - $valores['pagar_realizado'];
            $valores['saldo_realizado'] = $saldoRealizadoAcumulado;
            $saldoRealizadoTotal += $valores['receber_realizado'] - $valores['pagar_realizado'];
            
            // Calcular saldo previsto para esta data
            $receberPrevisto = 0;
            $pagarPrevisto = 0;
            
            foreach ($contasPrevistasReceber as $conta) {
                if ($conta->data_vencimento == $data) {
                    $receberPrevisto += (float)$conta->valor_integral;
                }
            }
            
            foreach ($contasPrevistasPagar as $conta) {
                if ($conta->data_vencimento == $data) {
                    $pagarPrevisto += (float)$conta->valor_integral;
                }
            }
            
            $saldoDiaPrevisto = $receberPrevisto - $pagarPrevisto;
            $saldoPrevistoAcumulado += $saldoDiaPrevisto;
            $valores['saldo_previsto'] = $saldoPrevistoAcumulado;
            $saldoPrevistoTotal += $saldoDiaPrevisto;
        }
        
        $title = "Relatório de Projeção de Fluxo de Caixa";
        
        return view('conta-receber.relatorio-projecao', compact(
            'contasReceber', 
            'contasPagar', 
            'datas', 
            'totaisReceber', 
            'totaisPagar',
            'saldoPrevistoTotal',
            'saldoRealizadoTotal',
            'start_date', 
            'end_date',
            'title'
        ));
    }

    /**
     * Relatório de inadimplência
     */
    public function relatorioInadimplencia(Request $request)
    {
        $empresa_id = request()->empresa_id;
        $hoje = date('Y-m-d');
        $dias_atraso = $request->input('dias_atraso', 0);
        $cliente_id = $request->input('cliente_id');
        $local_id = $request->get('local_id');
        $categoria_id = $request->input('categoria_id');
        
        // Obter locais do usuário
        $locais = __getLocaisAtivoUsuario();
        $locais = $locais->pluck(['id']);
        
        // Calcular data limite com base nos dias de atraso
        $data_limite = $dias_atraso ? date('Y-m-d', strtotime("-{$dias_atraso} days")) : null;
        
        // Contas vencidas
        $query = ContaReceber::where('empresa_id', $empresa_id)
            ->where('status', 0) // apenas pendentes
            ->whereDate('data_vencimento', '<', $hoje); // já vencidas
        
        // Filtrar por dias de atraso específico
        if ($data_limite) {
            $query->whereDate('data_vencimento', '>=', $data_limite);
        }
        
        // Aplicar filtros adicionais
        $contas = $query->when(!empty($cliente_id), function ($query) use ($cliente_id) {
                return $query->where('cliente_id', $cliente_id);
            })
            ->when($local_id, function ($query) use ($local_id) {
                return $query->where('local_id', $local_id);
            })
            ->when(!$local_id, function ($query) use ($locais) {
                return $query->whereIn('local_id', $locais);
            })
            ->when($categoria_id, function ($query) use ($categoria_id) {
                return $query->where('categoria_id', $categoria_id);
            })
            ->with(['cliente', 'categoria', 'localizacao'])
            ->orderBy('data_vencimento')
            ->get();
        
        // Agrupar por cliente
        $clientesAgrupados = [];
        $totalInadimplencia = 0;
        
        foreach ($contas as $conta) {
            $cliente_id = $conta->cliente_id;
            if (!isset($clientesAgrupados[$cliente_id])) {
                $clientesAgrupados[$cliente_id] = [
                    'cliente' => $conta->cliente,
                    'contas' => [],
                    'total' => 0,
                    'maior_atraso' => 0
                ];
            }
            
            $valorPendente = $conta->valorPendente();
            $diasAtraso = abs(floor((strtotime($hoje) - strtotime($conta->data_vencimento)) / (60 * 60 * 24)));
            
            $clientesAgrupados[$cliente_id]['contas'][] = $conta;
            $clientesAgrupados[$cliente_id]['total'] += $valorPendente;
            $clientesAgrupados[$cliente_id]['maior_atraso'] = max($clientesAgrupados[$cliente_id]['maior_atraso'], $diasAtraso);
            
            $totalInadimplencia += $valorPendente;
        }
        
        // Ordenar por maior valor total
        uasort($clientesAgrupados, function($a, $b) {
            return $b['total'] <=> $a['total'];
        });
        
        // Obter clientes para filtro
        $clientes = \App\Models\Cliente::where('empresa_id', $empresa_id)->orderBy('razao_social')->get();
        
        // Obter categorias para filtro
        $categorias = \App\Models\CategoriaConta::where('empresa_id', $empresa_id)
            ->where(function($query) {
                $query->where('tipo', 'receber')
                      ->orWhere('tipo', 'ambos');
            })
            ->where('ativo', true)
            ->orderBy('nome')
            ->get();
        
        // Agrupar contas por cliente
        $contasPorCliente = $contas->groupBy('cliente_id');
        $totalInadimplencia = $contas->sum('valor_integral') - $contas->sum('valor_pago');
        
        $title = "Relatório de Inadimplência";
        
        return view('conta-receber.relatorio-inadimplencia', compact(
            'contasPorCliente',
            'totalInadimplencia',
            'dias_atraso',
            'cliente_id',
            'categoria_id',
            'title',
            'clientes',
            'categorias',
            'clientesAgrupados'
        ));
    }

    private function gerarParcelasFuturas($conta, $periodicidade, $total_parcelas)
    {
        for ($i = 2; $i <= $total_parcelas; $i++) {
            $parcela = clone $conta;
            $parcela->status = 0;
            $parcela->valor_recebido = 0;
            $parcela->data_recebimento = null;
            $parcela->parent_id = $conta->id;
            $parcela->parcela_atual = $i;
            
            // Garantir que a descrição seja mantida
            $parcela->descricao = $conta->descricao;
            
            // Calcular próxima data de vencimento baseado na periodicidade
            $dataBase = $conta->data_vencimento;
            switch ($periodicidade) {
                case ContaReceber::PERIODICIDADE_DIARIA:
                    $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' days'));
                    break;
                case ContaReceber::PERIODICIDADE_SEMANAL:
                    $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' weeks'));
                    break;
                case ContaReceber::PERIODICIDADE_QUINZENAL:
                    $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 15) . ' days'));
                    break;
                case ContaReceber::PERIODICIDADE_MENSAL:
                    $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' months'));
                    break;
                case ContaReceber::PERIODICIDADE_BIMESTRAL:
                    $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 2) . ' months'));
                    break;
                case ContaReceber::PERIODICIDADE_TRIMESTRAL:
                    $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 3) . ' months'));
                    break;
                case ContaReceber::PERIODICIDADE_SEMESTRAL:
                    $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 6) . ' months'));
                    break;
                case ContaReceber::PERIODICIDADE_ANUAL:
                    $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' years'));
                    break;
                default:
                    $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' months'));
            }
            
            $parcela->data_vencimento = $dataVencimento;
            $parcela->save();
            
            $descricaoLog = "Vencimento (parcela $i): " . __data_pt($dataVencimento, 0) . " R$ " . __moeda($conta->valor_integral);
            __createLog($conta->empresa_id, 'Conta a Receber', 'cadastrar', $descricaoLog);
        }
    }

    /**
     * Método para recebimento parcial
     */
    public function receberParcial(Request $request, $id)
    {
        try {
            // Checar se o caixa está aberto
            if (!__isCaixaAberto()) {
                session()->flash("flash_warning", "Abrir caixa antes de continuar!");
                return redirect()->route('caixa.create');
            }
            
            // Obter usuário e caixa atual
            $usuario = Auth::user()->id;
            $caixa = Caixa::where('usuario_id', $usuario)->where('status', 1)->first();
            
            if (!$caixa) {
                throw new \Exception("Caixa não encontrado para o usuário atual.");
            }
            
            // Buscar conta a receber
            $conta = ContaReceber::findOrFail($id);
            __validaObjetoEmpresa($conta);
            
            // Verificar se não está totalmente recebida
            if ($conta->status == ContaReceber::STATUS_RECEBIDO) {
                session()->flash("flash_info", "Esta conta já está totalmente recebida.");
                return redirect()->back();
            }
            
            // Validar dados da requisição
            $request->validate([
                'valor_recebido' => 'required|numeric|min:0.01',
                'tipo_pagamento' => 'required|string',
                'data_recebimento' => 'required|date',
                'data_nova_vencimento' => 'required|date|after:data_recebimento',
            ]);
            
            // Calcular valor pendente atual
            $valor_integral = (float) $conta->valor_integral;
            $valor_ja_recebido = (float) ($conta->valor_recebido ?? 0);
            $valor_recebido_agora = (float) $request->valor_recebido;
            
            // Verificar se o valor recebido é maior que o pendente
            $valor_pendente = $valor_integral - $valor_ja_recebido;
            if ($valor_recebido_agora > $valor_pendente) {
                session()->flash("flash_warning", "Valor recebido excede o valor pendente da conta.");
                return redirect()->back();
            }
            
            // Se primeira vez que está recebendo parcialmente, guarda data original
            if ($conta->status == ContaReceber::STATUS_PENDENTE) {
                $conta->data_vencimento_original = $conta->data_vencimento;
            }
            
            // Atualizar conta
            $conta->valor_recebido = $valor_ja_recebido + $valor_recebido_agora;
            $conta->data_recebimento = $request->data_recebimento;
            $conta->tipo_pagamento = $request->tipo_pagamento;
            $conta->data_nova_vencimento = $request->data_nova_vencimento;
            $conta->caixa_id = $caixa->id;
            
            // Buscar parâmetros de juros e multa
            $config = $this->getConfiguracoesFinanceiras();
            
            // Verificar se está vencida para aplicar juros
            $hoje = date('Y-m-d');
            if ($conta->data_vencimento < $hoje) {
                // Calcular dias de atraso
                $data_vencimento = new \DateTime($conta->data_vencimento);
                $data_hoje = new \DateTime($hoje);
                $diff = $data_vencimento->diff($data_hoje);
                $dias_atraso = $diff->days;
                
                // Verificar carência
                if ($dias_atraso > $config->carencia_dias) {
                    // Descontar dias de carência (se houver)
                    $dias_atraso -= $config->carencia_dias;
                    
                    // Calcular juros (juros diário * dias de atraso)
                    $juros = ($config->juros_diario / 100) * $dias_atraso;
                    $valor_juros = ($valor_pendente - $valor_recebido_agora) * $juros;
                    
                    // Calcular multa (percentual fixo)
                    $multa = $config->multa_atraso / 100;
                    $valor_multa = ($valor_pendente - $valor_recebido_agora) * $multa;
                    
                    // Guardar valores calculados
                    $conta->juros_aplicados += $valor_juros;
                    $conta->multa_aplicada += $valor_multa;
                    
                    // Atualizar valor total com juros e multa
                    $conta->valor_integral = $valor_integral + $valor_juros + $valor_multa;
                }
            }
            
            // Verificar se é recebimento total ou parcial
            if ($conta->valor_recebido >= $valor_integral) {
                $conta->status = ContaReceber::STATUS_RECEBIDO;
            } else {
                $conta->status = ContaReceber::STATUS_PARCIAL;
                // Atualizar a data de vencimento
                $conta->data_vencimento = $request->data_nova_vencimento;
            }
            
            $conta->save();
            
            // Log da operação
            $descricaoLog = "Recebimento parcial de R$ " . __moeda($valor_recebido_agora) . 
                          " de " . $conta->cliente->razao_social . 
                          ". Nova data de vencimento: " . date('d/m/Y', strtotime($request->data_nova_vencimento));
            __createLog(request()->empresa_id, 'Conta a Receber', 'recebimento_parcial', $descricaoLog);
            
            session()->flash("flash_success", "Recebimento parcial realizado com sucesso.");
            
        } catch (\Exception $e) {
            __createLog(request()->empresa_id, 'Conta a Receber', 'erro', $e->getMessage());
            session()->flash("flash_error", "Erro ao processar recebimento parcial: " . $e->getMessage());
        }
        
        return redirect()->back();
    }

    /**
     * Método para obter as configurações financeiras
     */
    private function getConfiguracoesFinanceiras()
    {
        $config = \App\Models\ConfiguracaoFinanceira::where('empresa_id', request()->empresa_id)->first();
        
        if (!$config) {
            // Criar configuração default se não existir
            $config = \App\Models\ConfiguracaoFinanceira::create([
                'empresa_id' => request()->empresa_id,
                'juros_diario' => 0.033, // 1% ao mês ÷ 30 dias
                'multa_atraso' => 2.0,   // 2% fixo por atraso
                'carencia_dias' => 0,    // Sem carência
                'prazo_padrao_dias' => 30
            ]);
        }
        
        return $config;
    }

    /**
     * Realiza o reparcelamento de uma conta a receber
     */
    public function reparcelar(Request $request, $id)
    {
        $contaOriginal = ContaReceber::findOrFail($id);
        __validaObjetoEmpresa($contaOriginal);

        // Validar se a conta pode ser reparcelada (não pode estar recebida)
        if ($contaOriginal->status == ContaReceber::STATUS_RECEBIDO) {
            session()->flash('flash_error', 'Não é possível reparcelar uma conta já recebida.');
            return redirect()->back();
        }

        DB::beginTransaction();
        try {
            // Obtém os dados do formulário
            $totalParcelas = (int)$request->total_parcelas;
            $primeiraData = $request->primeira_data;
            $periodicidade = $request->periodicidade;
            $aplicarJurosMora = $request->has('juros_mora');
            
            // Valor pendente a ser dividido nas parcelas
            $valorPendente = $contaOriginal->valorPendente();
            
            // Se solicitado, calcular juros e mora sobre o valor pendente
            $jurosAplicados = 0;
            $multaAplicada = 0;
            
            if ($aplicarJurosMora && $contaOriginal->isVencida()) {
                // Buscar configuração financeira
                $configuracao = \App\Models\ConfiguracaoFinanceira::where('empresa_id', request()->empresa_id)->first();
                
                // Calcular dias de atraso
                $dataVencimento = Carbon::parse($contaOriginal->data_vencimento);
                $hoje = Carbon::now();
                $diasAtraso = $dataVencimento->diffInDays($hoje);
                
                // Aplicar juros diários
                $jurosDiario = $configuracao ? $configuracao->juros_diario : 0.033; // Padrão 0.033% ao dia
                $jurosTotal = ($jurosDiario / 100) * $diasAtraso * $valorPendente;
                $jurosAplicados = round($jurosTotal, 2);
                
                // Aplicar multa
                $multaAtraso = $configuracao ? $configuracao->multa_atraso : 2; // Padrão 2%
                $multaAplicada = round(($multaAtraso / 100) * $valorPendente, 2);
                
                // Adicionar juros e multa ao valor pendente
                $valorPendente += ($jurosAplicados + $multaAplicada);
            }
            
            // Calcular valor de cada parcela (arredondado para 2 casas decimais)
            $valorParcela = round($valorPendente / $totalParcelas, 2);
            
            // Verificar se tem diferença de centavos devido ao arredondamento
            $valorTotalParcelas = $valorParcela * $totalParcelas;
            $diferenca = $valorPendente - $valorTotalParcelas;
            
            // Criar array para armazenar as parcelas
            $parcelas = [];
            
            // Marcar a conta original como reparcelada
            $contaOriginal->status = ContaReceber::STATUS_REPARCELADA;
            $contaOriginal->data_recebimento = date('Y-m-d');
            $contaOriginal->observacao = $contaOriginal->observacao . "\nConta reparcelada em " . date('d/m/Y') . " em $totalParcelas parcelas.";
            $contaOriginal->save();
            
            // Disparar evento
            // event(new ContaReceberEvent($contaOriginal));
            
            // Registrar no log do sistema
            __createLog(request()->empresa_id, 'Conta a Receber', 'reparcelamento', "Conta #" . $contaOriginal->id . " reparcelada em " . $totalParcelas . " parcelas");
            
            // Gerar as parcelas
            for ($i = 1; $i <= $totalParcelas; $i++) {
                // Calcular data de vencimento de acordo com a periodicidade
                $dataVencimento = Carbon::parse($primeiraData);
                
                if ($i > 1) {
                    switch ($periodicidade) {
                        case ContaReceber::PERIODICIDADE_DIARIA:
                            $dataVencimento->addDays($i - 1);
                            break;
                        case ContaReceber::PERIODICIDADE_SEMANAL:
                            $dataVencimento->addWeeks($i - 1);
                            break;
                        case ContaReceber::PERIODICIDADE_QUINZENAL:
                            $dataVencimento->addDays(($i - 1) * 15);
                            break;
                        case ContaReceber::PERIODICIDADE_MENSAL:
                            $dataVencimento->addMonths($i - 1);
                            break;
                        case ContaReceber::PERIODICIDADE_BIMESTRAL:
                            $dataVencimento->addMonths(($i - 1) * 2);
                            break;
                        case ContaReceber::PERIODICIDADE_TRIMESTRAL:
                            $dataVencimento->addMonths(($i - 1) * 3);
                            break;
                        case ContaReceber::PERIODICIDADE_SEMESTRAL:
                            $dataVencimento->addMonths(($i - 1) * 6);
                            break;
                        case ContaReceber::PERIODICIDADE_ANUAL:
                            $dataVencimento->addYears($i - 1);
                            break;
                        default:
                            $dataVencimento->addMonths($i - 1);
                    }
                }
                
                // Ajustar o valor da última parcela para compensar diferenças de arredondamento
                $valorParcelaAtual = $valorParcela;
                if ($i == $totalParcelas && $diferenca != 0) {
                    $valorParcelaAtual += $diferenca;
                }
                
                // Criar a nova parcela
                $novaParcela = new ContaReceber();
                $novaParcela->fill([
                    'empresa_id' => $contaOriginal->empresa_id,
                    'cliente_id' => $contaOriginal->cliente_id,
                    'descricao' => "Parcela $i/$totalParcelas - Reparcelamento da conta #" . $contaOriginal->id,
                    'valor_integral' => $valorParcelaAtual,
                    'data_vencimento' => $dataVencimento->format('Y-m-d'),
                    'status' => ContaReceber::STATUS_PENDENTE,
                    'tipo_pagamento' => $contaOriginal->tipo_pagamento,
                    'local_id' => $contaOriginal->local_id,
                    'categoria_id' => $contaOriginal->categoria_id,
                    'parent_id' => $contaOriginal->id,
                    'total_parcelas' => $totalParcelas,
                    'parcela_atual' => $i,
                    'observacao' => "Parcela gerada por reparcelamento da conta #" . $contaOriginal->id
                ]);
                
                $novaParcela->save();
                $parcelas[] = $novaParcela;
            }
            
            // Registrar log da operação
            $descricaoLog = "Reparcelamento da conta #" . $contaOriginal->id . 
                           " do cliente " . $contaOriginal->cliente->razao_social . 
                           " no valor de R$ " . __moeda($valorPendente) . 
                           " em " . $totalParcelas . " parcelas";
                           
            if ($aplicarJurosMora && ($jurosAplicados > 0 || $multaAplicada > 0)) {
                $descricaoLog .= " (incluindo juros de R$ " . __moeda($jurosAplicados) . 
                                " e multa de R$ " . __moeda($multaAplicada) . ")";
            }
            
            __createLog(request()->empresa_id, 'Conta a Receber', 'reparcelamento', $descricaoLog);
            
            // Registrar no histórico
            $observacao = "Reparcelamento em $totalParcelas parcelas";
            if ($aplicarJurosMora && ($jurosAplicados > 0 || $multaAplicada > 0)) {
                $observacao .= " (incluindo juros de R$ " . __moeda($jurosAplicados) . 
                              " e multa de R$ " . __moeda($multaAplicada) . ")";
            }
            $this->registrarHistoricoReparcelamento($contaOriginal->id, $valorPendente, $observacao);
            
            DB::commit();
            
            session()->flash('flash_success', 'Conta reparcelada com sucesso em ' . $totalParcelas . ' parcelas.');
        return redirect()->route('conta-receber.index');
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            __createLog(request()->empresa_id, 'Conta a Receber', 'erro', $e->getMessage());
            session()->flash('flash_error', 'Erro ao reparcelar conta: ' . $e->getMessage());
            return redirect()->back();
        }
    }
    
    /**
     * Retorna o histórico de uma conta a receber via API
     */
    public function getHistorico($id)
    {
        try {
            $conta = ContaReceber::findOrFail($id);
            __validaObjetoEmpresa($conta);
            
            $historico = ContaReceberHistorico::where('conta_receber_id', $id)
                ->with('usuario')
                ->orderBy('data_movimento', 'desc')
                ->get();
                
            // Calcular valor restante após cada movimentação
            $valorRestante = $conta->valor_integral;
            $historicoFormatado = [];
            foreach ($historico as $item) {
                $valorRestante -= $item->valor;
                $historicoFormatado[] = [
                    'data_movimento' => $item->data_movimento,
                    'tipo_movimento' => $item->tipo_movimento,
                    'tipo_movimento_formatado' => __(ucfirst(str_replace('_', ' ', $item->tipo_movimento))),
                    'valor' => $item->valor,
                    'valor_restante' => $valorRestante,
                    'tipo_pagamento' => $item->tipo_pagamento,
                    'data_recebimento' => $conta->data_recebimento,
                    'data_vencimento' => $conta->data_vencimento,
                    'observacao' => $item->observacao ?? $item->descricao,
                    'usuario' => $item->usuario ? $item->usuario->name : null,
                ];
            }
            
            // Se for uma conta reparcelada, buscar as parcelas geradas
            if ($conta->status == ContaReceber::STATUS_REPARCELADA) {
                $parcelas = ContaReceber::where('parent_id', $id)
                    ->select('id', 'parcela_atual', 'total_parcelas', 'valor_integral', 'data_vencimento', 'created_at')
                    ->orderBy('parcela_atual')
                    ->get();
                
                return response()->json([
                    'historico' => $historicoFormatado,
                    'parcelas' => $parcelas,
                    'status' => $conta->status
                ]);
            }
            
            return response()->json(['historico' => $historicoFormatado]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}


