<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\MenuAccessLog;
use App\Models\UserSession;
use App\Models\Empresa;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class AcessosClientesController extends Controller
{
    /**
     * Construtor: restringe acesso apenas para SuperAdmin
     */
    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            if (!__isMaster()) {
                return redirect()->route('home')->with('error', 'Acesso negado. Apenas administradores podem acessar esta funcionalidade.');
            }
            return $next($request);
        });
    }

    /**
     * Exibe a página principal com o formulário de filtros
     */
    public function index()
    {
        $empresas = Empresa::orderBy('nome')->get();
        return view('acessos_clientes.index', compact('empresas'));
    }

    /**
     * Processa os dados e exibe o relatório com base nos filtros
     */
    public function relatorio(Request $request)
    {
        $startDate = $request->start_date;
        $endDate = $request->end_date;
        $empresaId = $request->empresa_id;
        $usuarioId = $request->usuario_id;
        $ignorarSuperAdmin = $request->has('ignorar_superadmin');
        
        // Consulta básica
        $query = MenuAccessLog::query()
            ->select(
                'menu',
                'submenu',
                DB::raw('COUNT(*) as total_acessos')
            )
            ->whereNotNull('menu');
            
        // Aplicar filtros
        if ($startDate) {
            $query->whereDate('created_at', '>=', $startDate);
        }
        
        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }
        
        if ($empresaId) {
            $query->where('empresa_id', $empresaId);
        }
        
        if ($usuarioId) {
            $query->where('usuario_id', $usuarioId);
        }
        
        // Excluir SuperAdmin se a flag estiver marcada
        if ($ignorarSuperAdmin) {
            $query->whereDoesntHave('usuario', function($q) {
                $q->where('email', env('MAILMASTER'));
            });
        }
        
        // Agrupar e ordenar
        $menusMaisAcessados = $query->groupBy('menu', 'submenu')
            ->orderBy('total_acessos', 'desc')
            ->get();
        
        // Calcular métricas adicionais
        $metricas = $this->calcularMetricas($startDate, $endDate, $empresaId, $usuarioId, $ignorarSuperAdmin);
        
        // Obter estatísticas por empresa
        $acessosPorEmpresa = $this->obterEstatisticasPorEmpresa($startDate, $endDate, $empresaId, $ignorarSuperAdmin);
        
        // Obter estatísticas por usuário
        $acessosPorUsuario = $this->obterEstatisticasPorUsuario($startDate, $endDate, $empresaId, $usuarioId, $ignorarSuperAdmin);
        
        // Obter telas mais acessadas por usuário/empresa
        $telasMaisAcessadas = $this->obterTelasMaisAcessadas($startDate, $endDate, $empresaId, $usuarioId, $ignorarSuperAdmin);
        
        // Obter empresas para o filtro
        $empresas = Empresa::orderBy('nome')->get();
        
        // Obter usuários para o filtro (com base na empresa selecionada)
        $usuarios = collect();
        if ($empresaId) {
            $usuarios = User::whereHas('empresa', function($query) use ($empresaId) {
                $query->where('empresa_id', $empresaId);
            })->get();
        }
        
        return view('acessos_clientes.relatorio', compact(
            'menusMaisAcessados', 
            'metricas', 
            'empresas',
            'usuarios',
            'startDate',
            'endDate',
            'empresaId',
            'usuarioId',
            'acessosPorEmpresa',
            'acessosPorUsuario',
            'telasMaisAcessadas',
            'ignorarSuperAdmin'
        ));
    }
    
    /**
     * Calcula métricas adicionais com base nos filtros
     */
    private function calcularMetricas($startDate, $endDate, $empresaId, $usuarioId, $ignorarSuperAdmin = false)
    {
        // Base query para sessões
        $sessionQuery = UserSession::query();
        
        // Aplicar filtros
        if ($startDate) {
            $sessionQuery->whereDate('created_at', '>=', $startDate);
        }
        
        if ($endDate) {
            $sessionQuery->whereDate('created_at', '<=', $endDate);
        }
        
        if ($empresaId) {
            $sessionQuery->where('empresa_id', $empresaId);
        }
        
        if ($usuarioId) {
            $sessionQuery->where('usuario_id', $usuarioId);
        }
        
        // Excluir SuperAdmin se a flag estiver marcada
        if ($ignorarSuperAdmin) {
            $sessionQuery->whereDoesntHave('usuario', function($q) {
                $q->where('email', env('MAILMASTER'));
            });
        }
        
        // Total de logins no período
        $totalLogins = $sessionQuery->count();
        
        // Tempo médio por sessão (em minutos)
        $tempoMedio = $sessionQuery->whereNotNull('session_duration')
            ->avg('session_duration');
        if ($tempoMedio) {
            $tempoMedio = round($tempoMedio / 60, 2); // Converter segundos para minutos
        } else {
            $tempoMedio = 0;
        }
        
        // Média de funcionalidades usadas por sessão
        $mediaFuncionalidades = $sessionQuery->avg('features_used') ?: 0;
        
        return [
            'frequencia_login' => $totalLogins,
            'tempo_medio_sessao' => $tempoMedio,
            'media_funcionalidades' => round($mediaFuncionalidades, 2)
        ];
    }
    
    /**
     * Obter estatísticas agregadas de acesso por empresa
     */
    private function obterEstatisticasPorEmpresa($startDate, $endDate, $empresaId, $ignorarSuperAdmin = false)
    {
        // Query base
        $query = MenuAccessLog::query()
            ->select(
                'empresa_id',
                DB::raw('COUNT(DISTINCT usuario_id) as total_usuarios'),
                DB::raw('COUNT(*) as total_acessos'),
                DB::raw('COUNT(DISTINCT DATE(created_at)) as dias_com_acesso'),
                DB::raw('COUNT(DISTINCT menu) as menus_distintos')
            )
            ->whereNotNull('empresa_id')
            ->groupBy('empresa_id');
        
        // Aplicar filtros
        if ($startDate) {
            $query->whereDate('created_at', '>=', $startDate);
        }
        
        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }
        
        if ($empresaId) {
            $query->where('empresa_id', $empresaId);
        }
        
        // Excluir SuperAdmin se a flag estiver marcada
        if ($ignorarSuperAdmin) {
            $query->whereDoesntHave('usuario', function($q) {
                $q->where('email', env('MAILMASTER'));
            });
        }
        
        // Executar a consulta
        $acessosPorEmpresa = $query->get();
        
        // Carregar dados das empresas
        $empresas = Empresa::whereIn('id', $acessosPorEmpresa->pluck('empresa_id'))->get()
            ->keyBy('id');
        
        // Adicionar nome da empresa aos resultados
        $acessosPorEmpresa->each(function($item) use ($empresas) {
            $item->nome_empresa = $empresas->get($item->empresa_id)->nome ?? 'Empresa desconhecida';
            
            // Calcular média de acessos por dia
            if ($item->dias_com_acesso > 0) {
                $item->media_acessos_por_dia = round($item->total_acessos / $item->dias_com_acesso, 2);
            } else {
                $item->media_acessos_por_dia = 0;
            }
            
            // Calcular média de acessos por usuário
            if ($item->total_usuarios > 0) {
                $item->media_acessos_por_usuario = round($item->total_acessos / $item->total_usuarios, 2);
            } else {
                $item->media_acessos_por_usuario = 0;
            }
        });
        
        return $acessosPorEmpresa->sortByDesc('total_acessos')->values();
    }
    
    /**
     * Obter estatísticas detalhadas de acesso por usuário
     */
    private function obterEstatisticasPorUsuario($startDate, $endDate, $empresaId, $usuarioId, $ignorarSuperAdmin = false)
    {
        // Query base
        $query = MenuAccessLog::query()
            ->select(
                'usuario_id',
                'empresa_id',
                DB::raw('COUNT(*) as total_acessos'),
                DB::raw('COUNT(DISTINCT DATE(created_at)) as dias_com_acesso'),
                DB::raw('COUNT(DISTINCT menu) as menus_distintos')
            )
            ->whereNotNull('usuario_id')
            ->groupBy('usuario_id', 'empresa_id');
        
        // Aplicar filtros
        if ($startDate) {
            $query->whereDate('created_at', '>=', $startDate);
        }
        
        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }
        
        if ($empresaId) {
            $query->where('empresa_id', $empresaId);
        }
        
        if ($usuarioId) {
            $query->where('usuario_id', $usuarioId);
        }
        
        // Excluir SuperAdmin se a flag estiver marcada
        if ($ignorarSuperAdmin) {
            $query->whereDoesntHave('usuario', function($q) {
                $q->where('email', env('MAILMASTER'));
            });
        }
        
        // Executar a consulta
        $acessosPorUsuario = $query->get();
        
        // Carregar dados dos usuários
        $usuarios = User::whereIn('id', $acessosPorUsuario->pluck('usuario_id'))->get()
            ->keyBy('id');
            
        // Carregar dados das empresas
        $empresas = Empresa::whereIn('id', $acessosPorUsuario->pluck('empresa_id'))->get()
            ->keyBy('id');
        
        // Adicionar nome do usuário e empresa aos resultados
        $acessosPorUsuario->each(function($item) use ($usuarios, $empresas) {
            $item->nome_usuario = $usuarios->get($item->usuario_id)->name ?? 'Usuário desconhecido';
            $item->nome_empresa = $empresas->get($item->empresa_id)->nome ?? 'Empresa desconhecida';
            
            // Calcular média de acessos por dia
            if ($item->dias_com_acesso > 0) {
                $item->media_acessos_por_dia = round($item->total_acessos / $item->dias_com_acesso, 2);
            } else {
                $item->media_acessos_por_dia = 0;
            }
        });
        
        return $acessosPorUsuario->sortByDesc('total_acessos')->values();
    }
    
    /**
     * Obter as telas mais acessadas por usuário/empresa
     */
    private function obterTelasMaisAcessadas($startDate, $endDate, $empresaId, $usuarioId, $ignorarSuperAdmin = false)
    {
        // Se temos usuário específico, agrupamos por menu/submenu
        if ($usuarioId) {
            $query = MenuAccessLog::query()
                ->select(
                    'menu',
                    'submenu',
                    DB::raw('COUNT(*) as total_acessos')
                )
                ->whereNotNull('menu')
                ->where('usuario_id', $usuarioId);
                
            if ($startDate) {
                $query->whereDate('created_at', '>=', $startDate);
            }
            
            if ($endDate) {
                $query->whereDate('created_at', '<=', $endDate);
            }
            
            if ($empresaId) {
                $query->where('empresa_id', $empresaId);
            }
            
            return $query->groupBy('menu', 'submenu')
                ->orderBy('total_acessos', 'desc')
                ->limit(10)
                ->get();
        }
        // Se temos empresa específica, agrupamos por usuário/menu
        else if ($empresaId) {
            $query = MenuAccessLog::query()
                ->select(
                    'usuario_id',
                    'menu',
                    'submenu',
                    DB::raw('COUNT(*) as total_acessos')
                )
                ->whereNotNull('menu')
                ->where('empresa_id', $empresaId);
                
            if ($startDate) {
                $query->whereDate('created_at', '>=', $startDate);
            }
            
            if ($endDate) {
                $query->whereDate('created_at', '<=', $endDate);
            }
            
            // Excluir SuperAdmin se a flag estiver marcada
            if ($ignorarSuperAdmin) {
                $query->whereDoesntHave('usuario', function($q) {
                    $q->where('email', env('MAILMASTER'));
                });
            }
            
            $result = $query->groupBy('usuario_id', 'menu', 'submenu')
                ->orderBy('total_acessos', 'desc')
                ->limit(20)
                ->get();
                
            // Carregar dados dos usuários
            $usuarios = User::whereIn('id', $result->pluck('usuario_id'))->get()
                ->keyBy('id');
                
            // Adicionar nome do usuário aos resultados
            $result->each(function($item) use ($usuarios) {
                $item->nome_usuario = $usuarios->get($item->usuario_id)->name ?? 'Usuário desconhecido';
            });
            
            return $result;
        }
        // Caso contrário, agrupamos por empresa/menu
        else {
            $query = MenuAccessLog::query()
                ->select(
                    'empresa_id',
                    'menu',
                    'submenu',
                    DB::raw('COUNT(*) as total_acessos')
                )
                ->whereNotNull('menu')
                ->whereNotNull('empresa_id');
                
            if ($startDate) {
                $query->whereDate('created_at', '>=', $startDate);
            }
            
            if ($endDate) {
                $query->whereDate('created_at', '<=', $endDate);
            }
            
            // Excluir SuperAdmin se a flag estiver marcada
            if ($ignorarSuperAdmin) {
                $query->whereDoesntHave('usuario', function($q) {
                    $q->where('email', env('MAILMASTER'));
                });
            }
            
            $result = $query->groupBy('empresa_id', 'menu', 'submenu')
                ->orderBy('total_acessos', 'desc')
                ->limit(20)
                ->get();
                
            // Carregar dados das empresas
            $empresas = Empresa::whereIn('id', $result->pluck('empresa_id'))->get()
                ->keyBy('id');
                
            // Adicionar nome da empresa aos resultados
            $result->each(function($item) use ($empresas) {
                $item->nome_empresa = $empresas->get($item->empresa_id)->nome ?? 'Empresa desconhecida';
            });
            
            return $result;
        }
    }
    
    /**
     * Retorna os usuários de uma empresa específica (para AJAX)
     */
    public function getUsuariosPorEmpresa(Request $request)
    {
        $empresaId = $request->empresa_id;
        
        if (!$empresaId) {
            return response()->json([]);
        }
        
        $usuarios = User::whereHas('empresa', function($query) use ($empresaId) {
            $query->where('empresa_id', $empresaId);
        })->select('id', 'name')->get();
        
        return response()->json($usuarios);
    }
} 