<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Consignacao;
use App\Models\ItemConsignacao;
use App\Models\Produto;
use App\Models\CategoriaProduto;
use App\Models\Marca;
use App\Models\Cliente;
use App\Models\Funcionario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ConsignacaoController extends Controller
{
    /**
     * Buscar produtos para consignação
     */
    public function buscaProdutos(Request $request)
    {
        $query = Produto::where('empresa_id', request()->empresa_id)
            ->where('status', 1);

        if ($request->categoria_id) {
            $query->where('categoria_id', $request->categoria_id);
        }

        if ($request->marca_id) {
            $query->where('marca_id', $request->marca_id);
        }

        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('nome', 'like', '%' . $request->search . '%')
                  ->orWhere('codigo', 'like', '%' . $request->search . '%')
                  ->orWhere('codigo_barras', 'like', '%' . $request->search . '%');
            });
        }

        $produtos = $query->with(['categoria', 'marca', 'variacoes'])
            ->orderBy('nome')
            ->paginate(20);

        return response()->json($produtos);
    }

    /**
     * Buscar produto por código de barras
     */
    public function buscaProdutoPorCodigo(Request $request)
    {
        $produto = Produto::where('empresa_id', request()->empresa_id)
            ->where('status', 1)
            ->where(function($query) use ($request) {
                $query->where('codigo_barras', $request->codigo)
                      ->orWhere('codigo', $request->codigo);
            })
            ->with(['categoria', 'marca', 'variacoes'])
            ->first();

        if (!$produto) {
            return response()->json(['error' => 'Produto não encontrado'], 404);
        }

        return response()->json($produto);
    }

    /**
     * Buscar categorias
     */
    public function categorias()
    {
        $categorias = CategoriaProduto::where('empresa_id', request()->empresa_id)
            ->where('categoria_id', null)
            ->with('subcategorias')
            ->orderBy('nome')
            ->get();

        return response()->json($categorias);
    }

    /**
     * Buscar marcas
     */
    public function marcas()
    {
        $marcas = Marca::where('empresa_id', request()->empresa_id)
            ->orderBy('nome')
            ->get();

        return response()->json($marcas);
    }

    /**
     * Buscar clientes
     */
    public function clientes(Request $request)
    {
        $query = Cliente::where('empresa_id', request()->empresa_id);

        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('nome', 'like', '%' . $request->search . '%')
                  ->orWhere('cpf_cnpj', 'like', '%' . $request->search . '%');
            });
        }

        $clientes = $query->orderBy('nome')->paginate(20);

        return response()->json($clientes);
    }

    /**
     * Buscar funcionários
     */
    public function funcionarios()
    {
        $funcionarios = Funcionario::where('empresa_id', request()->empresa_id)
            ->orderBy('nome')
            ->get();

        return response()->json($funcionarios);
    }

    /**
     * Adicionar produto à consignação
     */
    public function adicionarProduto(Request $request)
    {
        $request->validate([
            'produto_id' => 'required|exists:produtos,id',
            'produto_variacao_id' => 'nullable|exists:produto_variacaos,id',
            'quantidade' => 'required|numeric|min:0.001',
            'valor_unitario' => 'required|numeric|min:0.01',
            'observacao' => 'nullable|string'
        ]);

        $produto = Produto::findOrFail($request->produto_id);

        // Verificar se o produto tem estoque suficiente
        $estoqueDisponivel = $this->verificarEstoque($request->produto_id, $request->produto_variacao_id);
        
        if ($estoqueDisponivel < $request->quantidade) {
            return response()->json([
                'error' => 'Estoque insuficiente. Disponível: ' . $estoqueDisponivel
            ], 400);
        }

        $item = [
            'produto_id' => $request->produto_id,
            'produto_variacao_id' => $request->produto_variacao_id,
            'quantidade' => $request->quantidade,
            'valor_unitario' => $request->valor_unitario,
            'sub_total' => $request->quantidade * $request->valor_unitario,
            'observacao' => $request->observacao,
            'produto' => $produto
        ];

        return response()->json($item);
    }

    /**
     * Verificar estoque disponível
     */
    private function verificarEstoque($produtoId, $variacaoId = null)
    {
        // Implementar verificação de estoque
        // Por enquanto retorna um valor alto para não bloquear
        return 999999;
    }

    /**
     * Buscar itens de uma consignação
     */
    public function itensConsignacao($consignacaoId)
    {
        $itens = ItemConsignacao::with(['produto', 'produtoVariacao'])
            ->where('consignacao_id', $consignacaoId)
            ->get();

        return response()->json($itens);
    }

    /**
     * Atualizar quantidade de devolução
     */
    public function atualizarDevolucao(Request $request, $consignacaoId)
    {
        $request->validate([
            'item_id' => 'required|exists:item_consignacoes,id',
            'quantidade_devolvida' => 'required|numeric|min:0'
        ]);

        $item = ItemConsignacao::where('consignacao_id', $consignacaoId)
            ->findOrFail($request->item_id);

        $quantidadeMaxima = $item->quantidade_vendida;
        
        if ($request->quantidade_devolvida > $quantidadeMaxima) {
            return response()->json([
                'error' => 'Quantidade de devolução não pode ser maior que a quantidade vendida'
            ], 400);
        }

        $item->quantidade_devolvida = $request->quantidade_devolvida;
        $item->save();

        return response()->json($item);
    }

    /**
     * Buscar dados para impressão
     */
    public function dadosImpressao($consignacaoId)
    {
        $consignacao = Consignacao::with([
            'cliente', 
            'funcionario', 
            'usuario', 
            'itens.produto', 
            'itens.produtoVariacao'
        ])
        ->where('empresa_id', request()->empresa_id)
        ->findOrFail($consignacaoId);

        return response()->json($consignacao);
    }
}
