<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Produto;
use App\Models\Estoque;
use Illuminate\Support\Facades\DB;

class VerificarEstoque extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'estoque:verificar {--empresa_id= : ID da empresa específica} {--produto_id= : ID do produto específico} {--output= : Arquivo de saída (opcional)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Verifica a consistência dos estoques e gera relatório';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $empresa_id = $this->option('empresa_id');
        $produto_id = $this->option('produto_id');
        $output_file = $this->option('output');

        $this->info('Iniciando verificação de estoques...');

        // Query para buscar produtos
        $query = Produto::query();
        
        if ($empresa_id) {
            $query->where('empresa_id', $empresa_id);
        }
        
        if ($produto_id) {
            $query->where('id', $produto_id);
        }

        $produtos = $query->get();
        
        if ($produtos->isEmpty()) {
            $this->error('Nenhum produto encontrado com os critérios especificados.');
            return 1;
        }

        $this->info("Encontrados {$produtos->count()} produtos para verificar.");
        
        $bar = $this->output->createProgressBar($produtos->count());
        $bar->start();

        $relatorio = [];
        $problemas = [];

        foreach ($produtos as $produto) {
            $info = $this->verificarProduto($produto);
            
            if (!empty($info['problemas'])) {
                $problemas[] = $info;
            }
            
            $relatorio[] = $info;
            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);

        // Exibir resumo
        $this->exibirResumo($relatorio, $problemas);

        // Salvar relatório se solicitado
        if ($output_file) {
            $this->salvarRelatorio($relatorio, $output_file);
        }

        return 0;
    }

    private function verificarProduto($produto)
    {
        $info = [
            'produto_id' => $produto->id,
            'nome' => $produto->nome,
            'empresa_id' => $produto->empresa_id,
            'gerenciar_estoque' => $produto->gerenciar_estoque,
            'variacao_modelo_id' => $produto->variacao_modelo_id,
            'estoque_old' => $produto->estoque ? $produto->estoque->quantidade : 0,
            'estoque_real' => $produto->estoqueReal(),
            'estoque_formatado' => $produto->estoqueFormatado(),
            'registros_estoque' => [],
            'problemas' => []
        ];

        // Buscar todos os registros de estoque
        $estoques = Estoque::where('produto_id', $produto->id)->get();
        
        foreach ($estoques as $estoque) {
            $info['registros_estoque'][] = [
                'id' => $estoque->id,
                'quantidade' => $estoque->quantidade,
                'produto_variacao_id' => $estoque->produto_variacao_id,
                'local_id' => $estoque->local_id,
                'created_at' => $estoque->created_at,
                'updated_at' => $estoque->updated_at
            ];
        }

        // Verificar problemas
        if ($produto->gerenciar_estoque) {
            // Verificar se há registros duplicados
            $duplicados = $this->verificarDuplicados($estoques);
            if (!empty($duplicados)) {
                $info['problemas'][] = [
                    'tipo' => 'duplicados',
                    'descricao' => 'Registros duplicados encontrados',
                    'detalhes' => $duplicados
                ];
            }

            // Verificar inconsistência entre estoque antigo e real
            if (abs($info['estoque_old'] - $info['estoque_real']) > 0.01) {
                $info['problemas'][] = [
                    'tipo' => 'inconsistencia',
                    'descricao' => 'Diferença entre estoque antigo e real',
                    'estoque_old' => $info['estoque_old'],
                    'estoque_real' => $info['estoque_real'],
                    'diferenca' => $info['estoque_real'] - $info['estoque_old']
                ];
            }

            // Verificar produtos com variação sem estoque por variação
            if ($produto->variacao_modelo_id && $estoques->whereNull('produto_variacao_id')->count() > 0) {
                $info['problemas'][] = [
                    'tipo' => 'variacao_sem_estoque',
                    'descricao' => 'Produto com variação tem estoque sem especificar variação'
                ];
            }
        }

        return $info;
    }

    private function verificarDuplicados($estoques)
    {
        $duplicados = [];
        $agrupados = [];

        foreach ($estoques as $estoque) {
            $chave = ($estoque->produto_variacao_id ?? 'null') . '_' . ($estoque->local_id ?? 'null');
            if (!isset($agrupados[$chave])) {
                $agrupados[$chave] = [];
            }
            $agrupados[$chave][] = $estoque;
        }

        foreach ($agrupados as $chave => $grupo) {
            if (count($grupo) > 1) {
                $duplicados[] = [
                    'chave' => $chave,
                    'quantidade_registros' => count($grupo),
                    'registros' => $grupo->pluck('id')->toArray()
                ];
            }
        }

        return $duplicados;
    }

    private function exibirResumo($relatorio, $problemas)
    {
        $totalProdutos = count($relatorio);
        $produtosComProblemas = count($problemas);
        $totalRegistrosEstoque = collect($relatorio)->sum(function($item) {
            return count($item['registros_estoque']);
        });

        $this->info("📊 RESUMO DA VERIFICAÇÃO:");
        $this->info("   - Total de produtos: {$totalProdutos}");
        $this->info("   - Produtos com problemas: {$produtosComProblemas}");
        $this->info("   - Total de registros de estoque: {$totalRegistrosEstoque}");

        if ($produtosComProblemas > 0) {
            $this->warn("\n⚠️  PRODUTOS COM PROBLEMAS:");
            
            foreach ($problemas as $problema) {
                $this->warn("   - ID {$problema['produto_id']}: {$problema['nome']}");
                foreach ($problema['problemas'] as $p) {
                    $this->warn("     • {$p['descricao']}");
                }
            }

            $this->warn("\n💡 Para corrigir os problemas, execute:");
            $this->warn("   php artisan estoque:corrigir");
        } else {
            $this->info("\n✅ Nenhum problema encontrado!");
        }
    }

    private function salvarRelatorio($relatorio, $output_file)
    {
        $json = json_encode($relatorio, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        file_put_contents($output_file, $json);
        $this->info("📄 Relatório salvo em: {$output_file}");
    }
} 