<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Devolucao;
use App\Models\Caixa;
use Illuminate\Support\Facades\DB;

class UpdateDevolucoesCaixaId extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'devolucoes:update-caixa-id';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Atualiza o caixa_id das devoluções existentes baseado no usuário que criou';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Iniciando atualização do caixa_id nas devoluções...');

        // Buscar devoluções sem caixa_id
        $devolucoes = Devolucao::whereNull('caixa_id')->get();
        
        $this->info("Encontradas {$devolucoes->count()} devoluções sem caixa_id");

        $updated = 0;
        $errors = 0;

        foreach ($devolucoes as $devolucao) {
            try {
                // Buscar o caixa aberto pelo usuário na data da devolução
                $caixa = Caixa::where('usuario_id', $devolucao->usuario_id)
                    ->where('empresa_id', $devolucao->empresa_id)
                    ->where('created_at', '<=', $devolucao->created_at)
                    ->where(function($query) use ($devolucao) {
                        $query->where('data_fechamento', '>=', $devolucao->created_at)
                              ->orWhereNull('data_fechamento');
                    })
                    ->orderBy('created_at', 'desc')
                    ->first();

                if ($caixa) {
                    $devolucao->caixa_id = $caixa->id;
                    $devolucao->save();
                    $updated++;
                    $this->info("Devolução #{$devolucao->id} atualizada com caixa_id: {$caixa->id}");
                } else {
                    $this->warn("Devolução #{$devolucao->id} não encontrou caixa correspondente");
                    $errors++;
                }
            } catch (\Exception $e) {
                $this->error("Erro ao atualizar devolução #{$devolucao->id}: " . $e->getMessage());
                $errors++;
            }
        }

        $this->info("Atualização concluída!");
        $this->info("Devoluções atualizadas: {$updated}");
        $this->info("Erros: {$errors}");

        return 0;
    }
} 