<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Nfe;
use App\Models\Nfce;
use App\Models\FaturaNfe;
use App\Models\FaturaNfce;
use App\Models\TaxaPagamento;
use Illuminate\Support\Facades\DB;

class MigrarTaxasHistoricas extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'taxas:migrar {--empresa=* : IDs das empresas (opcional, todas se não informado)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Migra taxas históricas para vendas que não possuem taxa percentual registrada';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $empresaIds = $this->option('empresa');
        
        if (empty($empresaIds)) {
            // Se não for especificado, obter todas as empresas
            $empresaIds = DB::table('empresas')->pluck('id')->toArray();
        }
        
        $this->info('Iniciando migração de taxas históricas para ' . count($empresaIds) . ' empresas...');
        
        $totalNfes = 0;
        $totalNfces = 0;
        $totalFaturasNfe = 0;
        $totalFaturasNfce = 0;
        
        foreach ($empresaIds as $empresaId) {
            $this->info("Processando empresa ID: {$empresaId}");
            
            // Obter taxas atuais para referência
            $taxas = TaxaPagamento::where('empresa_id', $empresaId)->get()->keyBy(function($taxa) {
                $key = $taxa->tipo_pagamento;
                if (!empty($taxa->bandeira_cartao)) {
                    $key .= '_' . $taxa->bandeira_cartao;
                }
                return $key;
            });
            
            // Processar NFes
            $nfes = Nfe::where('empresa_id', $empresaId)
                      ->whereNull('taxa_percentual')
                      ->get();
            
            $this->info("NFes a serem processadas: " . count($nfes));
            
            foreach ($nfes as $nfe) {
                $tipoPagamento = $nfe->tipo_pagamento;
                $bandeiraCartao = $nfe->bandeira_cartao;
                $key = $tipoPagamento;
                
                if (!empty($bandeiraCartao) && $bandeiraCartao != '99') {
                    $key .= '_' . $bandeiraCartao;
                }
                
                if (isset($taxas[$key])) {
                    $nfe->taxa_percentual = $taxas[$key]->taxa;
                    $nfe->save();
                    $totalNfes++;
                } elseif (isset($taxas[$tipoPagamento])) {
                    // Tenta encontrar somente pelo tipo de pagamento
                    $nfe->taxa_percentual = $taxas[$tipoPagamento]->taxa;
                    $nfe->save();
                    $totalNfes++;
                }
            }
            
            // Processar NFCes
            $nfces = Nfce::where('empresa_id', $empresaId)
                       ->whereNull('taxa_percentual')
                       ->get();
            
            $this->info("NFCes a serem processadas: " . count($nfces));
            
            foreach ($nfces as $nfce) {
                $tipoPagamento = $nfce->tipo_pagamento;
                $bandeiraCartao = $nfce->bandeira_cartao;
                $key = $tipoPagamento;
                
                if (!empty($bandeiraCartao) && $bandeiraCartao != '99') {
                    $key .= '_' . $bandeiraCartao;
                }
                
                if (isset($taxas[$key])) {
                    $nfce->taxa_percentual = $taxas[$key]->taxa;
                    $nfce->save();
                    $totalNfces++;
                } elseif (isset($taxas[$tipoPagamento])) {
                    // Tenta encontrar somente pelo tipo de pagamento
                    $nfce->taxa_percentual = $taxas[$tipoPagamento]->taxa;
                    $nfce->save();
                    $totalNfces++;
                }
            }
            
            // Processar Faturas de NFe
            $faturasNfe = FaturaNfe::join('nfes', 'nfes.id', '=', 'fatura_nfes.nfe_id')
                                ->where('nfes.empresa_id', $empresaId)
                                ->whereNull('fatura_nfes.taxa_percentual')
                                ->select('fatura_nfes.*', 'nfes.bandeira_cartao')
                                ->get();
            
            $this->info("Faturas de NFe a serem processadas: " . count($faturasNfe));
            
            foreach ($faturasNfe as $fatura) {
                $tipoPagamento = $fatura->tipo_pagamento;
                $bandeiraCartao = $fatura->bandeira_cartao;
                $key = $tipoPagamento;
                
                if (!empty($bandeiraCartao) && $bandeiraCartao != '99') {
                    $key .= '_' . $bandeiraCartao;
                }
                
                if (isset($taxas[$key])) {
                    $fatura->taxa_percentual = $taxas[$key]->taxa;
                    $fatura->save();
                    $totalFaturasNfe++;
                } elseif (isset($taxas[$tipoPagamento])) {
                    // Tenta encontrar somente pelo tipo de pagamento
                    $fatura->taxa_percentual = $taxas[$tipoPagamento]->taxa;
                    $fatura->save();
                    $totalFaturasNfe++;
                }
            }
            
            // Processar Faturas de NFCe
            $faturasNfce = FaturaNfce::join('nfces', 'nfces.id', '=', 'fatura_nfces.nfce_id')
                                  ->where('nfces.empresa_id', $empresaId)
                                  ->whereNull('fatura_nfces.taxa_percentual')
                                  ->select('fatura_nfces.*', 'nfces.bandeira_cartao')
                                  ->get();
            
            $this->info("Faturas de NFCe a serem processadas: " . count($faturasNfce));
            
            foreach ($faturasNfce as $fatura) {
                $tipoPagamento = $fatura->tipo_pagamento;
                $bandeiraCartao = $fatura->bandeira_cartao;
                $key = $tipoPagamento;
                
                if (!empty($bandeiraCartao) && $bandeiraCartao != '99') {
                    $key .= '_' . $bandeiraCartao;
                }
                
                if (isset($taxas[$key])) {
                    $fatura->taxa_percentual = $taxas[$key]->taxa;
                    $fatura->save();
                    $totalFaturasNfce++;
                } elseif (isset($taxas[$tipoPagamento])) {
                    // Tenta encontrar somente pelo tipo de pagamento
                    $fatura->taxa_percentual = $taxas[$tipoPagamento]->taxa;
                    $fatura->save();
                    $totalFaturasNfce++;
                }
            }
        }
        
        $this->info('Migração concluída!');
        $this->info("Total processado:");
        $this->info("- NFes: {$totalNfes}");
        $this->info("- NFCes: {$totalNfces}");
        $this->info("- Faturas NFe: {$totalFaturasNfe}");
        $this->info("- Faturas NFCe: {$totalFaturasNfce}");
        
        return 0;
    }
} 