<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Produto;
use App\Models\Estoque;

class CorrigirEstoque extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'estoque:corrigir {--empresa_id= : ID da empresa específica} {--produto_id= : ID do produto específico}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Corrige registros de estoque inconsistentes e duplicados';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $empresa_id = $this->option('empresa_id');
        $produto_id = $this->option('produto_id');

        $this->info('Iniciando correção de estoques...');

        // Query para buscar produtos
        $query = Produto::query();
        
        if ($empresa_id) {
            $query->where('empresa_id', $empresa_id);
        }
        
        if ($produto_id) {
            $query->where('id', $produto_id);
        }

        $produtos = $query->get();
        
        if ($produtos->isEmpty()) {
            $this->error('Nenhum produto encontrado com os critérios especificados.');
            return 1;
        }

        $this->info("Encontrados {$produtos->count()} produtos para verificar.");
        
        $bar = $this->output->createProgressBar($produtos->count());
        $bar->start();

        $totalCorrecoes = 0;
        $produtosComProblemas = 0;

        foreach ($produtos as $produto) {
            $correcoes = $produto->corrigirEstoque();
            
            if (!empty($correcoes)) {
                $produtosComProblemas++;
                $totalCorrecoes += count($correcoes);
                
                $this->newLine();
                $this->warn("Produto ID {$produto->id} ({$produto->nome}) - {$correcoes[0]['registros_removidos']} registros consolidados");
            }
            
            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);

        if ($totalCorrecoes > 0) {
            $this->info("✅ Correção concluída!");
            $this->info("📊 Resumo:");
            $this->info("   - Produtos verificados: {$produtos->count()}");
            $this->info("   - Produtos com problemas: {$produtosComProblemas}");
            $this->info("   - Total de correções: {$totalCorrecoes}");
        } else {
            $this->info("✅ Nenhum problema de estoque encontrado!");
        }

        return 0;
    }
} 