<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Empresa;
use App\Models\ContaReceber;
use App\Models\ContaPagar;
use App\Models\Notificacao;
use App\Models\ItemNfe;
use App\Models\Produto;
use App\Models\Estoque;
use App\Models\ConfigGeral;
use App\Models\Agendamento;
use App\Models\ConfiguracaoAgendamento;
use App\Utils\WhatsAppUtil;
use Carbon\Carbon;

class AlertaCron extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'alerta:cron';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Cria alertas para empresas';

    /**
     * Execute the console command.
     */

    protected $whatsAppUtil;

    public function __construct(WhatsAppUtil $whatsAppUtil){
        parent::__construct();
        $this->whatsAppUtil = $whatsAppUtil;
    }

    public function handle()
    {
        $empresas = Empresa::where('status', 1)->get();

        foreach($empresas as $empresa){

            $config = ConfigGeral::where('empresa_id', $empresa->id)->first();
            $alertasAtivos = null;
            if($config != null){
                $alertasAtivos = json_decode($config->notificacoes);
            }
            
            // Alertas para contas a receber
            if($alertasAtivos == null || in_array('Contas a receber', $alertasAtivos)){
                // Contas que vencem hoje
                $contasReceber = ContaReceber::where('empresa_id', $empresa->id)
                ->where('status', 0)
                ->whereDate('data_vencimento', date('Y-m-d'))
                ->get();

                foreach($contasReceber as $conta){
                    $descricaoCurta = $conta->cliente->razao_social . " R$" . __moeda($conta->valor_integral);
                    $this->criaNotificacao('conta_recebers', $conta->id, $empresa, 'Conta a receber (Vencimento hoje)', $descricaoCurta, $conta);
                }
                
                // Contas que vencem em 3 dias
                $dataVencimento3Dias = Carbon::now()->addDays(3)->format('Y-m-d');
                $contasReceberFuturas = ContaReceber::where('empresa_id', $empresa->id)
                ->where('status', 0)
                ->whereDate('data_vencimento', $dataVencimento3Dias)
                ->get();
                
                foreach($contasReceberFuturas as $conta){
                    $descricaoCurta = $conta->cliente->razao_social . " R$" . __moeda($conta->valor_integral);
                    $this->criaNotificacao('conta_recebers', $conta->id, $empresa, 'Conta a receber (Vence em 3 dias)', $descricaoCurta, $conta, 'baixa');
                }
                
                // Contas vencidas há mais de 5 dias
                $dataVencida5Dias = Carbon::now()->subDays(5)->format('Y-m-d');
                $contasReceberVencidas = ContaReceber::where('empresa_id', $empresa->id)
                ->where('status', 0)
                ->whereDate('data_vencimento', '<=', $dataVencida5Dias)
                ->get();
                
                foreach($contasReceberVencidas as $conta){
                    $descricaoCurta = $conta->cliente->razao_social . " R$" . __moeda($conta->valor_integral) . " - ATRASADA";
                    $this->criaNotificacao('conta_recebers', $conta->id, $empresa, 'Conta a receber ATRASADA', $descricaoCurta, $conta, 'alta');
                }
            }

            // Alertas para contas a pagar
            if($alertasAtivos == null || in_array('Contas a pagar', $alertasAtivos)){
                // Contas que vencem hoje
                $contasPagar = ContaPagar::where('empresa_id', $empresa->id)
                ->where('status', 0)
                ->whereDate('data_vencimento', date('Y-m-d'))
                ->get();

                foreach($contasPagar as $conta){
                    $descricaoCurta = $conta->fornecedor->razao_social . " R$" . __moeda($conta->valor_integral);
                    $this->criaNotificacao('conta_pagars', $conta->id, $empresa, 'Conta a pagar (Vencimento hoje)', $descricaoCurta, $conta);
                }
                
                // Contas que vencem em 3 dias
                $dataVencimento3Dias = Carbon::now()->addDays(3)->format('Y-m-d');
                $contasPagarFuturas = ContaPagar::where('empresa_id', $empresa->id)
                ->where('status', 0)
                ->whereDate('data_vencimento', $dataVencimento3Dias)
                ->get();
                
                foreach($contasPagarFuturas as $conta){
                    $descricaoCurta = $conta->fornecedor->razao_social . " R$" . __moeda($conta->valor_integral);
                    $this->criaNotificacao('conta_pagars', $conta->id, $empresa, 'Conta a pagar (Vence em 3 dias)', $descricaoCurta, $conta, 'baixa');
                }
                
                // Contas não aprovadas que vencem em menos de 5 dias
                $dataVencimento5Dias = Carbon::now()->addDays(5)->format('Y-m-d');
                $contasPagarNaoAprovadas = ContaPagar::where('empresa_id', $empresa->id)
                ->where('status', 0)
                ->whereNull('aprovado_por')
                ->whereDate('data_vencimento', '<=', $dataVencimento5Dias)
                ->whereDate('data_vencimento', '>=', date('Y-m-d'))
                ->get();
                
                foreach($contasPagarNaoAprovadas as $conta){
                    $descricaoCurta = $conta->fornecedor->razao_social . " R$" . __moeda($conta->valor_integral);
                    $this->criaNotificacao('conta_pagars', $conta->id, $empresa, 'Conta a pagar AGUARDANDO APROVAÇÃO', $descricaoCurta, $conta, 'media');
                }
            }

            if($alertasAtivos == null || in_array('Alerta de validade', $alertasAtivos)){
                $produtosComAlertaValidade = Produto::where('empresa_id', $empresa->id)
                ->where('alerta_validade', '>', 0)->get();
                foreach($produtosComAlertaValidade as $produto){
                    $date = date('Y-m-d', strtotime(date('Y-m-d'). "+$produto->alerta_validade days"));
                    $itens = ItemNfe::where('produto_id', $produto->id)
                    ->whereDate('data_vencimento', $date)
                    ->get();

                    foreach($itens as $i){
                        $descricaoCurta = $i->produto->nome;
                        $this->criaNotificacao('compras', $i->id, $empresa, 'Alerta de vencimento', $descricaoCurta, $i, 'media');
                    }
                }
            }

            if($alertasAtivos == null || in_array('Alerta de estoque', $alertasAtivos)){
                $produtosComEstoqueMinimo = Produto::where('empresa_id', $empresa->id)
                ->where('estoque_minimo', '>', 0)->get();
                foreach($produtosComEstoqueMinimo as $produto){
                    $estoque = Estoque::where('produto_id', $produto->id)->first();

                    if($estoque != null && $estoque->quantidade <= $produto->estoque_minimo){
                        $descricaoCurta = $produto->nome;
                        $this->criaNotificacao('estoques', $estoque->id, $empresa, 'Alerta de estoque', $descricaoCurta, $estoque, 'media');
                    }
                }
            }

            $configuracaoAgendamento = ConfiguracaoAgendamento::where('empresa_id', $empresa->id)
            ->first();

            if($configuracaoAgendamento != null && $configuracaoAgendamento->token_whatsapp){
                $this->criaAlertaAgendamento($configuracaoAgendamento);
            }
        }
    }

    private function criaAlertaAgendamento($configuracao){
        // neste caso a notificação não é salva no sistema
        // apenas manda mensagem pelo whatsapp
        try{
            $hoje = date('Y-m-d');
            $amanha = date('Y-m-d', strtotime(date('Y-m-d'). '+1 days'));

            $agendamentos = Agendamento::
            where('empresa_id', $configuracao->empresa_id)
            ->where('status', 'agendado')
            ->where(function($q) use($hoje, $amanha, $configuracao){
                if($configuracao->avisa_hoje){
                    $q->whereDate('data', $hoje);
                } 
                if($configuracao->avisa_amanha){
                    $q->orWhereDate('data', $amanha);
                }
            })
            ->get();

            foreach($agendamentos as $a){
                $mensagem = "Olá " . $a->cliente->nome . ", ";
                
                $data = __data_pt($a->data, false);

                if(__data_pt($a->data, false) == __data_pt(date('Y-m-d'), false)){
                    $mensagem .= "você possui um agendamento hoje às " . $a->hora . ", " . $a->configuracao->descricao;
                }else{
                    $mensagem .= "você possui um agendamento amanha às " . $a->hora . ", " . $a->configuracao->descricao;
                }

                $mensagem .= "\nAtt! " . $a->empresa->nome;
                $this->whatsAppUtil->sendMessageWithToken($a->cliente->telefone, $mensagem, $configuracao->empresa_id, $configuracao->token_whatsapp);
            }
        }catch(\Exception $e){

        }
    }

    private function criaNotificacao($tabela, $referencia, $empresa, $titulo, $descricaoCurta, $objeto, $prioridade = 'baixa'){
        $item = Notificacao::where('empresa_id', $empresa->id)
        ->where('tabela', $tabela)
        ->where('referencia', $referencia)
        ->whereDate('created_at', date('Y-m-d'))
        ->first();

        if($item == null){
            $descricao = $this->getDescricao($tabela, $objeto);
            Notificacao::create([
                'empresa_id' => $empresa->id,
                'tabela' => $tabela,
                'descricao' => $descricao,
                'descricao_curta' => $descricaoCurta,
                'referencia' => $referencia,
                'status' => 1,
                'por_sistema' => 1,
                'prioridade' => $prioridade, 
                'visualizada' => 0,
                'titulo' => $titulo
            ]);
        }
    }

    private function getDescricao($tabela, $item){
        if($tabela == 'conta_recebers'){
            return view('notificacao.partials.conta_receber', compact('item'));
        }
        if($tabela == 'conta_pagars'){
            return view('notificacao.partials.conta_pagar', compact('item'));
        }
        if($tabela == 'compras'){
            return view('notificacao.partials.compras', compact('item'));
        }
        if($tabela == 'estoques'){
            return view('notificacao.partials.estoques', compact('item'));
        }
    }
}
