<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Troca;
use App\Models\ItemTroca;
use App\Models\Nfce;
use App\Models\Funcionario;
use App\Models\CategoriaProduto;
use App\Models\Caixa;
use App\Models\Empresa;
use App\Models\ConfigGeral;
use App\Utils\EstoqueUtil;
use App\Services\CupomNaoFiscal;

class TrocaController extends Controller
{
    protected $util;

    public function __construct(EstoqueUtil $util)
    {
        $this->util = $util;

        $this->middleware('permission:troca_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:troca_view', ['only' => ['show', 'index']]);
        $this->middleware('permission:troca_delete', ['only' => ['destroy']]);
    }

    public function index(Request $request){
        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $cliente_id = $request->get('cliente_id');

        $data = Troca::where('trocas.empresa_id', $request->empresa_id)
        ->select('trocas.*')
        ->join('nfces', 'nfces.id', '=', 'trocas.nfce_id')
        ->when(!empty($start_date), function ($query) use ($start_date) {
            return $query->whereDate('trocas.created_at', '>=', $start_date);
        })
        ->when(!empty($end_date), function ($query) use ($end_date,) {
            return $query->whereDate('trocas.created_at', '<=', $end_date);
        })
        ->when(!empty($cliente_id), function ($query) use ($cliente_id) {
            return $query->where('nfces.cliente_id', $cliente_id);
        })
        ->orderBy('trocas.created_at', 'desc')
        ->paginate(env("PAGINACAO"));
        return view('trocas.index', compact('data'));
    }

    public function create(Request $request){
        $codigo = $request->codigo;
        $numero_nfce = $request->numero_nfce;

        $item = Nfce::where('numero_sequencial', $codigo)->where('empresa_id', $request->empresa_id)
        ->first();
        if($item == null){
            $item = Nfce::where('numero', $numero_nfce)->where('empresa_id', $request->empresa_id)
            ->first();
        }

        if($item == null){
            session()->flash("flash_error", "Nenhuma venda encontrada!");
            return redirect()->back();
        }

        if (!__isCaixaAberto()) {
            session()->flash("flash_warning", "Abrir caixa antes de continuar!");
            return redirect()->route('caixa.create');
        }
        __validaObjetoEmpresa($item);

        $funcionarios = Funcionario::where('empresa_id', request()->empresa_id)->get();
        $cliente = $item->cliente;
        $funcionario = $item->funcionario;
        $caixa = __isCaixaAberto();
        $abertura = Caixa::where('usuario_id', get_id_user())
        ->where('status', 1)
        ->first();

        $isVendaSuspensa = 0;
        $categorias = CategoriaProduto::where('empresa_id', request()->empresa_id)->get();
        $config = ConfigGeral::where('empresa_id', request()->empresa_id)->first();

        $tiposPagamento = Nfce::tiposPagamento();
        // dd($tiposPagamento);
        if($config != null && $config->tipos_pagamento_pdv){
            $tipos_pagamento_pdv = json_decode($config->tipos_pagamento_pdv, true);
            if(is_array($tipos_pagamento_pdv) && sizeof($tipos_pagamento_pdv) > 0){
                $temp = [];
                foreach($tiposPagamento as $key => $t){
                    if(in_array($t, $tipos_pagamento_pdv)){
                        $temp[$key] = $t;
                    }
                }
                $tiposPagamento = $temp;
            }
        }
        $tiposPagamento['00'] = 'Vale Crédito';

        $msgTroca = "";
        if(sizeof($item->trocas) > 0){
            $msgTroca = "Essa venda já possui troca!";
        }

        return view('trocas.create', compact('item', 'funcionarios', 'cliente', 'funcionario', 'caixa', 'abertura', 
            'isVendaSuspensa', 'categorias', 'tiposPagamento', 'msgTroca'));
    }

    public function show($id)
    {
        $item = Troca::findOrFail($id);
        return view('trocas.show', compact('item'));
    }

    public function destroy($id)
    {
        $item = Troca::findOrFail($id);
        try {
            $descricaoLog = "#$item->numero_sequencial - R$ " . __moeda($item->valor_troca);

            \Log::info('Iniciando exclusão da troca:', [
                'troca_id' => $item->id,
                'nfce_id' => $item->nfce_id,
                'local_id' => $item->nfce ? $item->nfce->local_id : 'null'
            ]);

            foreach($item->itens as $i){
                \Log::info('Processando item da troca:', [
                    'item_id' => $i->id,
                    'produto_id' => $i->produto_id,
                    'produto' => $i->produto ? $i->produto->nome : 'null',
                    'gerenciar_estoque' => $i->produto ? $i->produto->gerenciar_estoque : 'null'
                ]);

                // Verificar se o produto existe e gerencia estoque
                if ($i->produto && $i->produto->gerenciar_estoque) {
                    $local_id = $item->nfce ? $item->nfce->local_id : null;
                    
                    \Log::info('Restaurando estoque:', [
                        'produto_id' => $i->produto->id,
                        'quantidade' => $i->quantidade,
                        'local_id' => $local_id
                    ]);
                    
                    $this->util->incrementaEstoque($i->produto->id, $i->quantidade, null, $local_id);
                } else {
                    \Log::warning('Produto não encontrado ou não gerencia estoque:', [
                        'produto_id' => $i->produto_id,
                        'produto_exists' => $i->produto ? 'sim' : 'não'
                    ]);
                }
            }
            
            $item->itens()->delete();
            $item->delete();

            __createLog(request()->empresa_id, 'PDV Troca', 'excluir', $descricaoLog);

            session()->flash("flash_success", "Removido com sucesso!");
        } catch (\Exception $e) {
            \Log::error('Erro ao excluir troca:', [
                'troca_id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            __createLog(request()->empresa_id, 'PDV Troca', 'erro', $e->getMessage());
            session()->flash("flash_error", "Algo deu errado: " . $e->getMessage());
        }
        return redirect()->back();
    }

    public function imprimir($id){
        \Log::info('Tentando imprimir troca ID: ' . $id);
        
        try {
            $item = Troca::with(['nfce', 'itens.produto'])->findOrFail($id);
            \Log::info('Troca encontrada: ' . $item->id);

            __validaObjetoEmpresa($item);
            $config = Empresa::where('id', $item->empresa_id)
            ->first();

            \Log::info('Configuração da empresa encontrada: ' . ($config ? $config->id : 'null'));
            \Log::info('NFCe da troca: ' . ($item->nfce ? $item->nfce->id : 'null'));
            \Log::info('Itens da troca: ' . ($item->itens ? $item->itens->count() : '0'));

            \Log::info('Gerando PDF da troca...');
            
            // Verificar se a NFCe existe
            if (!$item->nfce) {
                throw new \Exception('NFCe não encontrada para esta troca');
            }
            
            // Verificar se a configuração existe
            if (!$config) {
                throw new \Exception('Configuração da empresa não encontrada');
            }
            
            // Verificar se a NFCe tem itens
            if (!$item->nfce->itens || $item->nfce->itens->count() == 0) {
                \Log::warning('NFCe não tem itens, adicionando item vazio para evitar erro');
                // Criar um item vazio para evitar erro no PDF
                $item->nfce->itens = collect([(object)[
                    'produto' => (object)['id' => 0, 'nome' => 'Item removido', 'unidade' => 'UN'],
                    'quantidade' => 0,
                    'valor_unitario' => 0,
                    'sub_total' => 0
                ]]);
            }
            
            // Verificar se a troca tem itens
            if (!$item->itens || $item->itens->count() == 0) {
                \Log::warning('Troca não tem itens, adicionando item vazio para evitar erro');
                // Criar um item vazio para evitar erro no PDF
                $item->itens = collect([(object)[
                    'produto' => (object)['id' => 0, 'nome' => 'Item removido', 'unidade' => 'UN'],
                    'quantidade' => 0,
                    'valor_unitario' => 0
                ]]);
            }
            
            \Log::info('Parâmetros para CupomNaoFiscal:');
            \Log::info('- NFCe ID: ' . $item->nfce->id);
            \Log::info('- Config ID: ' . $config->id);
            \Log::info('- isPreVenda: 0');
            \Log::info('- Troca ID: ' . $item->id);
            \Log::info('- NFCe itens count: ' . $item->nfce->itens->count());
            \Log::info('- Troca itens count: ' . $item->itens->count());
            
            try {
                $service = new CupomNaoFiscal();
                $pdf = $service->render($item->nfce, $config, 0, $item);
                \Log::info('CupomNaoFiscal renderizado com sucesso');
            } catch (\Exception $e) {
                \Log::error('Erro ao instanciar CupomNaoFiscal: ' . $e->getMessage());
                \Log::error('Stack trace: ' . $e->getTraceAsString());
                throw $e;
            }
            
            return response($pdf)
            ->header('Content-Type', 'application/pdf');
        } catch (\Exception $e) {
            \Log::error('Erro ao imprimir troca: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return response('Erro ao gerar PDF: ' . $e->getMessage(), 500);
        }
    }

}
