<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Nfe;
use App\Models\Nfce;
use App\Models\ItemNfe;
use App\Models\ItemNfce;
use App\Models\FaturaNfe;
use App\Models\FaturaNfce;
use App\Models\Cliente;
use App\Models\ContaReceber;
use App\Models\CreditoCliente;
use App\Models\Caixa;
use App\Models\SangriaCaixa;
use App\Utils\EstoqueUtil;
use App\Models\Empresa;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Models\Devolucao;
use App\Models\ItemDevolucao;
use Dompdf\Dompdf;

class DevolucaoVendaController extends Controller
{
    protected $util;

    public function __construct(EstoqueUtil $util)
    {
        $this->util = $util;
        
        $this->middleware('permission:devolucao_venda_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:devolucao_venda_edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:devolucao_venda_view', ['only' => ['show', 'index']]);
        $this->middleware('permission:devolucao_venda_delete', ['only' => ['destroy']]);
    }

    /**
     * Lista todas as vendas disponíveis para devolução/troca
     */
    public function index(Request $request)
    {
        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $cliente_id = $request->get('cliente_id');
        $numero = $request->get('numero');

        // Obter empresa_id com fallback seguro
        $empresa_id = $this->obterEmpresaIdSeguro($request);
        
        // Em último caso, usar 1 como valor fixo (apenas para evitar o erro)
        if (empty($empresa_id)) {
            $empresa_id = 1; // Valor fixo como último recurso
            Log::warning('DevolucaoVendaController: usando empresa_id fixo como último recurso = ' . $empresa_id);
            
            // Adicionar um alerta na sessão para informar que estamos usando um valor padrão
            session()->flash("flash_warning", "Usando empresa padrão ID=1. Por favor, verifique suas configurações.");
        }
        
        // Verificar se a empresa existe
        try {
            $empresaExiste = Empresa::where('id', $empresa_id)->exists();
            if (!$empresaExiste) {
                Log::error('DevolucaoVendaController: empresa_id = ' . $empresa_id . ' não encontrado no banco de dados.');
                session()->flash("flash_error", "Empresa ID=" . $empresa_id . " não encontrada. Por favor, contate o administrador.");
                return view('devolucao_venda.index', [
                    'vendas' => [],
                    'debug_message' => "Erro: Empresa ID=" . $empresa_id . " não encontrada no banco de dados."
                ]);
            }
        } catch (\Exception $e) {
            Log::error('DevolucaoVendaController: erro ao verificar empresa: ' . $e->getMessage());
        }
        
        \Illuminate\Support\Facades\Log::debug('DevolucaoVendaController: empresa_id = ' . $empresa_id);
        
        // Contar NFCes sem filtro
        $total_nfces = Nfce::where('empresa_id', $empresa_id)->count();
        
        // Contar NFes sem filtro
        $total_nfes = Nfe::where('empresa_id', $empresa_id)->count();

        // Buscar as NFCe
        $nfces = Nfce::where('empresa_id', $empresa_id)
            ->when(!empty($start_date), function ($query) use ($start_date) {
                return $query->whereDate('created_at', '>=', $start_date);
            })
            ->when(!empty($end_date), function ($query) use ($end_date) {
                return $query->whereDate('created_at', '<=', $end_date);
            })
            ->when(!empty($cliente_id), function ($query) use ($cliente_id) {
                return $query->where('cliente_id', $cliente_id);
            })
            ->when(!empty($numero), function ($query) use ($numero) {
                return $query->where(function($q) use ($numero) {
                    $q->where('numero', 'like', "%{$numero}%")
                      ->orWhere('numero_sequencial', 'like', "%{$numero}%");
                });
            })
            ->with(['cliente', 'itens'])
            ->orderBy('created_at', 'desc')
            ->get();

        // Buscar as NFe - Apenas notas de saída conforme solicitado
        // tpNF == 1 indica nota de saída, tpNF == 0 indica nota de entrada
        $nfes = Nfe::where('empresa_id', $empresa_id)
            ->where('orcamento', 0) // Não incluir orçamentos
            ->where(function($query) {
                $query->where('tpNF', 1) // Apenas notas de saída (tpNF == 1)
                      ->orWhereNull('tpNF'); // Ou quando o campo não está definido (consideradas como saída)
            })
            ->when(!empty($start_date), function ($query) use ($start_date) {
                return $query->whereDate('created_at', '>=', $start_date);
            })
            ->when(!empty($end_date), function ($query) use ($end_date) {
                return $query->whereDate('created_at', '<=', $end_date);
            })
            ->when(!empty($cliente_id), function ($query) use ($cliente_id) {
                return $query->where('cliente_id', $cliente_id);
            })
            ->when(!empty($numero), function ($query) use ($numero) {
                return $query->where(function($q) use ($numero) {
                    $q->where('numero', 'like', "%{$numero}%")
                      ->orWhere('numero_sequencial', 'like', "%{$numero}%");
                });
            })
            ->with(['cliente', 'itens'])
            ->orderBy('created_at', 'desc')
            ->get();

        // Debug - verificar o resultado das consultas
        $count_nfces = count($nfces);
        $count_nfes = count($nfes);
        
        // Debug temporário para diagnóstico
        if ($count_nfces == 0 && $count_nfes == 0) {
            // Verificar se existem documentos na base
            if ($total_nfces == 0 && $total_nfes == 0) {
                return view('devolucao_venda.index', [
                    'vendas' => [],
                    'debug_message' => "Nenhuma nota encontrada para a empresa ID: $empresa_id. Total NFCes: $total_nfces. Total NFes: $total_nfes."
                ]);
            } else {
                return view('devolucao_venda.index', [
                    'vendas' => [],
                    'debug_message' => "Existem notas na base (NFCes: $total_nfces, NFes: $total_nfes), mas nenhuma corresponde aos filtros aplicados. Empresa ID: $empresa_id."
                ]);
            }
        }

        // Preparando dados para a view
        $vendas = [];
        
        // NFCe sempre são de saída, então não é necessário filtrar por tipo
        foreach ($nfces as $nfce) {
            // Verificar status de devolução
            $statusDevolucao = $this->verificarStatusDevolucao('nfce', $nfce->id);
            
            $vendas[] = [
                'id' => $nfce->id,
                'tipo' => 'nfce',
                'numero' => $nfce->numero,
                'numero_sequencial' => $nfce->numero_sequencial,
                'data' => $nfce->created_at,
                'cliente' => $nfce->cliente ? $nfce->cliente->razao_social : 'Consumidor Final',
                'total' => $nfce->total,
                'itens_count' => count($nfce->itens),
                'status_devolucao' => $statusDevolucao
            ];
        }

        foreach ($nfes as $nfe) {
            // Verificar status de devolução
            $statusDevolucao = $this->verificarStatusDevolucao('nfe', $nfe->id);
            
            $vendas[] = [
                'id' => $nfe->id,
                'tipo' => 'nfe',
                'numero' => $nfe->numero,
                'numero_sequencial' => $nfe->numero_sequencial,
                'data' => $nfe->created_at,
                'cliente' => $nfe->cliente ? $nfe->cliente->razao_social : 'Consumidor Final',
                'total' => $nfe->total,
                'itens_count' => count($nfe->itens),
                'status_devolucao' => $statusDevolucao
            ];
        }

        // Ordenar vendas por data decrescente
        usort($vendas, function($a, $b) {
            return $b['data'] <=> $a['data'];
        });
        
        // Debug message para a view
        $debug_message = "Empresa ID: $empresa_id. Total NFCes: $total_nfces, Filtradas: $count_nfces. Total NFes: $total_nfes, Filtradas: $count_nfes.";

        return view('devolucao_venda.index', compact('vendas', 'debug_message', 'empresa_id'));
    }

    /**
     * Exibe o formulário para realizar uma devolução
     */
    public function devolucao($tipo, $id)
    {
        if (!__isCaixaAberto()) {
            session()->flash("flash_warning", "Abrir caixa antes de continuar!");
            return redirect()->route('caixa.create');
        }

        if ($tipo == 'nfce') {
            $venda = Nfce::with(['itens.produto', 'cliente', 'fatura'])->findOrFail($id);
            
            // Carregar informações de devolução para cada item
            foreach($venda->itens as $key => $item) {
                // Obter a quantidade original quando o item foi vendido
                $quantidadeOriginal = $item->quantidade + $item->quantidade_devolvida;
                
                // Obter a quantidade já devolvida de todas as devoluções anteriores
                $quantidadeJaDevolvida = $item->quantidade_devolvida;
                
                // Calcular a quantidade disponível para devolução
                $quantidadeDisponivel = $quantidadeOriginal - $quantidadeJaDevolvida;
                
                // Apenas para exibição na interface
                $item->quantidade_original = $quantidadeOriginal;
                $item->quantidade_disponivel = $quantidadeDisponivel;
                
                // Verificações para a interface
                $item->ja_devolvido = $quantidadeJaDevolvida > 0;
                $item->totalmente_devolvido = $quantidadeDisponivel <= 0;
                
                // Se totalmente devolvido, remover dos itens disponíveis
                if ($item->totalmente_devolvido) {
                    $venda->itens->forget($key);
                }
            }
        } else { // nfe
            $venda = Nfe::with(['itens.produto', 'cliente', 'fatura'])->findOrFail($id);
            
            // Carregar informações de devolução para cada item
            foreach($venda->itens as $key => $item) {
                // Obter a quantidade original quando o item foi vendido
                $quantidadeOriginal = $item->quantidade + $item->quantidade_devolvida;
                
                // Obter a quantidade já devolvida de todas as devoluções anteriores
                $quantidadeJaDevolvida = $item->quantidade_devolvida;
                
                // Calcular a quantidade disponível para devolução
                $quantidadeDisponivel = $quantidadeOriginal - $quantidadeJaDevolvida;
                
                // Apenas para exibição na interface
                $item->quantidade_original = $quantidadeOriginal;
                $item->quantidade_disponivel = $quantidadeDisponivel;
                
                // Verificações para a interface
                $item->ja_devolvido = $quantidadeJaDevolvida > 0;
                $item->totalmente_devolvido = $quantidadeDisponivel <= 0;
                
                // Se totalmente devolvido, remover dos itens disponíveis
                if ($item->totalmente_devolvido) {
                    $venda->itens->forget($key);
                }
            }
        }

        // Verificar se há crediário nas faturas
        $temCrediario = false;
        $valorCrediario = 0;
        $contasReceber = collect();
        
        if ($venda->fatura && $venda->fatura->count() > 0) {
            $faturasCrediario = $venda->fatura->where('tipo_pagamento', '06');
            if ($faturasCrediario->count() > 0) {
                $temCrediario = true;
                $valorCrediario = $faturasCrediario->sum('valor');
                
                // Buscar contas a receber vinculadas ao crediário
                if ($tipo == 'nfce') {
                    $contasReceber = ContaReceber::where('nfce_id', $venda->id)
                        ->where('tipo_pagamento', '06')
                        ->where('status', '!=', ContaReceber::STATUS_RECEBIDO)
                        ->orderBy('data_vencimento', 'asc')
                        ->get();
                } else {
                    $contasReceber = ContaReceber::where('nfe_id', $venda->id)
                        ->where('tipo_pagamento', '06')
                        ->where('status', '!=', ContaReceber::STATUS_RECEBIDO)
                        ->orderBy('data_vencimento', 'asc')
                        ->get();
                }
            }
        }

        // Limitar os tipos de pagamento apenas para os solicitados
        $todosTiposPagamento = Nfce::tiposPagamento();
        $tiposPagamento = [
            '01' => $todosTiposPagamento['01'], // Dinheiro
            '17' => $todosTiposPagamento['17'], // Pagamento Instantâneo (PIX)
            '03' => $todosTiposPagamento['03'], // Cartão de Crédito
            '04' => $todosTiposPagamento['04'], // Cartão de Débito
            '00' => 'Vale Crédito'              // Vale Crédito
        ];

        // Buscar caixas disponíveis para a empresa do usuário
        $caixas = \App\Models\Caixa::where('empresa_id', Auth::user()->empresa_id)->get();
        $caixaSelecionado = $caixas->first() ? $caixas->first()->id : null;

        // Buscar todas as empresas e definir a selecionada
        $empresas = \App\Models\Empresa::all();
        $empresaSelecionada = Auth::user()->empresa_id ?? ($empresas->first() ? $empresas->first()->id : null);

        return view('devolucao_venda.devolucao', compact('venda', 'tipo', 'tiposPagamento', 'temCrediario', 'valorCrediario', 'contasReceber', 'caixas', 'caixaSelecionado', 'empresas', 'empresaSelecionada'));
    }

    /**
     * Imprime o comprovante de devolução ou troca
     */
    public function imprimirComprovante($id, $tipo = 'devolucao')
    {
        $devolucao = Devolucao::with(['itens.produto', 'cliente'])->findOrFail($id);
        $empresa = Empresa::findOrFail($devolucao->empresa_id);

        $p = view('devolucao_venda.comprovante', compact('devolucao', 'empresa', 'tipo'));

        $domPdf = new Dompdf(["enable_remote" => true]);
        $domPdf->loadHtml($p);
        $pdf = ob_get_clean();
        // Tamanho reduzido para caber em uma página
        $domPdf->set_paper(array(0,0,214,340));
        $domPdf->render();
        
        // Configurar para abrir em nova aba sem forçar download (Attachment = false)
        $filename = "Comprovante de " . ($tipo == 'troca' ? 'Troca' : 'Devolução') . " #$devolucao->id.pdf";
        $domPdf->stream($filename, [
            "Attachment" => false,
            "compress" => true
        ]);
        exit();
    }

    /**
     * Processa a devolução
     */
    public function storeDevolucao(Request $request)
    {
        $tipo = $request->tipo_venda;
        $id = $request->venda_id;
        // Buscar a venda correspondente
        if ($tipo == 'nfce') {
            $venda = Nfce::with(['itens.produto', 'cliente', 'fatura'])->findOrFail($id);
        } else {
            $venda = Nfe::with(['itens.produto', 'cliente', 'fatura'])->findOrFail($id);
        }
        $itens_ids = $request->itens_ids;
        $quantidades = $request->quantidades;
        $forma_pagamento = $request->forma_pagamento;
        $cliente_id = $request->cliente_id;
        $valorTotal = $this->brToFloat($request->valor_devolucao);

        $empresa_id = $this->obterEmpresaIdSeguro($request, $id, $tipo);
        if (empty($empresa_id)) {
            session()->flash("flash_error", "Não foi possível determinar a empresa para esta operação. Por favor, contate o administrador.");
            return redirect()->back();
        }

        DB::beginTransaction();
        try {
            // Validação correta: impedir devolução de item já devolvido (parcial ou total)
            foreach ($itens_ids as $key => $item_id) {
                $quantidadeSolicitada = __convert_value_bd($quantidades[$key]);
                if ($tipo == 'nfce') {
                    $item = \App\Models\ItemNfce::findOrFail($item_id);
                    $quantidadeOriginal = $item->quantidade + $item->quantidade_devolvida;
                    $quantidadeJaDevolvida = \App\Models\ItemDevolucao::where('tipo_item', 'item_nfce')
                        ->where('item_id', $item_id)
                        ->sum('quantidade');
                    $quantidadeDisponivel = $quantidadeOriginal - $quantidadeJaDevolvida;

                    if ($quantidadeDisponivel <= 0) {
                        throw new \Exception("O item '{$item->produto->nome}' já foi totalmente devolvido e não pode ser devolvido novamente.");
                    }
                    if ($quantidadeSolicitada > $quantidadeDisponivel) {
                        throw new \Exception("Não é possível devolver mais do que a quantidade disponível do item: {$item->produto->nome}");
                    }
                } else {
                    $item = \App\Models\ItemNfe::findOrFail($item_id);
                    $quantidadeOriginal = $item->quantidade + $item->quantidade_devolvida;
                    $quantidadeJaDevolvida = \App\Models\ItemDevolucao::where('tipo_item', 'item_nfe')
                        ->where('item_id', $item_id)
                        ->sum('quantidade');
                    $quantidadeDisponivel = $quantidadeOriginal - $quantidadeJaDevolvida;

                    if ($quantidadeDisponivel <= 0) {
                        throw new \Exception("O item '{$item->produto->nome}' já foi totalmente devolvido e não pode ser devolvido novamente.");
                    }
                    if ($quantidadeSolicitada > $quantidadeDisponivel) {
                        throw new \Exception("Não é possível devolver mais do que a quantidade disponível do item: {$item->produto->nome}");
                    }
                }
            }

            // ... processar itens, estoque, etc ...

            // Processar pagamento único
            if ($forma_pagamento == 'credito_loja') {
                \Log::info('Devolução: Crédito em loja para cliente_id=' . $cliente_id . ', valor=' . $valorTotal);
                $cliente = \App\Models\Cliente::find($cliente_id);
                if ($cliente) {
                    $cliente->valor_credito += $valorTotal;
                    $cliente->save();
                    __createLog($cliente->empresa_id ?? null, 'Crédito Cliente', 'adicionar', "Adicionado R$ " . __moeda($valorTotal) . " de crédito para o cliente {$cliente->razao_social} por devolução");
                    \Log::info('Devolução: Crédito lançado para cliente_id=' . $cliente->id . ', novo valor_credito=' . $cliente->valor_credito);
                } else {
                    \Log::error('Devolução: Cliente não encontrado para crédito em loja, id=' . $cliente_id);
                }
            }

            // ... restante do método (salvar devolução, itens, commit, etc) ...

            // Salvar a devolução no banco
            $devolucao = Devolucao::create([
                'empresa_id' => $empresa_id,
                'usuario_id' => auth()->user()->id,
                'venda_id' => $id,
                'tipo_venda' => $tipo,
                'total' => $valorTotal,
                'tipo_pagamento' => $forma_pagamento,
                'cliente_id' => $forma_pagamento == '00' ? $cliente_id : ($tipo == 'nfce' ? $venda->cliente_id : $venda->cliente_id),
                'observacao' => $request->observacao ?? 'Devolução de venda',
                'data_devolucao' => now(),
                'caixa_id' => __isCaixaAberto() ? __isCaixaAberto()->id : null
            ]);
            
            $devolucao_id = $devolucao->id;
            
            // Chamar o processamento financeiro da devolução (dinheiro, pix, cartão, etc)
            if (in_array($forma_pagamento, ['01', '17', '03', '04', '00', '06'])) {
                $this->processarPagamentoDevolucao($forma_pagamento, $valorTotal, $cliente_id, $id, $tipo);
            }
            
            // Salvar os itens da devolução
            foreach ($itens_ids as $key => $item_id) {
                $quantidade = __convert_value_bd($quantidades[$key]);
                
                if ($tipo == 'nfce') {
                    $item = ItemNfce::findOrFail($item_id);
                    
                    ItemDevolucao::create([
                        'devolucao_id' => $devolucao->id,
                        'item_id' => $item_id,
                        'tipo_item' => 'item_nfce',
                        'quantidade' => $quantidade,
                        'valor_unitario' => $item->valor_unitario,
                        'produto_id' => $item->produto_id,
                        'variacao_id' => $item->variacao_id
                    ]);
                    if ($item->produto && $item->produto->gerenciar_estoque) {
                        $this->util->incrementaEstoque(
                            $item->produto_id,
                            $quantidade,
                            $item->variacao_id ?? null,
                            $venda->local_id
                        );
                    }
                } else {
                    $item = ItemNfe::findOrFail($item_id);
                    
                    ItemDevolucao::create([
                        'devolucao_id' => $devolucao->id,
                        'item_id' => $item_id,
                        'tipo_item' => 'item_nfe',
                        'quantidade' => $quantidade,
                        'valor_unitario' => $item->valor_unitario,
                        'produto_id' => $item->produto_id,
                        'variacao_id' => null
                    ]);
                    if ($item->produto && $item->produto->gerenciar_estoque) {
                        $this->util->incrementaEstoque(
                            $item->produto_id,
                            $quantidade,
                            null,
                            $venda->local_id
                        );
                    }
                }
            }
            
            // Criar log da operação
            __createLog(
                $empresa_id, 
                'Devolução de Venda', 
                'cadastrar', 
                "Devolução de {$tipo} #{$venda->numero_sequencial} - R$ " . __moeda($valorTotal)
            );
            
            DB::commit();
            
            // Perguntar se o usuário deseja imprimir o comprovante
            session()->flash("flash_success", "Devolução realizada com sucesso!");
            session()->flash("imprimir_comprovante", $devolucao_id);
            session()->flash("tipo_comprovante", "devolucao");
            
            return redirect()->route('devolucao_venda.index');
            
        } catch (\Exception $e) {
            DB::rollback();
            __createLog($empresa_id, 'Devolução de Venda', 'erro', $e->getMessage());
            // Exibir erro com SweetAlert
            session()->flash('sweet_error', $e->getMessage());
            return redirect()->back();
        }
    }

    /**
     * Exibe o formulário para realizar uma troca
     */
    public function troca($tipo, $id)
    {
        if (!__isCaixaAberto()) {
            session()->flash("flash_warning", "Abrir caixa antes de continuar!");
            return redirect()->route('caixa.create');
        }

        if ($tipo == 'nfce') {
            $venda = Nfce::with(['itens.produto', 'cliente', 'fatura'])->findOrFail($id);
            
            // Carregar informações de devolução para cada item
            foreach($venda->itens as $key => $item) {
                // Obter a quantidade original quando o item foi vendido
                $quantidadeOriginal = $item->quantidade + $item->quantidade_devolvida;
                
                // Obter a quantidade já devolvida de todas as devoluções anteriores
                $quantidadeJaDevolvida = $item->quantidade_devolvida;
                
                // Calcular a quantidade disponível para devolução
                $quantidadeDisponivel = $quantidadeOriginal - $quantidadeJaDevolvida;
                
                // Apenas para exibição na interface
                $item->quantidade_original = $quantidadeOriginal;
                $item->quantidade_disponivel = $quantidadeDisponivel;
                
                // Verificações para a interface
                $item->ja_devolvido = $quantidadeJaDevolvida > 0;
                $item->totalmente_devolvido = $quantidadeDisponivel <= 0;
                
                // Se totalmente devolvido, remover dos itens disponíveis
                if ($item->totalmente_devolvido) {
                    $venda->itens->forget($key);
                }
            }
        } else { // nfe
            $venda = Nfe::with(['itens.produto', 'cliente', 'fatura'])->findOrFail($id);
            
            // Carregar informações de devolução para cada item
            foreach($venda->itens as $key => $item) {
                // Obter a quantidade original quando o item foi vendido
                $quantidadeOriginal = $item->quantidade + $item->quantidade_devolvida;
                
                // Obter a quantidade já devolvida de todas as devoluções anteriores
                $quantidadeJaDevolvida = $item->quantidade_devolvida;
                
                // Calcular a quantidade disponível para devolução
                $quantidadeDisponivel = $quantidadeOriginal - $quantidadeJaDevolvida;
                
                // Apenas para exibição na interface
                $item->quantidade_original = $quantidadeOriginal;
                $item->quantidade_disponivel = $quantidadeDisponivel;
                
                // Verificações para a interface
                $item->ja_devolvido = $quantidadeJaDevolvida > 0;
                $item->totalmente_devolvido = $quantidadeDisponivel <= 0;
                
                // Se totalmente devolvido, remover dos itens disponíveis
                if ($item->totalmente_devolvido) {
                    $venda->itens->forget($key);
                }
            }
        }

        // Verificar se há crediário nas faturas
        $temCrediario = false;
        $valorCrediario = 0;
        $contasReceber = collect();
        
        if ($venda->fatura && $venda->fatura->count() > 0) {
            $faturasCrediario = $venda->fatura->where('tipo_pagamento', '06');
            if ($faturasCrediario->count() > 0) {
                $temCrediario = true;
                $valorCrediario = $faturasCrediario->sum('valor');
                
                // Buscar contas a receber vinculadas ao crediário
                if ($tipo == 'nfce') {
                    $contasReceber = ContaReceber::where('nfce_id', $venda->id)
                        ->where('tipo_pagamento', '06')
                        ->where('status', '!=', ContaReceber::STATUS_RECEBIDO)
                        ->orderBy('data_vencimento', 'asc')
                        ->get();
                } else {
                    $contasReceber = ContaReceber::where('nfe_id', $venda->id)
                        ->where('tipo_pagamento', '06')
                        ->where('status', '!=', ContaReceber::STATUS_RECEBIDO)
                        ->orderBy('data_vencimento', 'asc')
                        ->get();
                }
            }
        }

        // Obter todos os tipos de pagamento
        $todosTiposPagamento = Nfce::tiposPagamento();
        
        // Adicionar Vale Crédito
        $todosTiposPagamento['00'] = 'Vale Crédito';
        
        // Para a tela de troca, vamos definir uma variável para controlar o script JS
        // que irá filtrar os métodos de pagamento com base no valor da diferença
        $tiposPagamentoCliente = [
            '01' => $todosTiposPagamento['01'], // Dinheiro
            '17' => $todosTiposPagamento['17'], // Pagamento Instantâneo (PIX)
            '03' => $todosTiposPagamento['03'], // Cartão de Crédito
            '04' => $todosTiposPagamento['04'], // Cartão de Débito
            '00' => 'Vale Crédito'              // Vale Crédito
        ];
        
        // Passar todos os tipos de pagamento e os específicos para devolução ao cliente
        return view('devolucao_venda.troca', compact('venda', 'tipo', 'todosTiposPagamento', 'tiposPagamentoCliente', 'temCrediario', 'valorCrediario', 'contasReceber'));
    }

    /**
     * Processa a troca
     * 
     * Processa a troca de produtos, calculando diferenças de valores:
     * 
     * 1. Se o valor da devolução for maior que o valor da troca (diferença positiva):
     *    - O cliente recebe a diferença conforme o método de pagamento escolhido
     * 
     * 2. Se o valor da troca for maior que o valor da devolução (diferença negativa):
     *    - O cliente paga a diferença conforme o método de pagamento escolhido
     *    - Se o pagamento for em dinheiro e já existir uma fatura em dinheiro, a diferença
     *      é somada à fatura existente
     *    - Caso contrário, uma nova fatura é criada com o valor da diferença
     */
    public function storeTroca(Request $request)
    {
        $tipo = $request->tipo_venda;
        $id = $request->venda_id;
        $itens_ids = $request->itens_ids;
        $quantidades = $request->quantidades;
        $produtos_troca = $request->produtos_troca;
        $quantidades_troca = $request->quantidades_troca;
        $tipo_pagamento = $request->tipo_pagamento;
        $cliente_id = $request->cliente_id;

        // Obter empresa_id com fallback seguro
        $empresa_id = $this->obterEmpresaIdSeguro($request, $id, $tipo);
        
        // Se ainda não tiver empresa_id, impossível continuar
        if (empty($empresa_id)) {
            session()->flash("flash_error", "Não foi possível determinar a empresa para esta operação. Por favor, contate o administrador.");
            return redirect()->back();
        }

        // Verificar se foi selecionado um cliente quando o pagamento for crédito em loja
        if ($tipo_pagamento == '00') {
            if (empty($cliente_id) || $cliente_id == "0" || $cliente_id == "") {
                session()->flash("flash_error", "Para gerar crédito em loja é necessário selecionar um cliente.");
                return redirect()->back()->withInput();
            }
            
            // Verificar se o cliente existe
            $cliente = Cliente::find($cliente_id);
            if (!$cliente) {
                session()->flash("flash_error", "Cliente selecionado não encontrado. Por favor, selecione outro cliente.");
                return redirect()->back()->withInput();
            }
        }

        // Inicializa variáveis
        $valorDevolucao = 0;
        $valorTroca = 0;
        $devolucao_id = null;

        DB::beginTransaction();

        try {
            // Verificar se existem produtos para troca
            if (empty($produtos_troca) || count($produtos_troca) == 0) {
                throw new \Exception("Nenhum produto selecionado para troca!");
            }
            
            // Verificar se existe caixa aberto
            $caixa = __isCaixaAberto();
            if (!$caixa) {
                throw new \Exception("Não há caixa aberto para realizar a troca!");
            }
            
            if ($tipo == 'nfce') {
                $venda = Nfce::with(['itens.produto', 'cliente'])->findOrFail($id);
                
                // Processar itens devolvidos
                foreach ($itens_ids as $key => $item_id) {
                    $item = ItemNfce::with('produto')->findOrFail($item_id);
                    $quantidade = __convert_value_bd($quantidades[$key]);
                    
                    // Verificar se quantidade a ser devolvida é válida
                    if ($quantidade > $item->quantidade) {
                        throw new \Exception("Quantidade a devolver não pode ser maior que a quantidade vendida!");
                    }
                    
                    // Verificar se o item já foi devolvido (total ou parcialmente)
                    $quantidadeJaDevolvida = $this->getQuantidadeJaDevolvida('item_nfce', $item_id);
                    
                    // Não precisamos verificar a quantidade disponível, pois o campo quantidade já representa a quantidade atual
                    
                    // Incrementar estoque
                    if ($item->produto && $item->produto->gerenciar_estoque) {
                        $this->util->incrementaEstoque(
                            $item->produto_id, 
                            $quantidade, 
                            $item->variacao_id ?? null,
                            $venda->local_id
                        );
                    }
                    
                    // Calcular valor proporcional
                    $valorItem = ($item->valor_unitario * $quantidade);
                    $valorDevolucao += $valorItem;
                    
                    // Processar atualização do item, vendas e faturas - para item devolvido/trocado
                    $this->processarAtualizacaoItemDevolvido('nfce', $item_id, $quantidade, $valorItem, $tipo_pagamento);
                }
                
                // Processar produtos da troca
                if ($produtos_troca) {
                    foreach ($produtos_troca as $key => $produto_id) {
                        $produto = \App\Models\Produto::findOrFail($produto_id);
                        $quantidade = __convert_value_bd($quantidades_troca[$key]);
                        
                        // Reduzir estoque
                        if ($produto->gerenciar_estoque) {
                            $this->util->reduzEstoque(
                                $produto->id, 
                                $quantidade, 
                                null,
                                $venda->local_id
                            );
                        }
                        
                        // Calcular valor dos novos produtos
                        $valorItem = ($produto->valor_unitario * $quantidade);
                        $valorTroca += $valorItem;
                        
                        // Para cada produto de troca, criar um item substituto para um dos itens devolvidos
                        if (isset($itens_ids[$key])) {
                            $itemOriginalId = $itens_ids[$key];
                            $quantidadeOriginal = __convert_value_bd($quantidades[$key]);
                            
                            // Criar um novo item na venda como troca do item original
                            $this->processarItemTrocado('nfce', $itemOriginalId, $quantidadeOriginal, $produto_id, $quantidade, $produto->valor_unitario, $venda);
                        }
                    }
                }
                
            } else { // nfe
                $venda = Nfe::with(['itens.produto', 'cliente'])->findOrFail($id);
                
                // Processar itens devolvidos
                foreach ($itens_ids as $key => $item_id) {
                    $item = ItemNfe::with('produto')->findOrFail($item_id);
                    $quantidade = __convert_value_bd($quantidades[$key]);
                    
                    // Verificar se quantidade a ser devolvida é válida
                    if ($quantidade > $item->quantidade) {
                        throw new \Exception("Quantidade a devolver não pode ser maior que a quantidade vendida!");
                    }
                    
                    // Verificar se o item já foi devolvido (total ou parcialmente)
                    $quantidadeJaDevolvida = $this->getQuantidadeJaDevolvida('item_nfe', $item_id);
                    
                    // Não precisamos verificar a quantidade disponível, pois o campo quantidade já representa a quantidade atual
                    
                    // Incrementar estoque
                    if ($item->produto && $item->produto->gerenciar_estoque) {
                        $this->util->incrementaEstoque(
                            $item->produto_id, 
                            $quantidade, 
                            null,
                            $venda->local_id
                        );
                    }
                    
                    // Calcular valor proporcional
                    $valorItem = ($item->valor_unitario * $quantidade);
                    $valorDevolucao += $valorItem;
                    
                    // Processar atualização do item, vendas e faturas - para item devolvido/trocado
                    $this->processarAtualizacaoItemDevolvido('nfe', $item_id, $quantidade, $valorItem, $tipo_pagamento);
                }
                
                // Processar produtos da troca
                if ($produtos_troca) {
                    foreach ($produtos_troca as $key => $produto_id) {
                        $produto = \App\Models\Produto::findOrFail($produto_id);
                        $quantidade = __convert_value_bd($quantidades_troca[$key]);
                        
                        // Reduzir estoque
                        if ($produto->gerenciar_estoque) {
                            $this->util->reduzEstoque(
                                $produto->id, 
                                $quantidade, 
                                null,
                                $venda->local_id
                            );
                        }
                        
                        // Calcular valor dos novos produtos
                        $valorItem = ($produto->valor_unitario * $quantidade);
                        $valorTroca += $valorItem;
                        
                        // Para cada produto de troca, criar um item substituto para um dos itens devolvidos
                        if (isset($itens_ids[$key])) {
                            $itemOriginalId = $itens_ids[$key];
                            $quantidadeOriginal = __convert_value_bd($quantidades[$key]);
                            
                            // Criar um novo item na venda como troca do item original
                            $this->processarItemTrocado('nfe', $itemOriginalId, $quantidadeOriginal, $produto_id, $quantidade, $produto->valor_unitario, $venda);
                        }
                    }
                }
            }
            
            // Processar a diferença de valores
            $diferenca = $valorDevolucao - $valorTroca;
            
            if ($diferenca > 0) {
                // Cliente recebe a diferença
                if ($tipo_pagamento == '00') {
                    // Se for crédito em loja, usar o cliente selecionado no formulário
                    $this->processarPagamentoDevolucao($tipo_pagamento, $diferenca, $cliente_id, $venda->id, $tipo);
                } else {
                    // Para outros tipos de pagamento, usar o cliente da venda (se houver)
                    $this->processarPagamentoDevolucao($tipo_pagamento, $diferenca, $venda->cliente_id, $venda->id, $tipo);
                }
            } else if ($diferenca < 0) {
                // Cliente paga a diferença (valor a maior na troca)
                $valorDiferenca = abs($diferenca);
                
                Log::info("Cliente precisa pagar diferença de R$ {$valorDiferenca} na troca ID {$venda->id}");
                
                if ($tipo == 'nfce') {
                    // Quando há diferença a ser paga pelo cliente, simplesmente registramos
                    // o valor total dos novos produtos, já que a venda original será "zerada"
                    // pela devolução e precisamos registrar o novo valor total
                    
                    // Criar nova fatura para registrar o valor total dos novos produtos
                    $fatura = new FaturaNfce();
                    $fatura->nfce_id = $venda->id;
                    $fatura->valor = $valorTroca; // Usar o valor total dos novos produtos
                    $fatura->tipo_pagamento = $tipo_pagamento;
                    $fatura->observacao = "Valor total dos novos produtos na troca em " . date('d/m/Y H:i:s');
                    $fatura->save();
                    
                    Log::info("Criada nova fatura NFCE ID {$fatura->id} com valor total dos novos produtos R$ {$valorTroca} (inclui diferença de R$ {$valorDiferenca}) e tipo_pagamento {$tipo_pagamento}");
                } else { // nfe
                    // Mesmo processo para NFE
                    
                    // Criar nova fatura para registrar o valor total dos novos produtos
                    $fatura = new FaturaNfe();
                    $fatura->nfe_id = $venda->id;
                    $fatura->valor = $valorTroca; // Usar o valor total dos novos produtos
                    $fatura->tipo_pagamento = $tipo_pagamento;
                    $fatura->observacao = "Valor total dos novos produtos na troca em " . date('d/m/Y H:i:s');
                    $fatura->save();
                    
                    Log::info("Criada nova fatura NFE ID {$fatura->id} com valor total R$ {$valorTotal} (inclui diferença de R$ {$valorDiferenca}) e tipo_pagamento {$tipo_pagamento}");
                }
            }
            
            // Salvar a troca no banco
            $devolucao = Devolucao::create([
                'empresa_id' => $empresa_id,
                'usuario_id' => auth()->user()->id,
                'venda_id' => $id,
                'tipo_venda' => $tipo,
                'valor_total' => $valorDevolucao,
                'tipo_pagamento' => $tipo_pagamento,
                'cliente_id' => $tipo_pagamento == '00' ? $cliente_id : ($tipo == 'nfce' ? $venda->cliente_id : $venda->cliente_id),
                'observacao' => $request->observacao ?? 'Troca de produtos',
                'data_devolucao' => now(),
                'is_troca' => true,
                'valor_diferenca' => $diferenca,
                'caixa_id' => __isCaixaAberto() ? __isCaixaAberto()->id : null
            ]);
            
            $devolucao_id = $devolucao->id;
            
            // Salvar os itens da troca
            foreach ($itens_ids as $key => $item_id) {
                $quantidade = __convert_value_bd($quantidades[$key]);
                
                if ($tipo == 'nfce') {
                    $item = ItemNfce::findOrFail($item_id);
                    
                    ItemDevolucao::create([
                        'devolucao_id' => $devolucao->id,
                        'item_id' => $item_id,
                        'tipo_item' => 'item_nfce',
                        'quantidade' => $quantidade,
                        'valor_unitario' => $item->valor_unitario,
                        'produto_id' => $item->produto_id,
                        'variacao_id' => $item->variacao_id
                    ]);
                    if ($item->produto && $item->produto->gerenciar_estoque) {
                        $this->util->incrementaEstoque(
                            $item->produto_id,
                            $quantidade,
                            $item->variacao_id ?? null,
                            $venda->local_id
                        );
                    }
                } else {
                    $item = ItemNfe::findOrFail($item_id);
                    
                    ItemDevolucao::create([
                        'devolucao_id' => $devolucao->id,
                        'item_id' => $item_id,
                        'tipo_item' => 'item_nfe',
                        'quantidade' => $quantidade,
                        'valor_unitario' => $item->valor_unitario,
                        'produto_id' => $item->produto_id,
                        'variacao_id' => null
                    ]);
                    if ($item->produto && $item->produto->gerenciar_estoque) {
                        $this->util->incrementaEstoque(
                            $item->produto_id,
                            $quantidade,
                            null,
                            $venda->local_id
                        );
                    }
                }
            }
            
            // Criar log da operação
            __createLog(
                $empresa_id, 
                'Troca de Produtos', 
                'cadastrar', 
                "Troca de {$tipo} #{$venda->numero_sequencial} - Devolvidos: R$ " . __moeda($valorDevolucao) . " - Novos: R$ " . __moeda($valorTroca)
            );
            
            DB::commit();
            
            // Perguntar se o usuário deseja imprimir o comprovante
            session()->flash("flash_success", "Troca realizada com sucesso!");
            session()->flash("imprimir_comprovante", $devolucao_id);
            session()->flash("tipo_comprovante", "troca");
            
            return redirect()->route('devolucao_venda.index');
            
        } catch (\Exception $e) {
            DB::rollback();
            __createLog($empresa_id, 'Troca de Produtos', 'erro', $e->getMessage());
            session()->flash("flash_error", "Erro ao processar troca: " . $e->getMessage());
            return redirect()->back();
        }
    }

    /**
     * Processa a devolução financeira conforme o método de pagamento escolhido
     */
    private function processarPagamentoDevolucao($tipo_pagamento, $valor, $cliente_id, $venda_id, $tipo_venda)
    {
        if ($tipo_pagamento == '00') { // Crédito em loja
            // Verificar se cliente_id foi informado
            if (empty($cliente_id) || $cliente_id == "0" || $cliente_id == "") {
                throw new \Exception("Para gerar crédito em loja é necessário um cliente cadastrado");
            }
            
            // Registra crédito para o cliente
            $cliente = Cliente::findOrFail($cliente_id);
            
            // Conforme solicitado, não usamos mais a tabela credito_cliente para devolução/troca
            // Apenas atualizamos diretamente o saldo de crédito do cliente
            $cliente->valor_credito += $valor;
            $cliente->save();
            
            // Registrar o histórico da operação no log do sistema
            __createLog(
                $cliente->empresa_id ?? null, 
                'Crédito Cliente', 
                'adicionar', 
                "Adicionado R$ " . __moeda($valor) . " de crédito para o cliente {$cliente->razao_social} por devolução/troca {$tipo_venda} #{$venda_id}"
            );
        } 
        else if ($tipo_pagamento == '06') { // Crediário
            // Para crediário, o valor será abatido das contas a receber
            // A lógica de abatimento já está implementada no método atualizarFaturasPorTipoPagamento
            // que é chamado pelo processarAtualizacaoItemDevolvido
            
            // Registrar o histórico da operação no log do sistema
            __createLog(
                auth()->user()->empresa_id ?? null, 
                'Devolução Crediário', 
                'abatimento', 
                "Abatimento de R$ " . __moeda($valor) . " nas parcelas do crediário por devolução/troca {$tipo_venda} #{$venda_id}"
            );
        }
        else if ($tipo_pagamento == '01') { // Dinheiro
            // NÃO registrar sangria para devolução em dinheiro
            // O cálculo do caixa já considera a devolução em dinheiro separadamente
            return true;
        }
        else { // Outros métodos (cartão, etc.)
            // Registra apenas para controle
            $caixa = Caixa::where('status', 1)
                ->where('usuario_id', auth()->user()->id)
                ->first();
                
            if (!$caixa) {
                throw new \Exception("Nenhum caixa aberto encontrado!");
            }
            
            // Aqui você pode implementar a lógica específica para cada tipo de pagamento
            // Por hora, apenas registramos a devolução
        }
        
        return true;
    }

    private function verificarStatusDevolucao($tipo, $venda_id)
    {
        // Buscar todas as devoluções para esta venda
        $devolucoes = Devolucao::where('tipo_venda', $tipo)
                              ->where('venda_id', $venda_id)
                              ->with('itens')
                              ->get();
        
        if ($devolucoes->isEmpty()) {
            return 'Não';
        }
        
        // Obter todos os itens da venda original
        $itensOriginais = [];
        if ($tipo == 'nfce') {
            $itensOriginais = ItemNfce::where('nfce_id', $venda_id)->get()->keyBy('id');
        } else {
            $itensOriginais = ItemNfe::where('nfe_id', $venda_id)->get()->keyBy('id');
        }
        
        // Se não houver itens originais, algo está errado
        if ($itensOriginais->isEmpty()) {
            return 'Erro';
        }
        
        // Calcular quantidade total devolvida para cada item
        $quantidadesDevolvidas = [];
        foreach ($devolucoes as $devolucao) {
            foreach ($devolucao->itens as $item) {
                if (!isset($quantidadesDevolvidas[$item->item_id])) {
                    $quantidadesDevolvidas[$item->item_id] = 0;
                }
                $quantidadesDevolvidas[$item->item_id] += $item->quantidade;
            }
        }
        
        // Verificar se todos os itens foram totalmente devolvidos
        $todosTotalmenteDevolvidos = true;
        $algunsItensDevolvidos = false;
        
        foreach ($itensOriginais as $id => $itemOriginal) {
            $quantidadeDevolvida = isset($quantidadesDevolvidas[$id]) ? $quantidadesDevolvidas[$id] : 0;
            
            if ($quantidadeDevolvida > 0) {
                $algunsItensDevolvidos = true;
            }
            
            if ($quantidadeDevolvida < $itemOriginal->quantidade) {
                $todosTotalmenteDevolvidos = false;
            }
        }
        
        if ($todosTotalmenteDevolvidos) {
            return 'Total';
        } else if ($algunsItensDevolvidos) {
            return 'Parcial';
        } else {
            return 'Não';
        }
    }

    /**
     * Retorna a quantidade já devolvida de um item
     */
    private function getQuantidadeJaDevolvida($tipo_item, $item_id)
    {
        return ItemDevolucao::where('tipo_item', $tipo_item)
                          ->where('item_id', $item_id)
                          ->sum('quantidade');
    }

    /**
     * Função auxiliar para obter o empresa_id de forma segura
     */
    private function obterEmpresaIdSeguro(Request $request, $venda_id = null, $tipo_venda = null)
    {
        // Obter empresa_id com fallback seguro
        $empresa_id = $request->empresa_id;
        
        // Se empresa_id não estiver definido no request, tentar session
        if (empty($empresa_id) && session()->has('empresa_id')) {
            $empresa_id = session('empresa_id');
            Log::debug('DevolucaoVendaController: empresa_id obtido da sessão = ' . $empresa_id);
        }
        
        // Se empresa_id ainda não estiver definido, tentar obtê-lo do usuário autenticado 
        if (empty($empresa_id) && auth()->check()) {
            // Verificar se usuário tem relação empresa
            if (auth()->user()->empresa) {
                // Verificar se a relação empresa tem empresa associada
                if (auth()->user()->empresa->empresa) {
                    $empresa_id = auth()->user()->empresa->empresa->id;
                    Log::debug('DevolucaoVendaController: empresa_id obtido do auth()->user()->empresa->empresa->id = ' . $empresa_id);
                } else {
                    $empresa_id = auth()->user()->empresa->empresa_id;
                    Log::debug('DevolucaoVendaController: empresa_id obtido do auth()->user()->empresa->empresa_id = ' . $empresa_id);
                }
            } else {
                $empresa_id = auth()->user()->empresa_id;
                Log::debug('DevolucaoVendaController: empresa_id obtido do auth()->user()->empresa_id = ' . $empresa_id);
            }
        }
        
        // Verificar diretamente o modelo usuário no Auth
        if (empty($empresa_id) && Auth::check()) {
            try {
                if (Auth::user()->empresa && Auth::user()->empresa->empresa) {
                    $empresa_id = Auth::user()->empresa->empresa->id;
                    Log::debug('DevolucaoVendaController: empresa_id obtido do Auth::user()->empresa->empresa->id = ' . $empresa_id);
                }
            } catch (\Exception $e) {
                Log::error('DevolucaoVendaController: erro ao tentar obter empresa do Auth::user(): ' . $e->getMessage());
            }
        }
        
        // Em último caso, usar o empresa_id da própria venda
        if (empty($empresa_id) && $venda_id && $tipo_venda) {
            if ($tipo_venda == 'nfce') {
                $venda = Nfce::find($venda_id);
                if ($venda) {
                    $empresa_id = $venda->empresa_id;
                    Log::debug('DevolucaoVendaController: empresa_id obtido da venda (NFCe) = ' . $empresa_id);
                }
            } else {
                $venda = Nfe::find($venda_id);
                if ($venda) {
                    $empresa_id = $venda->empresa_id;
                    Log::debug('DevolucaoVendaController: empresa_id obtido da venda (NFe) = ' . $empresa_id);
                }
            }
        }
        
        // Tentar obter da variável de ambiente
        if (empty($empresa_id) && env('DEFAULT_EMPRESA_ID')) {
            $empresa_id = env('DEFAULT_EMPRESA_ID');
            Log::debug('DevolucaoVendaController: usando empresa_id padrão da variável de ambiente DEFAULT_EMPRESA_ID = ' . $empresa_id);
        }
        
        return $empresa_id;
    }

    /**
     * Processa a atualização dos itens, marcar como devolvidos, e atualizar os valores das vendas e faturas
     */
    private function processarAtualizacaoItemDevolvido($tipo, $item_id, $quantidade, $valorItem, $tipo_pagamento = null)
    {
        try {
            if ($tipo == 'nfce') {
                $item = ItemNfce::findOrFail($item_id);
                
                // Atualizar quantidade devolvida
                $novaQuantidadeDevolvida = $item->quantidade_devolvida + $quantidade;
                
                // Atualizar o item original
                if ($novaQuantidadeDevolvida >= $item->quantidade) {
                    $item->status = 1; // Devolvido totalmente
                }
                
                $item->quantidade_devolvida = $novaQuantidadeDevolvida;
                
                // Atualizar a quantidade restante
                $novaQuantidade = $item->quantidade - $quantidade;
                $item->quantidade = $novaQuantidade;
                
                // Recalcular o subtotal com base na nova quantidade
                $item->sub_total = $item->valor_unitario * $novaQuantidade;
                
                // Salvar as alterações
                $item->save();
                
                // Atualizar o total da venda
                $venda = Nfce::findOrFail($item->nfce_id);
                $venda->total = $venda->total - $valorItem;
                $venda->save();
                
                // Atualizar as faturas relacionadas priorizando o tipo de pagamento escolhido
                $this->atualizarFaturas($tipo, $venda->id, $valorItem, $tipo_pagamento);
                
            } else { // nfe
                $item = ItemNfe::findOrFail($item_id);
                
                // Atualizar quantidade devolvida
                $novaQuantidadeDevolvida = $item->quantidade_devolvida + $quantidade;
                
                // Atualizar o item original
                if ($novaQuantidadeDevolvida >= $item->quantidade) {
                    $item->status = 1; // Devolvido totalmente
                }
                
                $item->quantidade_devolvida = $novaQuantidadeDevolvida;
                
                // Atualizar a quantidade restante
                $novaQuantidade = $item->quantidade - $quantidade;
                $item->quantidade = $novaQuantidade;
                
                // Recalcular o subtotal com base na nova quantidade
                $item->sub_total = $item->valor_unitario * $novaQuantidade;
                
                // Salvar as alterações
                $item->save();
                
                // Atualizar o total da venda
                $venda = Nfe::findOrFail($item->nfe_id);
                $venda->total = $venda->total - $valorItem;
                $venda->save();
                
                // Atualizar as faturas relacionadas priorizando o tipo de pagamento escolhido
                $this->atualizarFaturas($tipo, $venda->id, $valorItem, $tipo_pagamento);
            }
            
            return true;
        } catch (\Exception $e) {
            Log::error('Erro ao processar atualização de item devolvido: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Atualiza as faturas (reduzindo proporcionalmente) e as contas a receber associadas
     */
    private function atualizarFaturas($tipo, $venda_id, $valorDevolucao, $tipo_pagamento = null)
    {
        try {
            if ($tipo_pagamento) {
                // Se temos um tipo de pagamento específico, usar a nova lógica
                return $this->atualizarFaturasPorTipoPagamento($tipo, $venda_id, $valorDevolucao, $tipo_pagamento);
            }
            
            // Lógica anterior para compatibilidade (abate proporcionalmente de todas as faturas)
            if ($tipo == 'nfce') {
                $venda = Nfce::findOrFail($venda_id);
                $faturas = FaturaNfce::where('nfce_id', $venda_id)->get();
                
                if ($faturas->isEmpty()) {
                    return; // Sem faturas para atualizar
                }
                
                // Verificar qual é o valor total das faturas
                $valorTotalFaturas = $faturas->sum('valor');
                
                // Se não há valor, não precisamos fazer nada
                if ($valorTotalFaturas <= 0) {
                    return;
                }
                
                // Calcular a proporção que cada fatura deve perder baseado no valor devolvido
                $proporcao = $valorDevolucao / $valorTotalFaturas;
                
                foreach ($faturas as $fatura) {
                    // Calculamos quanto deve ser reduzido desta fatura específica
                    $valorReducao = $fatura->valor * $proporcao;
                    
                    // Não permitir valor negativo
                    $novoValor = max(0, $fatura->valor - $valorReducao);
                    
                    // Atualizar a fatura
                    $fatura->valor = $novoValor;
                    $fatura->save();
                    
                    // Atualizar contas a receber vinculadas
                    $this->atualizarContasReceber('nfce', $venda_id, $valorReducao);
                }
            } else { // nfe
                $venda = Nfe::findOrFail($venda_id);
                $faturas = FaturaNfe::where('nfe_id', $venda_id)->get();
                
                if ($faturas->isEmpty()) {
                    return; // Sem faturas para atualizar
                }
                
                // Verificar qual é o valor total das faturas
                $valorTotalFaturas = $faturas->sum('valor');
                
                // Se não há valor, não precisamos fazer nada
                if ($valorTotalFaturas <= 0) {
                    return;
                }
                
                // Calcular a proporção que cada fatura deve perder baseado no valor devolvido
                $proporcao = $valorDevolucao / $valorTotalFaturas;
                
                foreach ($faturas as $fatura) {
                    // Calculamos quanto deve ser reduzido desta fatura específica
                    $valorReducao = $fatura->valor * $proporcao;
                    
                    // Não permitir valor negativo
                    $novoValor = max(0, $fatura->valor - $valorReducao);
                    
                    // Atualizar a fatura
                    $fatura->valor = $novoValor;
                    $fatura->save();
                    
                    // Atualizar contas a receber vinculadas
                    $this->atualizarContasReceber('nfe', $venda_id, $valorReducao);
                }
            }
        } catch (\Exception $e) {
            Log::error('Erro ao atualizar faturas: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Atualiza as faturas priorizando o tipo de pagamento escolhido para devolução
     * Método novo que implementa a lógica solicitada para abater valores nas faturas
     * conforme a forma de pagamento escolhida para a devolução
     */
    private function atualizarFaturasPorTipoPagamento($tipo, $venda_id, $valorDevolucao, $tipo_pagamento)
    {
        try {
            $faturas = [];
            $valorRestante = $valorDevolucao;
            
            if ($tipo == 'nfce') {
                $faturas = FaturaNfce::where('nfce_id', $venda_id)->orderBy('tipo_pagamento')->get();
                $classeFatura = FaturaNfce::class;
                $classePrimaryKey = 'nfce_id';
            } else { // nfe
                $faturas = FaturaNfe::where('nfe_id', $venda_id)->orderBy('tipo_pagamento')->get();
                $classeFatura = FaturaNfe::class;
                $classePrimaryKey = 'nfe_id';
            }
            
            if ($faturas->isEmpty()) {
                return; // Sem faturas para atualizar
            }
            
            // Log de informações iniciais para diagnóstico
            Log::info("Atualizando faturas por tipo de pagamento: tipo={$tipo}, venda_id={$venda_id}, valorDevolucao={$valorDevolucao}, tipo_pagamento={$tipo_pagamento}");
            Log::info("Faturas encontradas: " . $faturas->count());
            
            // Primeiro, procura faturas com o mesmo tipo de pagamento escolhido para devolução
            $faturasMesmoPagamento = $faturas->where('tipo_pagamento', $tipo_pagamento);
            
            Log::info("Faturas com mesmo tipo de pagamento ({$tipo_pagamento}): " . $faturasMesmoPagamento->count());
            
            // Processar abatimento nas faturas de mesmo tipo de pagamento
            foreach ($faturasMesmoPagamento as $fatura) {
                if ($valorRestante <= 0) {
                    break; // Já abatemos todo o valor necessário
                }
                
                $valorFatura = $fatura->valor;
                $valorAbater = min($valorFatura, $valorRestante);
                
                // Atualizar valor da fatura
                $novoValorFatura = $valorFatura - $valorAbater;
                $fatura->valor = $novoValorFatura;
                $fatura->save();
                
                // Descontar do valor restante
                $valorRestante -= $valorAbater;
                
                Log::info("Abatido R$ {$valorAbater} da fatura ID {$fatura->id} (tipo_pagamento: {$fatura->tipo_pagamento})");
                Log::info("Valor restante para abater: R$ {$valorRestante}");
                
                // Atualizar contas a receber vinculadas
                $this->atualizarContasReceber($tipo, $venda_id, $valorAbater);
            }
            
            // Se ainda há valor a abater, processa as outras faturas
            if ($valorRestante > 0) {
                Log::info("Ainda restam R$ {$valorRestante} para abater, processando outras faturas");
                
                // Faturas de outros tipos de pagamento, ordenadas pelo valor (maior para menor)
                $outrasFormas = $faturas->where('tipo_pagamento', '!=', $tipo_pagamento)->sortByDesc('valor');
                
                foreach ($outrasFormas as $fatura) {
                    if ($valorRestante <= 0) {
                        break; // Já abatemos todo o valor necessário
                    }
                    
                    $valorFatura = $fatura->valor;
                    $valorAbater = min($valorFatura, $valorRestante);
                    
                    // Atualizar valor da fatura
                    $novoValorFatura = $valorFatura - $valorAbater;
                    $fatura->valor = $novoValorFatura;
                    $fatura->save();
                    
                    // Descontar do valor restante
                    $valorRestante -= $valorAbater;
                    
                    Log::info("Abatido R$ {$valorAbater} da fatura ID {$fatura->id} (tipo_pagamento: {$fatura->tipo_pagamento})");
                    Log::info("Valor restante para abater: R$ {$valorRestante}");
                    
                    // Atualizar contas a receber vinculadas
                    $this->atualizarContasReceber($tipo, $venda_id, $valorAbater);
                }
            }
            
            // Se ainda houver valor restante, algo deu errado
            if ($valorRestante > 0) {
                Log::warning("Não foi possível abater o valor total. Valor restante: R$ {$valorRestante}");
            }
            
            return true;
            
        } catch (\Exception $e) {
            Log::error('Erro ao atualizar faturas por tipo de pagamento: ' . $e->getMessage());
            Log::error($e->getTraceAsString());
            throw $e;
        }
    }
    
    /**
     * Atualiza as contas a receber vinculadas à venda
     */
    private function atualizarContasReceber($tipo, $venda_id, $valorReducao)
    {
        try {
            // Encontra as contas a receber vinculadas à venda
            if ($tipo == 'nfce') {
                $contas = ContaReceber::where('nfce_id', $venda_id)
                                    ->where('status', '!=', ContaReceber::STATUS_RECEBIDO)
                                    ->orderBy('data_vencimento', 'asc') // Ordenar por vencimento para abater primeiro as mais antigas
                                    ->get();
            } else { // nfe
                $contas = ContaReceber::where('nfe_id', $venda_id)
                                    ->where('status', '!=', ContaReceber::STATUS_RECEBIDO)
                                    ->orderBy('data_vencimento', 'asc') // Ordenar por vencimento para abater primeiro as mais antigas
                                    ->get();
            }
            
            if ($contas->isEmpty()) {
                return; // Não há contas a receber pendentes
            }
            
            // Calcular valor total das contas pendentes
            $valorTotalContas = $contas->sum(function($conta) {
                return $conta->valor_integral - $conta->valor_recebido;
            });
            
            // Se não há valor pendente, não precisamos fazer nada
            if ($valorTotalContas <= 0) {
                return;
            }
            
            // Para crediário, vamos abater as parcelas em ordem de vencimento (FIFO)
            $valorRestante = $valorReducao;
            
            foreach ($contas as $conta) {
                if ($valorRestante <= 0) {
                    break; // Já abatemos todo o valor necessário
                }
                
                $valorPendente = $conta->valor_integral - $conta->valor_recebido;
                
                if ($valorPendente <= 0) {
                    continue; // Esta conta já foi totalmente recebida
                }
                
                // Calcular quanto abater desta conta
                $valorAbater = min($valorPendente, $valorRestante);
                
                // Reduzir o valor integral da conta
                $novoValorIntegral = $conta->valor_integral - $valorAbater;
                
                // Registrar observação sobre a devolução
                $observacao = $conta->observacao ?? '';
                $observacao .= "\nRedução de R$ " . number_format($valorAbater, 2, ',', '.') . " devido à devolução/troca em " . date('d/m/Y H:i:s');
                
                // Atualizar a conta
                $conta->valor_integral = $novoValorIntegral;
                $conta->observacao = $observacao;
                
                // Se o valor integral se igualar ao valor recebido, marcar como recebido
                if ($conta->valor_integral <= $conta->valor_recebido) {
                    $conta->status = ContaReceber::STATUS_RECEBIDO;
                    $conta->data_recebimento = now();
                }
                
                $conta->save();
                
                // Descontar do valor restante
                $valorRestante -= $valorAbater;
                
                Log::info("Abatido R$ {$valorAbater} da conta a receber ID {$conta->id} (vencimento: {$conta->data_vencimento})");
                Log::info("Valor restante para abater: R$ {$valorRestante}");
            }
            
            // Se ainda houver valor restante, algo deu errado
            if ($valorRestante > 0) {
                Log::warning("Não foi possível abater o valor total das contas a receber. Valor restante: R$ {$valorRestante}");
            }
            
        } catch (\Exception $e) {
            Log::error('Erro ao atualizar contas a receber: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Processo para item trocado - cria um novo item em substituição
     */
    private function processarItemTrocado($tipo, $item_id, $quantidade, $produto_id, $quantidade_nova, $valor_unitario_novo, $venda)
    {
        try {
            if ($tipo == 'nfce') {
                $item = ItemNfce::findOrFail($item_id);
                
                // Calcular o valor do item que está sendo trocado
                $valorItemTrocado = $item->valor_unitario * $quantidade;
                
                // Atualizar o item original como trocado
                $novaQuantidadeDevolvida = $item->quantidade_devolvida + $quantidade;
                
                // Se trocou tudo, marcar como totalmente trocado
                if ($novaQuantidadeDevolvida >= $item->quantidade) {
                    $item->status = 2; // Trocado
                }
                
                $item->quantidade_devolvida = $novaQuantidadeDevolvida;
                
                // Atualiza a quantidade restante
                $novaQuantidade = $item->quantidade - $quantidade;
                $item->quantidade = $novaQuantidade;
                
                // Recalcula o subtotal com base na nova quantidade
                $item->sub_total = $item->valor_unitario * $novaQuantidade;
                
                // Criar o novo item substituindo este
                $produto = \App\Models\Produto::findOrFail($produto_id);
                
                $novoItem = ItemNfce::create([
                    'nfce_id' => $venda->id,
                    'produto_id' => $produto_id,
                    'quantidade' => $quantidade_nova,
                    'valor_unitario' => $valor_unitario_novo,
                    'sub_total' => $quantidade_nova * $valor_unitario_novo,
                    'perc_icms' => $item->perc_icms,
                    'perc_pis' => $item->perc_pis,
                    'perc_cofins' => $item->perc_cofins,
                    'perc_ipi' => $item->perc_ipi,
                    'cst_csosn' => $item->cst_csosn,
                    'cst_pis' => $item->cst_pis,
                    'cst_cofins' => $item->cst_cofins,
                    'cst_ipi' => $item->cst_ipi,
                    'perc_red_bc' => $item->perc_red_bc,
                    'cfop' => $item->cfop,
                    'ncm' => $produto->ncm,
                    'origem' => $produto->origem,
                    'cEnq' => $item->cEnq,
                    'pST' => $item->pST,
                    'vBCSTRet' => $item->vBCSTRet,
                    'cest' => $produto->cest
                ]);
                
                // Calcular o valor do novo item
                $valorNovoItem = $valor_unitario_novo * $quantidade_nova;
                
                // Atualizar o valor total da venda, removendo o valor do item trocado e adicionando o valor do novo item
                $venda->valor_devolvido = ($venda->valor_devolvido ?? 0) + $valorItemTrocado;
                $venda->total = $venda->total - $valorItemTrocado + $valorNovoItem;
                $venda->tem_devolucao = 1;
                $venda->save();
                
                // Referenciar o novo item no antigo
                $item->item_substituto_id = $novoItem->id;
                $item->save();
                
                // Após o incremento de estoque, registre o movimento:
                \App\Models\MovimentoEstoque::create([
                    'produto_id'         => $item->produto_id,
                    'quantidade_anterior'=> null, // pode ser preenchido se disponível
                    'quantidade_nova'    => null, // pode ser preenchido se disponível
                    'diferenca'          => $quantidade,
                    'tipo'               => 'entrada',
                    'operacao'           => 'devolucao_venda',
                    'observacao'         => 'Devolução de venda #' . ($venda->numero_sequencial ?? $venda->numero ?? $venda->id),
                    'user_id'            => auth()->user()->id,
                    'inventario_id'      => null
                ]);
                
                return $novoItem;
                
            } else { // nfe
                $item = ItemNfe::findOrFail($item_id);
                
                // Calcular o valor do item que está sendo trocado
                $valorItemTrocado = $item->valor_unitario * $quantidade;
                
                // Atualizar o item original como trocado
                $novaQuantidadeDevolvida = $item->quantidade_devolvida + $quantidade;
                
                // Se trocou tudo, marcar como totalmente trocado
                if ($novaQuantidadeDevolvida >= $item->quantidade) {
                    $item->status = 2; // Trocado
                }
                
                $item->quantidade_devolvida = $novaQuantidadeDevolvida;
                
                // Atualiza a quantidade restante
                $novaQuantidade = $item->quantidade - $quantidade;
                $item->quantidade = $novaQuantidade;
                
                // Recalcula o subtotal com base na nova quantidade
                $item->sub_total = $item->valor_unitario * $novaQuantidade;
                
                // Criar o novo item substituindo este
                $produto = \App\Models\Produto::findOrFail($produto_id);
                
                $novoItem = ItemNfe::create([
                    'nfe_id' => $venda->id,
                    'produto_id' => $produto_id,
                    'quantidade' => $quantidade_nova,
                    'valor_unitario' => $valor_unitario_novo,
                    'sub_total' => $quantidade_nova * $valor_unitario_novo,
                    'perc_icms' => $item->perc_icms,
                    'perc_pis' => $item->perc_pis,
                    'perc_cofins' => $item->perc_cofins,
                    'perc_ipi' => $item->perc_ipi,
                    'cst_csosn' => $item->cst_csosn,
                    'cst_pis' => $item->cst_pis,
                    'cst_cofins' => $item->cst_cofins,
                    'cst_ipi' => $item->cst_ipi,
                    'perc_red_bc' => $item->perc_red_bc,
                    'cfop' => $item->cfop,
                    'ncm' => $produto->ncm,
                    'origem' => $produto->origem
                ]);
                
                // Calcular o valor do novo item
                $valorNovoItem = $valor_unitario_novo * $quantidade_nova;
                
                // Atualizar o valor total da venda, removendo o valor do item trocado e adicionando o valor do novo item
                $venda->valor_devolvido = ($venda->valor_devolvido ?? 0) + $valorItemTrocado;
                $venda->total = $venda->total - $valorItemTrocado + $valorNovoItem;
                $venda->tem_devolucao = 1;
                $venda->save();
                
                // Referenciar o novo item no antigo
                $item->item_substituto_id = $novoItem->id;
                $item->save();
                
                // Após o incremento de estoque, registre o movimento:
                \App\Models\MovimentoEstoque::create([
                    'produto_id'         => $item->produto_id,
                    'quantidade_anterior'=> null, // pode ser preenchido se disponível
                    'quantidade_nova'    => null, // pode ser preenchido se disponível
                    'diferenca'          => $quantidade,
                    'tipo'               => 'entrada',
                    'operacao'           => 'devolucao_venda',
                    'observacao'         => 'Devolução de venda #' . ($venda->numero_sequencial ?? $venda->numero ?? $venda->id),
                    'user_id'            => auth()->user()->id,
                    'inventario_id'      => null
                ]);
                
                return $novoItem;
            }
        } catch (\Exception $e) {
            Log::error('Erro ao processar item trocado: ' . $e->getMessage());
            throw $e;
        }
    }

    public function linhaProdutoTrocaAdd(Request $request)
    {
        // Validar se o produto faz parte da venda original (NFC-e)
        $venda = Nfce::with('itens.produto')->find($request->venda_id);
        if (!$venda) {
            return response()->json('Venda não encontrada!', 404);
        }
        $item = $venda->itens->where('produto_id', $request->produto_id)->first();
        if (!$item) {
            return response()->json('Produto não faz parte da venda!', 400);
        }
        // Montar a linha HTML para a tabela de troca/devolução
        return view('devolucao_venda.partials.row_troca', compact('item'))->render();
    }

    private function brToFloat($valor) {
        return floatval(str_replace(',', '.', preg_replace('/[.]/', '', $valor)));
    }
} 