<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Consignacao;
use App\Models\ItemConsignacao;
use App\Models\Cliente;
use App\Models\Funcionario;
use App\Models\Produto;
use App\Models\CategoriaProduto;
use App\Models\Marca;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class ConsignacaoApiController extends Controller
{
    /**
     * Obter empresa_id da sessão do usuário
     */
    private function getEmpresaId()
    {
        if (Auth::check()) {
            return Auth::user()->empresa_id;
        }
        
        // Fallback para request()->empresa_id se disponível
        return request()->empresa_id;
    }

    /**
     * Buscar produtos para consignação
     */
    public function buscarProdutos(Request $request)
    {
        $termo = $request->get('termo', '');
        $categoria_id = $request->get('categoria_id');
        $marca_id = $request->get('marca_id');
        $search = $request->get('search', '');
        
        $query = Produto::where('empresa_id', $this->getEmpresaId())
            ->where('status', 1)
            ->where('produtos.status', 1);

        if ($termo) {
            $query->where(function($q) use ($termo) {
                $q->where('nome', 'like', "%{$termo}%")
                  ->orWhere('codigo', 'like', "%{$termo}%")
                  ->orWhere('codigo_barras', 'like', "%{$termo}%");
            });
        }

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('nome', 'like', "%{$search}%")
                  ->orWhere('codigo', 'like', "%{$search}%")
                  ->orWhere('codigo_barras', 'like', "%{$search}%");
            });
        }

        if ($categoria_id) {
            $query->where('categoria_id', $categoria_id);
        }

        if ($marca_id) {
            $query->where('marca_id', $marca_id);
        }

        $produtos = $query->with(['categoria', 'marca', 'unidadeMedida'])
            ->orderBy('nome')
            ->limit(20)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $produtos
        ]);
    }

    /**
     * Buscar produto por código de barras
     */
    public function buscarProdutoPorCodigo(Request $request)
    {
        $codigo = $request->get('codigo');
        
        if (!$codigo) {
            return response()->json([
                'success' => false,
                'message' => 'Código não informado'
            ], 400);
        }

        $produto = Produto::where('empresa_id', $this->getEmpresaId())
            ->where('status', 1)
            ->where(function($query) use ($codigo) {
                $query->where('codigo_barras', $codigo)
                      ->orWhere('codigo', $codigo)
                      ->orWhere('nome', 'like', "%{$codigo}%");
            })
            ->with(['categoria', 'marca', 'unidadeMedida'])
            ->first();

        if (!$produto) {
            return response()->json([
                'success' => false,
                'message' => 'Produto não encontrado'
            ], 404);
        }

        return response()->json($produto);
    }

    /**
     * Buscar categorias
     */
    public function buscarCategorias()
    {
        $categorias = CategoriaProduto::where('empresa_id', $this->getEmpresaId())
            ->orderBy('nome')
            ->get();

        return response()->json($categorias);
    }

    /**
     * Buscar marcas
     */
    public function buscarMarcas()
    {
        $marcas = Marca::where('empresa_id', $this->getEmpresaId())
            ->orderBy('nome')
            ->get();

        return response()->json($marcas);
    }

    /**
     * Buscar clientes
     */
    public function buscarClientes(Request $request)
    {
        $termo = $request->get('termo', '');
        
        $clientes = Cliente::where('empresa_id', $this->getEmpresaId())
            ->where(function($query) use ($termo) {
                $query->where('nome', 'like', "%{$termo}%")
                      ->orWhere('cpf_cnpj', 'like', "%{$termo}%");
            })
            ->orderBy('nome')
            ->limit(20)
            ->get();

        return response()->json([
            'success' => true,
            'clientes' => $clientes
        ]);
    }

    /**
     * Buscar funcionários
     */
    public function buscarFuncionarios(Request $request)
    {
        $termo = $request->get('termo', '');
        
        $funcionarios = Funcionario::where('empresa_id', $this->getEmpresaId())
            ->where('nome', 'like', "%{$termo}%")
            ->orderBy('nome')
            ->limit(20)
            ->get();

        return response()->json([
            'success' => true,
            'funcionarios' => $funcionarios
        ]);
    }

    /**
     * Obter dados do produto
     */
    public function obterProduto(Request $request)
    {
        $produto_id = $request->get('produto_id');
        
        $produto = Produto::where('empresa_id', $this->getEmpresaId())
            ->where('id', $produto_id)
            ->with(['categoria', 'marca', 'unidadeMedida'])
            ->first();

        if (!$produto) {
            return response()->json([
                'success' => false,
                'message' => 'Produto não encontrado'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'produto' => $produto
        ]);
    }

    /**
     * Calcular total da consignação
     */
    public function calcularTotal(Request $request)
    {
        $itens = $request->get('itens', []);
        $total = 0;

        foreach ($itens as $item) {
            $quantidade = floatval($item['quantidade'] ?? 0);
            $valor_unitario = floatval($item['valor_unitario'] ?? 0);
            $total += $quantidade * $valor_unitario;
        }

        return response()->json([
            'success' => true,
            'total' => $total,
            'total_formatado' => 'R$ ' . number_format($total, 2, ',', '.')
        ]);
    }

    /**
     * Verificar estoque do produto
     */
    public function verificarEstoque(Request $request)
    {
        $produto_id = $request->get('produto_id');
        $produto_variacao_id = $request->get('produto_variacao_id');
        $quantidade = floatval($request->get('quantidade', 0));

        $produto = Produto::where('empresa_id', $this->getEmpresaId())
            ->where('id', $produto_id)
            ->first();

        if (!$produto) {
            return response()->json([
                'success' => false,
                'message' => 'Produto não encontrado'
            ], 404);
        }

        // Aqui você pode implementar a lógica de verificação de estoque
        // Por enquanto, vamos assumir que sempre há estoque disponível
        $estoque_disponivel = 999999; // Valor fictício

        $tem_estoque = $estoque_disponivel >= $quantidade;

        return response()->json([
            'success' => true,
            'tem_estoque' => $tem_estoque,
            'estoque_disponivel' => $estoque_disponivel,
            'quantidade_solicitada' => $quantidade
        ]);
    }

    /**
     * Obter estatísticas da consignação
     */
    public function estatisticas(Request $request)
    {
        $consignacao_id = $request->get('consignacao_id');
        
        $consignacao = Consignacao::where('empresa_id', $this->getEmpresaId())
            ->where('id', $consignacao_id)
            ->with('itens')
            ->first();

        if (!$consignacao) {
            return response()->json([
                'success' => false,
                'message' => 'Consignação não encontrada'
            ], 404);
        }

        $total_consignado = $consignacao->itens->sum('quantidade');
        $total_vendido = $consignacao->itens->sum('quantidade_vendida');
        $total_devolvido = $consignacao->itens->sum('quantidade_devolvida');
        $total_disponivel = $total_consignado - $total_vendido + $total_devolvido;

        return response()->json([
            'success' => true,
            'estatisticas' => [
                'total_consignado' => $total_consignado,
                'total_vendido' => $total_vendido,
                'total_devolvido' => $total_devolvido,
                'total_disponivel' => $total_disponivel,
                'percentual_vendido' => $total_consignado > 0 ? ($total_vendido / $total_consignado) * 100 : 0,
                'percentual_devolvido' => $total_vendido > 0 ? ($total_devolvido / $total_vendido) * 100 : 0
            ]
        ]);
    }

    /**
     * Atualizar quantidade vendida de um item
     */
    public function atualizarQuantidadeVendida(Request $request)
    {
        $item_id = $request->get('item_id');
        $quantidade_vendida = floatval($request->get('quantidade_vendida', 0));

        $item = ItemConsignacao::whereHas('consignacao', function($query) {
            $query->where('empresa_id', $this->getEmpresaId());
        })->where('id', $item_id)->first();

        if (!$item) {
            return response()->json([
                'success' => false,
                'message' => 'Item não encontrado'
            ], 404);
        }

        if ($quantidade_vendida > $item->quantidade) {
            return response()->json([
                'success' => false,
                'message' => 'Quantidade vendida não pode ser maior que a quantidade consignada'
            ], 400);
        }

        try {
            $item->update([
                'quantidade_vendida' => $quantidade_vendida
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Quantidade vendida atualizada com sucesso',
                'item' => $item->fresh()
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao atualizar quantidade vendida: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obter histórico de movimentações da consignação
     */
    public function historicoMovimentacoes(Request $request)
    {
        $consignacao_id = $request->get('consignacao_id');
        
        // Aqui você pode implementar a lógica para buscar o histórico
        // de movimentações da consignação (vendas, devoluções, etc.)
        
        return response()->json([
            'success' => true,
            'movimentacoes' => []
        ]);
    }
} 