<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Nfce;
use App\Models\Empresa;
use App\Models\Caixa;
use App\Models\Contigencia;
use App\Models\EmailConfig;
use App\Models\UsuarioEmissao;
use App\Services\NFCeService;
use App\Utils\EmailUtil;
use Mail;
use NFePHP\DA\NFe\Danfce;
use App\Utils\SiegUtil;
use App\Models\FaturaNfce;
use Illuminate\Support\Facades\Log;
use App\Models\Produto;

class NFCePainelController extends Controller
{
    protected $emailUtil;
    protected $siegUtil;

    public function __construct(EmailUtil $util, SiegUtil $siegUtil){
        $this->emailUtil = $util;
        $this->siegUtil = $siegUtil;

        if (!is_dir(public_path('xml_nfce'))) {
            mkdir(public_path('xml_nfce'), 0777, true);
        }
        if (!is_dir(public_path('xml_nfce_cancelada'))) {
            mkdir(public_path('xml_nfce_cancelada'), 0777, true);
        }
    }

    /**
     * Sanitize data to valid UTF-8 recursively and build a JSON response without encoding errors
     */
    private function jsonSafe($data, int $status = 200)
    {
        $clean = $this->toUtf8($data);
        return response()->json($clean, $status, [], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    /**
     * Recursively convert strings to valid UTF-8 to avoid Malformed UTF-8 errors
     */
    private function toUtf8($data)
    {
        if (is_string($data)) {
            // Try to detect and convert invalid encodings
            if (!mb_detect_encoding($data, 'UTF-8', true)) {
                $data = mb_convert_encoding($data, 'UTF-8', 'UTF-8, ISO-8859-1, ISO-8859-15, ASCII');
            }
            // Remove/ignore invalid UTF-8 sequences if any remain
            $data = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/u', '', $data);
            return $data;
        }
        if (is_array($data)) {
            foreach ($data as $k => $v) {
                $data[$k] = $this->toUtf8($v);
            }
            return $data;
        }
        if (is_object($data)) {
            // Convert object to array, sanitize, then back if needed
            $arr = [];
            foreach ((array)$data as $k => $v) {
                $arr[$k] = $this->toUtf8($v);
            }
            return $arr;
        }
        return $data;
    }

    public function emitir(Request $request){
        try{
            Log::info('NFCePainelController - Iniciando emissão:', [
                'request_id' => $request->id,
                'request_all' => $request->all()
            ]);
            
            $nfce = Nfce::findOrFail($request->id);
            Log::info('NFCePainelController - NFCe encontrada:', [
                'nfce_id' => $nfce->id,
                'nfce_numero' => $nfce->numero,
                'nfce_estado' => $nfce->estado,
                'nfce_empresa_id' => $nfce->empresa_id
            ]);
            
            $empresa = Empresa::findOrFail($nfce->empresa_id);
            $empresa = __objetoParaEmissao($empresa, $nfce->local_id);

            // Atualiza as informações tributárias dos produtos da NFCe
            $this->atualizaTributacaoProdutos($nfce);

            // Verifica e atualiza o número da nota se necessário
            if ($nfce->numero == 0 || $nfce->estado == 'novo' || $nfce->estado == 'rejeitado') {
                $ambiente = (int)$empresa->ambiente;
                
                // Verifica se existe configuração de usuário emissão
                $configUsuarioEmissao = UsuarioEmissao::where('usuario_empresas.empresa_id', $nfce->empresa_id)
                ->join('usuario_empresas', 'usuario_empresas.usuario_id', '=', 'usuario_emissaos.usuario_id')
                ->select('usuario_emissaos.*')
                ->where('usuario_emissaos.usuario_id', $nfce->user_id)
                ->first();
                
                if ($configUsuarioEmissao != null) {
                    // Se existe configuração por usuário, usa o próximo número dessa configuração
                    $proximoNumero = $configUsuarioEmissao->numero_ultima_nfce + 1;
                    $nfce->numero = $proximoNumero;
                } else {
                    // Caso contrário, usa a configuração da empresa
                    if ($ambiente == 1) { // Produção
                        $proximoNumero = $empresa->numero_ultima_nfce_producao + 1;
                        $nfce->numero = $proximoNumero;
                    } else { // Homologação
                        $proximoNumero = $empresa->numero_ultima_nfce_homologacao + 1;
                        $nfce->numero = $proximoNumero;
                    }
                }
                
                // Salva o número atualizado
                $nfce->save();
                
                // Log da atualização do número
                __createLog($nfce->empresa_id, 'NFCe', 'atualizar', "Número da NFCe atualizado para $nfce->numero antes da transmissão");
            }

            if($empresa->arquivo == null){
                return response()->json("Certificado não encontrado para este emitente", 401);
            }

            // Garantir que o desconto seja salvo corretamente
            if ($request->has('desconto') && $request->desconto > 0) {
                $nfce->desconto = __convert_value_bd($request->desconto);
                $nfce->valor_produtos = $nfce->total + $nfce->desconto; // Valor antes do desconto
                $nfce->save();
                
                Log::info('NFCe Painel - Desconto atualizado:', [
                    'nfce_id' => $nfce->id,
                    'desconto' => $nfce->desconto,
                    'valor_produtos' => $nfce->valor_produtos,
                    'total' => $nfce->total
                ]);
            } else if ($request->has('valor_produtos') && $request->has('total')) {
                $valorProdutos = __convert_value_bd($request->valor_produtos);
                $total = __convert_value_bd($request->total);
                if ($valorProdutos > $total) {
                    $nfce->desconto = $valorProdutos - $total;
                    $nfce->valor_produtos = $valorProdutos;
                    $nfce->total = $total;
                    $nfce->save();
                    
                    Log::info('NFCe Painel - Desconto calculado pela diferença:', [
                        'nfce_id' => $nfce->id,
                        'valor_produtos' => $valorProdutos,
                        'total' => $total,
                        'desconto_calculado' => $nfce->desconto
                    ]);
                }
            }

            // NOVA VERIFICAÇÃO FINAL
            if ($nfce->desconto == 0) {
                $itensTotal = $nfce->itens->sum('sub_total');
                if ($itensTotal > $nfce->total) {
                    $nfce->desconto = $itensTotal - $nfce->total;
                    $nfce->valor_produtos = $itensTotal;
                    $nfce->save();
                    
                    Log::info('NFCe Painel - Desconto final calculado:', [
                        'nfce_id' => $nfce->id,
                        'itens_total' => $itensTotal,
                        'total' => $nfce->total,
                        'desconto_final' => $nfce->desconto
                    ]);
                }
            }

            $configUsuarioEmissao = UsuarioEmissao::where('usuario_empresas.empresa_id', $nfce->empresa_id)
            ->join('usuario_empresas', 'usuario_empresas.usuario_id', '=', 'usuario_emissaos.usuario_id')
            ->select('usuario_emissaos.*')
            ->where('usuario_emissaos.usuario_id', $nfce->user_id)
            ->first();

            $nfe_service = new NFCeService([
                "atualizacao" => date('Y-m-d h:i:s'),
                "tpAmb" => (int)$nfce->ambiente,
                "razaosocial" => $empresa->nome,
                "siglaUF" => $empresa->cidade->uf,
                "cnpj" => preg_replace('/[^0-9]/', '', $empresa->cpf_cnpj),
                "schemes" => "PL_009_V4",
                "versao" => "4.00",
                "CSC" => $empresa->csc,
                "CSCid" => $empresa->csc_id
            ], $empresa);

            Log::info('NFCePainelController - Gerando XML da NFCe...');
            $doc = $nfe_service->gerarXml($nfce);

            if(!isset($doc['erros_xml'])){
                Log::info('NFCePainelController - XML gerado com sucesso:', [
                    'chave' => $doc['chave'],
                    'numero' => $doc['numero']
                ]);
                
                $xml = $doc['xml'];
                $chave = $doc['chave'];

                $xmlTemp = simplexml_load_string($xml);

                $itensComErro = "";
                $regime = $empresa->tributacao;
                foreach ($xmlTemp->infNFe->det as $item) {
                    if (isset($item->imposto->ICMS)) {
                        $icms = (array_values((array)$item->imposto->ICMS));
                        if(sizeof($icms) == 0){
                            $itensComErro .= " Produto " . $item->prod->xProd . " não formando a TAG ICMS, confira se o CST do item corresponde a tributação, regime configurado: $regime";
                        }
                    }
                }

                if($itensComErro){
                    return $this->jsonSafe($itensComErro, 403);
                }
                try{
                    Log::info('NFCePainelController - Assinando XML...');
                    $signed = $nfe_service->sign($xml);
                    Log::info('NFCePainelController - XML assinado com sucesso');
                    
                    // contigencia condicional
                    if($this->getContigencia($empresa->id)){
                        Log::info('NFCePainelController - Emitindo em contingência');
                        if(!is_dir(public_path('xml_nfce_contigencia'))){
                            mkdir(public_path('xml_nfce_contigencia'), 0777, true);
                        }

                        $nfce->contigencia = 1;
                        $nfce->reenvio_contigencia = 0;
                        $nfce->chave = $chave;

                        $nfce->estado = 'aprovado';
                        $nfce->numero = $doc['numero'];
                        $nfce->data_emissao = date('Y-m-d H:i:s');
                        $nfce->save();
                        
                        if($configUsuarioEmissao == null){
                            if($empresa->ambiente == 2){
                                $empresa->numero_ultima_nfce_homologacao = $doc['numero'];
                            }else{
                                $empresa->numero_ultima_nfce_producao = $doc['numero'];
                            }
                            $empresa->save();
                        }else{
                            $configUsuarioEmissao->numero_ultima_nfce = $doc['numero'];
                            $configUsuarioEmissao->save();
                        }

                        if($request->tipo_nfce == 'fiscal'){
                            // NOVA FUNCIONALIDADE: Agrupar pagamentos do mesmo tipo antes de persistir no banco
                            // Isso garante que os pagamentos de mesmo tipo serão agrupados no XML
                            if($nfce->fatura->count() > 0 && !$nfce->contigencia) {
                                // Criar array para agrupar pagamentos
                                $pagamentosAgrupados = [];
                                
                                // Agrupar pagamentos por tipo
                                foreach($nfce->fatura as $fatura) {
                                    $tipoPagamento = $fatura->tipo_pagamento;
                                    
                                    // Normalização de tipos específicos
                                    if($tipoPagamento == '30'){
                                        $tipoPagamento = '03';
                                    } elseif($tipoPagamento == '31'){
                                        $tipoPagamento = '04';
                                    } elseif($tipoPagamento == '32'){
                                        $tipoPagamento = '17';
                                    }
                                    
                                    // Regra para tipo '06'
                                    if($tipoPagamento == '06'){
                                        $tipoPagamento = '05'; 
                                    }
                                    
                                    // Inicializar grupo se não existir
                                    if(!isset($pagamentosAgrupados[$tipoPagamento])) {
                                        $pagamentosAgrupados[$tipoPagamento] = [
                                            'tipo_pagamento' => $tipoPagamento,
                                            'data_vencimento' => $fatura->data_vencimento,
                                            'valor' => 0
                                        ];
                                    }
                                    
                                    // Somar valor ao grupo correspondente
                                    $pagamentosAgrupados[$tipoPagamento]['valor'] += $fatura->valor;
                                }
                                
                                // Remover faturas existentes e criar novas com valores agrupados
                                $nfce->fatura()->delete();
                                
                                foreach($pagamentosAgrupados as $pagamento) {
                                    FaturaNfce::create([
                                        'nfce_id' => $nfce->id,
                                        'tipo_pagamento' => $pagamento['tipo_pagamento'],
                                        'data_vencimento' => $pagamento['data_vencimento'] ?? date('Y-m-d'),
                                        'valor' => $pagamento['valor']
                                    ]);
                                }
                            }
                        }

                        file_put_contents(public_path('xml_nfce_contigencia/').$chave.'.xml', $signed);
                        $data = [
                            'recibo' => '',
                            'chave' => $chave,
                            'contigencia' => 1
                        ];
                        $descricaoLog = 'Emitida em contigência número ' . $doc['numero'];
                        __createLog($nfce->empresa_id, 'NFCe', 'transmitir', $descricaoLog);
                        return $this->jsonSafe($data, 200);

                    }else{
                        Log::info('NFCePainelController - Transmitindo para SEFAZ...');
                        $resultado = $nfe_service->transmitir($signed, $doc['chave']);
                        Log::info('NFCePainelController - Resultado da transmissão:', $resultado);

                        if ($resultado['erro'] == 0) {
                            $nfce->chave = $doc['chave'];
                            $nfce->estado = 'aprovado';

                            if($configUsuarioEmissao == null){
                                if($empresa->ambiente == 2){
                                    $empresa->numero_ultima_nfce_homologacao = $doc['numero'];
                                }else{
                                    $empresa->numero_ultima_nfce_producao = $doc['numero'];
                                }
                                $empresa->save();
                            }else{
                                $configUsuarioEmissao->numero_ultima_nfce = $doc['numero'];
                                $configUsuarioEmissao->save();
                            }

                            $nfce->numero = $doc['numero'];
                            $nfce->recibo = $resultado['success'];
                            $nfce->data_emissao = date('Y-m-d H:i:s');

                            $nfce->save();

                            $data = [
                                'recibo' => $resultado['success'],
                                'chave' => $nfce->chave
                            ];

                            $descricaoLog = "Emitida número $nfce->numero - $nfce->chave APROVADA";
                            __createLog($nfce->empresa_id, 'NFCe', 'transmitir', $descricaoLog);
                            
                            // Log para debug
                            Log::info('NFCe emitida com sucesso:', [
                                'nfce_id' => $nfce->id,
                                'numero' => $nfce->numero,
                                'chave' => $nfce->chave,
                                'recibo' => $resultado['success'],
                                'estado' => $nfce->estado
                            ]);

                            try{
                                $fileDir = public_path('xml_nfce/').$nfce->chave.'.xml';
                                $this->emailUtil->enviarXmlContador($nfce->empresa_id, $fileDir, 'NFCe', $nfce->chave);
                            }catch(\Exception $e){

                            }

                            try{
                                $fileDir = public_path('xml_nfce/').$nfce->chave.'.xml';
                                $this->siegUtil->enviarXml($nfce->empresa_id, $fileDir);
                            }catch(\Exception $e){

                            }

                            return $this->jsonSafe($data, 200);
                        }else{
                            $recibo = isset($resultado['recibo']) ? $resultado['recibo'] : null;

                            $error = $resultado['error'];

                            // Verificar se é uma mensagem de autorização pendente
                            if (is_string($error) && strpos($error, 'Autorização pendente') !== false) {
                                // Se for autorização pendente, retorna sucesso com o recibo
                                $nfce->chave = $doc['chave'];
                                $nfce->recibo = $recibo;
                                $nfce->estado = 'aprovado';
                                $nfce->data_emissao = date('Y-m-d H:i:s');
                                $nfce->save();
                                
                                $data = [
                                    'recibo' => $recibo,
                                    'chave' => $nfce->chave
                                ];
                                
                                $descricaoLog = "Emitida número $nfce->numero - $nfce->chave (Autorização pendente)";
                                __createLog($nfce->empresa_id, 'NFCe', 'transmitir', $descricaoLog);
                                
                                return $this->jsonSafe($data, 200);
                            }

                            if($nfce->chave == ''){
                                $nfce->chave = $doc['chave'];
                            }

                            if($nfce->signed_xml == null){
                                $nfce->signed_xml = $signed;
                            }
                            if($nfce->recibo == null){
                                $nfce->recibo = $recibo;
                            }
                            $nfce->estado = 'rejeitado';
                            $nfce->save();

                            if(isset($error['protNFe'])){
                                $motivo = $error['protNFe']['infProt']['xMotivo'];
                                $cStat = $error['protNFe']['infProt']['cStat'];

                                $nfce->motivo_rejeicao = substr("[$cStat] $motivo", 0, 200);
                                $nfce->save();

                                $descricaoLog = "REJEITADA $nfce->chave - $motivo";
                                __createLog($nfce->empresa_id, 'NFCe', 'erro', $descricaoLog);

                                return $this->jsonSafe("[$cStat] $motivo", 403);
                            }else{
                                return $this->jsonSafe($error, 403);
                            }
                        }
                    }
                }catch(\Exception $e){
                    __createLog($nfce->empresa_id, 'NFCe', 'erro', $e->getMessage());
                    return $this->jsonSafe($e->getMessage(), 404);
                }

            }else{
                Log::error('NFCePainelController - Erro na geração do XML:', $doc['erros_xml']);
                return $this->jsonSafe($doc['erros_xml'], 401);
            }
        }catch(\Exception $e){
            Log::error('NFCePainelController - Erro geral na emissão:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'nfce_id' => $request->id ?? null
            ]);
            __createLog($nfce->empresa_id, 'NFCe', 'erro', $e->getMessage());
            return $this->jsonSafe($e->getMessage(), 404);
        }
    }

    private function getContigencia($empresa_id){
        $active = Contigencia::
        where('empresa_id', $empresa_id)
        ->where('status', 1)
        ->where('documento', 'NFCe')
        ->first();
        return $active != null ? 1 : 0;
    }

    public function cancelar(Request $request)
    {
        $nfce = Nfce::findOrFail($request->id);
        $empresa = Empresa::findOrFail($nfce->empresa_id);
        $empresa = __objetoParaEmissao($empresa, $nfce->local_id);

        if ($nfce != null) {
            $cnpj = preg_replace('/[^0-9]/', '', $empresa->cpf_cnpj);
            $nfe_service = new NFCeService([
                "atualizacao" => date('Y-m-d h:i:s'),
                "tpAmb" => (int)$nfce->ambiente,
                "razaosocial" => $empresa->nome,
                "siglaUF" => $empresa->cidade->uf,
                "cnpj" => preg_replace('/[^0-9]/', '', $empresa->cpf_cnpj),
                "schemes" => "PL_009_V4",
                "versao" => "4.00",
            ], $empresa);
            $doc = $nfe_service->cancelar($nfce, $request->motivo);

            if (!isset($doc['erro'])) {
                $nfce->estado = 'cancelado';
                $nfce->save();
                // return response()->json($doc, 200);
                $motivo = $doc['retEvento']['infEvento']['xMotivo'];
                $cStat = $doc['retEvento']['infEvento']['cStat'];

                if($cStat == 135){
                    $descricaoLog = "CANCELADA $nfce->chave";
                    __createLog($nfce->empresa_id, 'NFCe', 'cancelar', $descricaoLog);

                    try{
                        $fileDir = public_path('xml_nfce_cancelada/').$nfce->chave.'.xml';
                        $this->siegUtil->enviarXml($nfce->empresa_id, $fileDir);
                    }catch(\Exception $e){
                    }

                    return response()->json("[$cStat] $motivo", 200);
                }else{
                    $descricaoLog = "ERRO CANCELAR: $nfce->chave";
                    __createLog($nfce->empresa_id, 'NFCe', 'erro', $descricaoLog);
                    return response()->json("[$cStat] $motivo", 401);
                }
            } else {
                $arr = $doc['data'];
                $cStat = $arr['retEvento']['infEvento']['cStat'];
                $motivo = $arr['retEvento']['infEvento']['xMotivo'];
                
                __createLog($nfce->empresa_id, 'NFCe', 'erro', "[$cStat] $motivo");
                return response()->json("[$cStat] $motivo", $doc['status']);
            }
        } else {
            return response()->json('Consulta não encontrada', 404);
        }
    }

    public function consultar(Request $request)
    {
        $nfce = Nfce::findOrFail($request->id);
        $empresa = Empresa::findOrFail($nfce->empresa_id);
        $empresa = __objetoParaEmissao($empresa, $nfce->local_id);
        
        if ($nfce != null) {
            $cnpj = preg_replace('/[^0-9]/', '', $empresa->cpf_cnpj);
            $nfe_service = new NFCeService([
                "atualizacao" => date('Y-m-d h:i:s'),
                "tpAmb" => (int)$nfce->ambiente,
                "razaosocial" => $empresa->nome,
                "siglaUF" => $empresa->cidade->uf,
                "cnpj" => preg_replace('/[^0-9]/', '', $empresa->cpf_cnpj),
                "schemes" => "PL_009_V4",
                "versao" => "4.00",
            ], $empresa);
            $consulta = $nfe_service->consultar($nfce);
            if (!isset($consulta['erro'])) {
                try{

                    $motivo = $consulta['protNFe']['infProt']['xMotivo'];
                    $cStat = $consulta['protNFe']['infProt']['cStat'];
                    if($cStat == 100){
                        return response()->json("[$cStat] $motivo", 200);
                    }else{
                        return response()->json("[$cStat] $motivo", 401);
                    }
                }catch(\Exception $e){
                    return response()->json($consulta['cStat'] . " " . $consulta['xMotivo'], 404);
                }
            }else{
                return response()->json($consulta['data'], $consulta['status']);
            }
        } else {
            return response()->json('Consulta não encontrada', 404);
        }
    }

    public function consultaStatusSefaz(Request $request){
        $caixa = Caixa::where('usuario_id', $request->usuario_id)->where('status', 1)->first();
        $empresa = Empresa::findOrFail($request->empresa_id);
        $empresa = __objetoParaEmissao($empresa, $caixa ? $caixa->local_id : null);

        $nfce_service = new NFCeService([
            "atualizacao" => date('Y-m-d h:i:s'),
            "tpAmb" => (int)$empresa->ambiente,
            "razaosocial" => $empresa->nome,
            "siglaUF" => $empresa->cidade->uf,
            "cnpj" => preg_replace('/[^0-9]/', '', $empresa->cpf_cnpj),
            "schemes" => "PL_009_V4",
            "versao" => "4.00",
        ], $empresa);
        $consulta = $nfce_service->consultaStatus((int)$empresa->ambiente, $empresa->cidade->uf);
        return $this->jsonSafe($consulta, 200);
    }

    public function transmitirContigencia(Request $request){

        $nfce = Nfce::findOrFail($request->id);

        $empresa = Empresa::findOrFail($nfce->empresa_id);
        $empresa = __objetoParaEmissao($empresa, $nfce->local_id);
        
        if($empresa->arquivo == null){
            return response()->json("Certificado não encontrado para este emitente", 401);
        }

        if(!file_exists(public_path('xml_nfce_contigencia/'.$nfce->chave.'.xml'))){
            return response()->json("arquivo não existe", 401);
        }
        $nfe_service = new NFCeService([
            "atualizacao" => date('Y-m-d h:i:s'),
            "tpAmb" => (int)$nfce->ambiente,
            "razaosocial" => $empresa->nome,
            "siglaUF" => $empresa->cidade->uf,
            "cnpj" => preg_replace('/[^0-9]/', '', $empresa->cpf_cnpj),
            "schemes" => "PL_009_V4",
            "versao" => "4.00",
            "CSC" => $empresa->csc,
            "CSCid" => $empresa->csc_id
        ], $empresa);

        $xml = file_get_contents(public_path('xml_nfce_contigencia/'.$nfce->chave.'.xml'));
        $resultado = $nfe_service->transmitir($xml, $nfce->chave);

        if ($resultado['erro'] == 0) {

            $nfce->estado = 'aprovado';
            $nfce->reenvio_contigencia = 1;
            $nfce->save();

            $data = [
                'recibo' => $resultado['success'],
                'chave' => $nfce->chave
            ];
            return response()->json($data, 200);

        }else{
            $error = $resultado['error'];
            $nfce->estado = 'rejeitado';
            $nfce->save();

            if(isset($error['protNFe'])){
                $motivo = $error['protNFe']['infProt']['xMotivo'];
                $cStat = $error['protNFe']['infProt']['cStat'];

                $nfce->motivo_rejeicao = substr("[$cStat] $motivo", 0, 200);
                $nfce->save();

                return $this->jsonSafe("[$cStat] $motivo", 403);
            }else{
                return $this->jsonSafe($error, 403);
            }
        }

    }

    public function sendMail(Request $request){
        $email = $request->email;
        $xml = $request->xml;
        $danfe = $request->danfe;
        $id = $request->id;

        $nfce = Nfce::findOrFail($id);
        if(!$email){
            return response()->json('Informe o email!', 404);
        }

        $docs = [];
        if($xml){
            $docs[] = public_path('xml_nfce/').$nfce->chave.'.xml';
        }
        if($danfe){
            $this->gerarDanfceTemporaria($nfce);
            $docs[] = public_path('danfce/').$nfce->chave.'.pdf';
        }

        $emailConfig = EmailConfig::where('empresa_id', $nfce->empresa_id)
        ->where('status', 1)
        ->first();
        try{
            if($emailConfig != null){

                $body = view('mail.nfce', compact('nfce'));
                $result = $this->emailUtil->enviaEmailPHPMailer($email, 'Envio de documento', $body, $emailConfig, $docs);
            }else{
                Mail::send('mail.nfce', ['nfce' => $nfce], function($m) use ($email, $docs){
                    $nomeEmail = env('MAIL_FROM_NAME');
                    $m->from(env('MAIL_USERNAME'), $nomeEmail);
                    $m->subject('Envio de documento');
                    foreach($docs as $f){
                        $m->attach($f); 
                    }
                    $m->to($email);
                });
            }

            $this->unlinkr(public_path('danfce'));
            return $this->jsonSafe("Email enviado!", 200);
        }catch(\Exception $e){
            return $this->jsonSafe($e->getMessage(), 404);
        }

    }

    private function gerarDanfceTemporaria($nfce){
        if (!is_dir(public_path('danfce'))) {
            mkdir(public_path('danfce'), 0777, true);
        }
        $xml = file_get_contents(public_path('xml_nfce/').$nfce->chave.'.xml');
        $danfe = new Danfce($xml);
        $pdf = $danfe->render();
        file_put_contents(public_path('danfce/') . $nfce->chave . '.pdf', $pdf);

    }

    private function unlinkr($dir){ 
        $files = array_diff(scandir($dir), array('.', '..')); 

        foreach ($files as $file) { 
            (is_dir("$dir/$file")) ? $this->unlinkr("$dir/$file") : unlink("$dir/$file"); 
        }

        return rmdir($dir); 
    }

    /**
     * Atualiza as informações tributárias dos produtos da NFCe
     * 
     * @param Nfce $nfce A NFCe cujos itens devem ser atualizados
     * @return void
     */
    private function atualizaTributacaoProdutos($nfce)
    {
        // Verifica se a NFCe tem itens
        if (!$nfce->itens || $nfce->itens->isEmpty()) {
            return;
        }

        // Para cada item da NFCe
        foreach ($nfce->itens as $item) {
            // Busca o produto atualizado
            $produto = Produto::find($item->produto_id);
            
            if (!$produto) {
                continue;
            }
            
            // Atualiza a tributação do produto com base na localização, se necessário
            $produto = __tributacaoProdutoLocalVenda($produto, $nfce->local_id);
            
            // Atualiza os dados tributários do item
            $item->update([
                'ncm' => $produto->ncm,
                'cest' => $produto->cest,
                'cfop' => $produto->cfop_estadual,
                'perc_icms' => $produto->perc_icms,
                'perc_pis' => $produto->perc_pis,
                'perc_cofins' => $produto->perc_cofins,
                'perc_ipi' => $produto->perc_ipi,
                'cst_csosn' => $produto->cst_csosn,
                'cst_pis' => $produto->cst_pis,
                'cst_cofins' => $produto->cst_cofins,
                'cst_ipi' => $produto->cst_ipi,
                'perc_red_bc' => $produto->perc_red_bc,
                'codigo_beneficio_fiscal' => $produto->codigo_beneficio_fiscal
            ]);
        }
    }

}
