<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Estoque;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ConsolidarEstoques extends Command
{
    protected $signature = 'consolidar:estoques';
    protected $description = 'Consolida estoques duplicados por produto e variação, deixando apenas um registro com local_id = NULL';

    public function handle()
    {
        $this->info('Iniciando consolidação de estoques...');
        DB::beginTransaction();
        try {
            // Produtos SEM variação
            $estoquesSemVariacao = Estoque::select('produto_id', DB::raw('SUM(quantidade) as total'))
                ->whereNull('produto_variacao_id')
                ->groupBy('produto_id')
                ->get();

            foreach ($estoquesSemVariacao as $e) {
                $registro = Estoque::where('produto_id', $e->produto_id)
                    ->whereNull('produto_variacao_id')
                    ->whereNull('local_id')
                    ->first();
                if ($registro) {
                    $registro->quantidade = $e->total;
                    $registro->save();
                } else {
                    Estoque::create([
                        'produto_id' => $e->produto_id,
                        'produto_variacao_id' => null,
                        'quantidade' => $e->total,
                        'local_id' => null,
                        'created_at' => Carbon::now(),
                        'updated_at' => Carbon::now(),
                    ]);
                }
            }

            // Produtos COM variação
            $estoquesComVariacao = Estoque::select('produto_id', 'produto_variacao_id', DB::raw('SUM(quantidade) as total'))
                ->whereNotNull('produto_variacao_id')
                ->groupBy('produto_id', 'produto_variacao_id')
                ->get();

            foreach ($estoquesComVariacao as $e) {
                $registro = Estoque::where('produto_id', $e->produto_id)
                    ->where('produto_variacao_id', $e->produto_variacao_id)
                    ->whereNull('local_id')
                    ->first();
                if ($registro) {
                    $registro->quantidade = $e->total;
                    $registro->save();
                } else {
                    Estoque::create([
                        'produto_id' => $e->produto_id,
                        'produto_variacao_id' => $e->produto_variacao_id,
                        'quantidade' => $e->total,
                        'local_id' => null,
                        'created_at' => Carbon::now(),
                        'updated_at' => Carbon::now(),
                    ]);
                }
            }

            // Excluir duplicados (local_id diferente de null)
            $deletados = Estoque::whereNotNull('local_id')->delete();

            DB::commit();
            $this->info('Consolidação concluída!');
            $this->info('Registros de estoque excluídos (local_id != NULL): ' . $deletados);
        } catch (\Exception $e) {
            DB::rollBack();
            $this->error('Erro ao consolidar estoques: ' . $e->getMessage());
        }
    }
} 