<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

require_once 'config/database.php';
$database = new Database();
$db = $database->getConnection();

// Definir título da página
$titulo = "Contas a Receber";

// Processar filtros
$where_conditions = [];
$params = [];

// Apenas filtrar por usuário se não for administrador
if (!isset($_SESSION['is_admin']) || $_SESSION['is_admin'] != 1) {
    $where_conditions[] = "cr.usuario_id = :usuario_id";
    $params[':usuario_id'] = $_SESSION['user_id'];
}

// Aplicar filtros de data somente se foram informados
if (isset($_GET['data_inicio']) && $_GET['data_inicio'] != '') {
    $where_conditions[] = "cr.data_vencimento >= :data_inicio";
    $params[':data_inicio'] = $_GET['data_inicio'];
}

if (isset($_GET['data_fim']) && $_GET['data_fim'] != '') {
    $where_conditions[] = "cr.data_vencimento <= :data_fim";
    $params[':data_fim'] = $_GET['data_fim'];
}

if (isset($_GET['status']) && $_GET['status'] != '') {
    $where_conditions[] = "cr.status = :status";
    $params[':status'] = $_GET['status'];
}

// Construir query
$where_clause = "";
if (!empty($where_conditions)) {
    $where_clause = "WHERE " . implode(" AND ", $where_conditions);
}

// Buscar contas a receber
$query = "SELECT cr.*, l.descricao, fp.nome as forma_pagamento, 
                 DATE_FORMAT(cr.data_vencimento, '%d/%m/%Y') as data_vencimento_formatada,
                 DATE_FORMAT(cr.data_pagamento, '%d/%m/%Y') as data_pagamento_formatada,
                 DATE_FORMAT(l.data_lancamento, '%d/%m/%Y') as data_lancamento_formatada,
                 u.nome as nome_usuario
          FROM contas_receber cr
          JOIN lancamentos l ON cr.lancamento_id = l.id
          JOIN formas_pagamento fp ON l.forma_pagamento_id = fp.id
          JOIN usuarios u ON cr.usuario_id = u.id
          " . $where_clause . "
          ORDER BY cr.data_vencimento DESC";

$stmt = $db->prepare($query);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->execute();
$contas = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calcular totais
$total_geral = 0;
$total_pendente = 0;
$total_pago = 0;

foreach ($contas as $conta) {
    $total_geral += $conta['valor'];
    if ($conta['status'] == 'PAGO') {
        $total_pago += $conta['valor'];
    } else {
        $total_pendente += $conta['valor'];
    }
}

// Buscar formas de pagamento para o modal
$query_formas = "SELECT id, nome FROM formas_pagamento WHERE ativo = 1 AND id IN(1,2) ";
$stmt_formas = $db->prepare($query_formas);
$stmt_formas->execute();
$formas_pagamento = $stmt_formas->fetchAll(PDO::FETCH_ASSOC);

// Conteúdo da página
ob_start();
?>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-header">
        <h6 class="mb-0">Filtros</h6>
    </div>
    <div class="card-body">
        <form method="GET" id="formFiltros" class="row g-3">
            <div class="col-md-3">
                <label class="form-label">Data Início</label>
                <input type="date" name="data_inicio" class="form-control" value="<?php echo isset($_GET['data_inicio']) ? $_GET['data_inicio'] : ''; ?>">
            </div>
            <div class="col-md-3">
                <label class="form-label">Data Fim</label>
                <input type="date" name="data_fim" class="form-control" value="<?php echo isset($_GET['data_fim']) ? $_GET['data_fim'] : ''; ?>">
            </div>
            <div class="col-md-3">
                <label class="form-label">Status</label>
                <select name="status" class="form-select">
                    <option value="">Todos</option>
                    <option value="PENDENTE" <?php echo (isset($_GET['status']) && $_GET['status'] == 'PENDENTE') ? 'selected' : ''; ?>>Pendente</option>
                    <option value="PAGO" <?php echo (isset($_GET['status']) && $_GET['status'] == 'PAGO') ? 'selected' : ''; ?>>Pago</option>
                </select>
            </div>
            <div class="col-md-3 d-flex align-items-end">
                <button type="submit" class="btn btn-primary me-2">
                    <i class="fas fa-search"></i> Filtrar
                </button>
                <button type="button" class="btn btn-secondary me-2" onclick="limparFiltros()">
                    <i class="fas fa-eraser"></i> Limpar
                </button>
                <button type="button" class="btn btn-success" onclick="window.print()">
                    <i class="fas fa-print"></i> Imprimir
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Totais -->
<div class="row mb-4">
    <div class="col-md-4">
        <div class="card bg-primary text-white">
            <div class="card-body">
                <h6 class="card-title">Total Geral</h6>
                <h4 class="mb-0">R$ <?php echo number_format($total_geral, 2, ',', '.'); ?></h4>
            </div>
        </div>
    </div>
    <div class="col-md-4">
        <div class="card bg-warning text-white">
            <div class="card-body">
                <h6 class="card-title">Total Pendente</h6>
                <h4 class="mb-0">R$ <?php echo number_format($total_pendente, 2, ',', '.'); ?></h4>
            </div>
        </div>
    </div>
    <div class="col-md-4">
        <div class="card bg-success text-white">
            <div class="card-body">
                <h6 class="card-title">Total Pago</h6>
                <h4 class="mb-0">R$ <?php echo number_format($total_pago, 2, ',', '.'); ?></h4>
            </div>
        </div>
    </div>
</div>

<!-- Tabela de Contas -->
<div class="card">
    <div class="card-header">
        <h5 class="mb-0">Contas a Receber</h5>
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table" id="contasTable">
                <thead>
                    <tr>
                        <th>Descrição</th>
                        <th>Valor</th>
                        <th>Data Lançamento</th>
                        <th>Vencimento</th>
                        <th>Data Pagamento</th>
                        <th>Status</th>
                        <?php if (isset($_SESSION['is_admin']) && $_SESSION['is_admin'] == 1): ?>
                        <th>Usuário</th>
                        <?php endif; ?>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($contas as $conta): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($conta['descricao']); ?></td>
                            <td>R$ <?php echo number_format($conta['valor'], 2, ',', '.'); ?></td>
                            <td><?php echo $conta['data_lancamento_formatada']; ?></td>
                            <td><?php echo $conta['data_vencimento_formatada']; ?></td>
                            <td><?php echo $conta['data_pagamento_formatada'] ?? '-'; ?></td>
                            <td>
                                <span class="badge <?php echo $conta['status'] == 'PAGO' ? 'bg-success' : 'bg-warning'; ?>">
                                    <?php echo $conta['status']; ?>
                                </span>
                            </td>
                            <?php if (isset($_SESSION['is_admin']) && $_SESSION['is_admin'] == 1): ?>
                            <td><?php echo htmlspecialchars($conta['nome_usuario']); ?></td>
                            <?php endif; ?>
                            <td>
                                <?php if ($conta['status'] == 'PENDENTE'): ?>
                                    <button type="button" class="btn btn-sm btn-success" onclick="registrarPagamento(<?php echo $conta['id']; ?>, <?php echo $conta['valor']; ?>, '<?php echo htmlspecialchars($conta['descricao']); ?>')">
                                        <i class="fas fa-money-bill-wave"></i> Pagar
                                    </button>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal de Pagamento -->
<div class="modal fade" id="modalPagamento" tabindex="-1" aria-labelledby="modalPagamentoLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="modalPagamentoLabel">Registrar Pagamento</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="formPagamento">
                    <input type="hidden" id="conta_id" name="conta_id">
                    <div class="mb-3">
                        <label for="descricao" class="form-label">Descrição</label>
                        <input type="text" class="form-control" id="descricao" name="descricao">
                    </div>
                    <div class="mb-3">
                        <label for="data_pagamento" class="form-label">Data do Pagamento</label>
                        <input type="date" class="form-control" id="data_pagamento" name="data_pagamento" required value="<?php echo date('Y-m-d'); ?>">
                    </div>
                    <div class="mb-3">
                        <label for="valor" class="form-label">Valor</label>
                        <input type="text" class="form-control" id="valor" name="valor" readonly>
                    </div>
                    <div class="mb-3">
                        <label for="forma_pagamento" class="form-label">Forma de Pagamento</label>
                        <select class="form-select" id="forma_pagamento" name="forma_pagamento" required>
                            <?php foreach ($formas_pagamento as $forma): ?>
                                <option value="<?php echo $forma['id']; ?>"><?php echo htmlspecialchars($forma['nome']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <button type="button" class="btn btn-success" onclick="confirmarPagamento()">Confirmar Pagamento</button>
            </div>
        </div>
    </div>
</div>

<!-- Área de Impressão -->
<div id="printArea" style="display: none;">
    <div style="text-align: center;">
        <h2 style="margin: 0; font-size: 18px;">Sistema Financeiro</h2>
        <h3 style="margin: 10px 0; font-size: 16px;">Relatório de Contas a Receber</h3>
        <div style="margin: 10px 0; font-size: 14px;">
            <?php if (isset($_GET['data_inicio']) && $_GET['data_inicio'] != ''): ?>
                <p style="margin: 5px 0;">Período: <?php echo date('d/m/Y', strtotime($_GET['data_inicio'])); ?> 
                    <?php echo isset($_GET['data_fim']) && $_GET['data_fim'] != '' ? ' até ' . date('d/m/Y', strtotime($_GET['data_fim'])) : ''; ?></p>
            <?php endif; ?>
            <?php if (isset($_GET['status']) && $_GET['status'] != ''): ?>
                <p style="margin: 5px 0;">Status: <?php echo $_GET['status']; ?></p>
            <?php endif; ?>
        </div>
    </div>
    
    <table style="width: 100%; border-collapse: collapse; margin: 20px 0; font-size: 14px; font-family: Arial, sans-serif;">
        <thead>
            <tr>
                <th style="border: 1px solid #ddd; padding: 8px; background-color: #f8f9fa; text-align: left;">Descrição</th>
                <th style="border: 1px solid #ddd; padding: 8px; background-color: #f8f9fa; text-align: right;">Valor</th>
                <th style="border: 1px solid #ddd; padding: 8px; background-color: #f8f9fa; text-align: left;">Data Lançamento</th>
                <th style="border: 1px solid #ddd; padding: 8px; background-color: #f8f9fa; text-align: left;">Vencimento</th>
                <th style="border: 1px solid #ddd; padding: 8px; background-color: #f8f9fa; text-align: left;">Data Pagamento</th>
                <th style="border: 1px solid #ddd; padding: 8px; background-color: #f8f9fa; text-align: left;">Status</th>
                <?php if (isset($_SESSION['is_admin']) && $_SESSION['is_admin'] == 1): ?>
                <th style="border: 1px solid #ddd; padding: 8px; background-color: #f8f9fa; text-align: left;">Usuário</th>
                <?php endif; ?>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($contas as $conta): ?>
            <tr>
                <td style="border: 1px solid #ddd; padding: 8px;"><?php echo htmlspecialchars($conta['descricao']); ?></td>
                <td style="border: 1px solid #ddd; padding: 8px; text-align: right;">R$ <?php echo number_format($conta['valor'], 2, ',', '.'); ?></td>
                <td style="border: 1px solid #ddd; padding: 8px;"><?php echo $conta['data_lancamento_formatada']; ?></td>
                <td style="border: 1px solid #ddd; padding: 8px;"><?php echo $conta['data_vencimento_formatada']; ?></td>
                <td style="border: 1px solid #ddd; padding: 8px;"><?php echo $conta['data_pagamento_formatada'] ?? '-'; ?></td>
                <td style="border: 1px solid #ddd; padding: 8px;"><?php echo $conta['status']; ?></td>
                <?php if (isset($_SESSION['is_admin']) && $_SESSION['is_admin'] == 1): ?>
                <td style="border: 1px solid #ddd; padding: 8px;"><?php echo htmlspecialchars($conta['nome_usuario']); ?></td>
                <?php endif; ?>
            </tr>
            <?php endforeach; ?>
        </tbody>
        <tfoot>
            <tr>
                <td colspan="<?php echo (isset($_SESSION['is_admin']) && $_SESSION['is_admin'] == 1) ? 6 : 5; ?>" style="border: 1px solid #ddd; padding: 8px; text-align: right; background-color: #f8f9fa;"><strong>Total Geral:</strong></td>
                <td style="border: 1px solid #ddd; padding: 8px; text-align: right; background-color: #f8f9fa;"><strong>R$ <?php echo number_format($total_geral, 2, ',', '.'); ?></strong></td>
            </tr>
            <tr>
                <td colspan="<?php echo (isset($_SESSION['is_admin']) && $_SESSION['is_admin'] == 1) ? 6 : 5; ?>" style="border: 1px solid #ddd; padding: 8px; text-align: right; background-color: #fff3cd;"><strong>Total Pendente:</strong></td>
                <td style="border: 1px solid #ddd; padding: 8px; text-align: right; background-color: #fff3cd;"><strong>R$ <?php echo number_format($total_pendente, 2, ',', '.'); ?></strong></td>
            </tr>
            <tr>
                <td colspan="<?php echo (isset($_SESSION['is_admin']) && $_SESSION['is_admin'] == 1) ? 6 : 5; ?>" style="border: 1px solid #ddd; padding: 8px; text-align: right; background-color: #d4edda;"><strong>Total Pago:</strong></td>
                <td style="border: 1px solid #ddd; padding: 8px; text-align: right; background-color: #d4edda;"><strong>R$ <?php echo number_format($total_pago, 2, ',', '.'); ?></strong></td>
            </tr>
        </tfoot>
    </table>
    
    <div style="text-align: center; margin-top: 20px; font-size: 12px; color: #666;">
        <p style="margin: 5px 0;">Relatório gerado em <?php echo date('d/m/Y H:i:s'); ?></p>
    </div>
</div>

<!-- Estilos para Impressão -->
<style type="text/css" media="print">
    @page {
        size: portrait;
        margin: 2cm;
    }
    
    body * {
        visibility: hidden;
    }
    
    #printArea, #printArea * {
        visibility: visible;
    }
    
    #printArea {
        position: absolute;
        left: 0;
        top: 0;
        width: 100%;
        display: block !important;
    }
    
    .no-print {
        display: none !important;
    }
    
    /* Garantir que as cores de fundo sejam impressas */
    * {
        -webkit-print-color-adjust: exact !important;
        print-color-adjust: exact !important;
    }
</style>

<script>
$(document).ready(function() {
    $("#contasTable").DataTable({
        language: {
            url: "//cdn.datatables.net/plug-ins/1.11.5/i18n/pt-BR.json"
        },
        order: [[2, "asc"]], // Ordenar por data de vencimento
        pageLength: 10
    });
});

function limparFiltros() {
    window.location.href = "contas_receber.php";
}

// Função para imprimir
function imprimirRelatorio() {
    window.print();
}

function registrarPagamento(contaId, valor, descricao) {
    document.getElementById('conta_id').value = contaId;
    document.getElementById('valor').value = formatarMoeda(valor);
    document.getElementById('descricao').value = descricao;
    var modal = new bootstrap.Modal(document.getElementById('modalPagamento'));
    modal.show();
}

function formatarMoeda(valor) {
    return valor.toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });
}

function confirmarPagamento() {
    const formData = new FormData(document.getElementById('formPagamento'));
    
    fetch('processar_pagamento.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Pagamento registrado com sucesso!');
            window.location.reload();
        } else {
            alert('Erro ao registrar pagamento: ' + data.message);
        }
    })
    .catch(error => {
        alert('Erro ao processar pagamento: ' + error);
    });
}
</script>

<?php
$content = ob_get_clean();

// JavaScript adicional
$extra_js = '';

// Incluir o layout
require_once "includes/layout.php";
?>
