<?php

namespace App\Services;

use Dompdf\Dompdf;
use Dompdf\Options;

class CupomNaoFiscal
{
    protected $venda;
    protected $config;
    protected $tipo;

    public function __construct($venda, $config, $tipo = 0)
    {
        $this->venda = $venda;
        $this->config = $config;
        $this->tipo = $tipo;
    }

    public function render()
    {
        $options = new Options();
        $options->set('isRemoteEnabled', true);
        $options->set('isHtml5ParserEnabled', true);

        $dompdf = new Dompdf($options);
        $dompdf->loadHtml($this->getHtml());
        
        // Configura o papel para tamanho de bobina térmica
        // Configura para papel 80mm com margens mínimas
        $dompdf->setPaper(array(0, 0, 302, 1000));
        $dompdf->set_option('dpi', 150);
        $dompdf->set_option('defaultFont', 'Courier');
        
        $dompdf->render();
        
        return $dompdf->output();
    }

    protected function getHtml()
    {
        $html = '
        <html>
        <head>
            <meta charset="utf-8">
            <style>
                @page {
                    margin: 1mm;
                }
                body { 
                    font-family: monospace;
                    font-size: 11pt;
                    margin: 0;
                    padding: 0;
                    width: 100%;
                    line-height: 1.3;
                }
                .header {
                    text-align: center;
                    margin-bottom: 15px;
                    width: 100%;
                    padding: 0 0 10px 0;
                    border-bottom: 1px dashed #000;
                }
                .empresa-nome {
                    font-size: 14pt;
                    font-weight: bold;
                }
                .item {
                    margin: 3px 0;
                }
                .item-nome {
                    font-weight: bold;
                }
                .total {
                    margin-top: 15px;
                    border-top: 1px dashed #000;
                    padding-top: 10px;
                }
                .subtotal {
                    text-align: right;
                    margin: 8px 0;
                }
                .pagamento {
                    margin: 8px 0;
                    padding: 5px 0;
                    border-top: 1px dashed #000;
                    border-bottom: 1px dashed #000;
                }
                .centered {
                    text-align: center;
                }
                .divisor {
                    border-top: 1px dashed #000;
                    margin: 8px 0;
                }
                .cliente-info {
                    margin: 8px 0;
                    padding: 5px 0;
                }
            </style>
        </head>
        <body>';
        
        // Cabeçalho com dados da empresa
        $html .= '<div class="header">';
        $html .= '<div class="empresa-nome">' . $this->config->razao_social . '</div>';
        $html .= $this->config->endereco . ', ' . $this->config->numero . '<br>';
        if($this->config->complemento) {
            $html .= $this->config->complemento . '<br>';
        }
        $html .= $this->config->bairro . ' - ' . $this->config->municipio . '/' . $this->config->UF . '<br>';
        $html .= 'CEP: ' . $this->config->cep . '<br>';
        $html .= 'CNPJ: ' . $this->formataCNPJ($this->config->cnpj) . '<br>';
        if($this->config->ie) {
            $html .= 'IE: ' . $this->config->ie . '<br>';
        }
        $html .= '</div>';
        
        $html .= '<div class="centered">---------------------------------------</div>';
        $html .= '<div class="centered"><strong>COMPROVANTE NÃO FISCAL</strong></div>';
        $html .= '<div class="centered">---------------------------------------</div>';
        
        // Dados da venda
        $html .= '<div class="item">Nº Venda: ' . str_pad($this->venda->id, 6, "0", STR_PAD_LEFT) . '</div>';
        $html .= '<div class="item">Data: ' . date('d/m/Y H:i:s', strtotime($this->venda->created_at)) . '</div>';
        // Verifica se existe vendedor/funcionário associado
        if($this->venda->funcionario) {
            $html .= '<div class="item">Vendedor: ' . $this->venda->funcionario->nome . '</div>';
        }
        
        // Dados do cliente se houver
        if($this->venda->cliente) {
            $html .= '<div class="cliente-info">';
            $html .= '<strong>CLIENTE</strong><br>';
            $html .= $this->venda->cliente->nome . '<br>';
            if($this->venda->cliente->cpf_cnpj) {
                $html .= 'CPF/CNPJ: ' . $this->formataCPFCNPJ($this->venda->cliente->cpf_cnpj) . '<br>';
            }
            if($this->venda->cliente->telefone) {
                $html .= 'Fone: ' . $this->venda->cliente->telefone . '<br>';
            }
            $html .= '</div>';
            $html .= '<div class="divisor"></div>';
        }
        
        // Cabeçalho dos itens
        $html .= '<div class="divisor"></div>';
        $html .= '<table width="100%" style="font-size: 10pt; border-collapse: collapse; margin: 0; padding: 0;">';
        $html .= '<tr style="border-bottom: 1px dashed #000;">';
        $html .= '<td width="45%"><strong>ITEM</strong></td>';
        $html .= '<td width="20%"><strong>QTD</strong></td>';
        $html .= '<td width="15%"><strong>VL UN</strong></td>';
        $html .= '<td width="20%" align="right"><strong>TOTAL</strong></td>';
        $html .= '</tr>';
        
        // Itens da venda
        $i = 1;
        foreach ($this->venda->itens as $item) {
            $html .= '<tr>';
            $html .= '<td colspan="4" style="padding-top: 2px;">' . str_pad($i, 3, "0", STR_PAD_LEFT) . ' - ' . $item->produto->nome . '</td>';
            $html .= '</tr>';
            $html .= '<tr style="padding-bottom: 1px;">';
            $html .= '<td></td>';
            // Debug dos valores
            $html .= '<!-- DEBUG: ' . json_encode([
                'quantidade_raw' => $item->quantidade,
                'valor_raw' => $item->valor,
                'valor_produto' => $item->produto->valor ?? 'N/A',
                'preco_venda' => $item->produto->preco_venda ?? 'N/A'
            ]) . ' -->';

            // Tenta pegar o valor do produto se o valor do item estiver zerado
            $valor_unitario = floatval($item->valor ?: ($item->produto->valor ?? ($item->produto->preco_venda ?? 0)));
            $quantidade = floatval($item->quantidade ?: 1);
            $valor_total = $quantidade * $valor_unitario;
            
            $html .= '<td>' . number_format($quantidade, 3, ',', '.') . ' ' . ($item->produto->unidade_venda ?? 'UN') . '</td>';
            $html .= '<td>R$ ' . number_format($valor_unitario, 2, ',', '.') . '</td>';
            $html .= '<td align="right">R$ ' . number_format($valor_total, 2, ',', '.') . '</td>';
            $html .= '</tr>';
            $i++;
        }
        
        $html .= '</table>';
        
        // Subtotal e descontos
        // Calcula o subtotal manualmente
        $subtotal = 0;
        foreach ($this->venda->itens as $item) {
            $subtotal += (floatval($item->valor) * floatval($item->quantidade));
        }

        $html .= '<div class="subtotal">';
        $html .= 'Subtotal: R$ ' . number_format($subtotal, 2, ',', '.') . '<br>';
        if($this->venda->desconto > 0) {
            $html .= 'Desconto: R$ ' . number_format($this->venda->desconto, 2, ',', '.') . '<br>';
        }
        $html .= '</div>';
        
        // Total e Pagamento
        $html .= '<div class="total">';
        $html .= '<table width="100%" style="font-size: 10pt;">';
        $html .= '<tr>';
        $html .= '<td><strong>VALOR TOTAL</strong></td>';
        $html .= '<td align="right"><strong>R$ ' . number_format($subtotal - ($this->venda->desconto ?? 0), 2, ',', '.') . '</strong></td>';
        $html .= '</tr>';
        $html .= '<tr>';
        $html .= '<td>VALOR RECEBIDO</td>';
        $html .= '<td align="right">R$ ' . number_format(($this->venda->total + 10), 2, ',', '.') . '</td>';
        $html .= '</tr>';
        $html .= '<tr>';
        $html .= '<td>DESCONTOS</td>';
        $html .= '<td align="right">R$ ' . number_format($this->venda->desconto ?? 0, 2, ',', '.') . '</td>';
        $html .= '</tr>';
        $html .= '<tr>';
        $html .= '<td>TROCO</td>';
        $html .= '<td align="right">R$ 10,00</td>';
        $html .= '</tr>';
        $html .= '</table>';
        $html .= '</div>';

        $html .= '<div class="divisor"></div>';
        $html .= '<div style="text-align: left; margin-top: 3px;"><strong>FORMA DE PAGAMENTO</strong></div>';
        $html .= '<div style="text-align: left;">DINHEIRO</div>';
        
        // Rodapé
        $html .= '<div class="centered" style="margin-top: 40px; margin-bottom: 60px;">';
        $html .= '---------------------------------------<br>';
        $html .= '<div style="margin: 15px 0;"><strong>OBRIGADO PELA PREFERÊNCIA!</strong></div>';
        $html .= '<div style="margin: 8px 0;">Não tem valor fiscal</div>';
        $html .= '<div style="margin-top: 8px;">' . date('d/m/Y H:i:s') . '</div>';
        if($this->config->site) {
            $html .= $this->config->site;
        }
        $html .= '</div>';
        
        $html .= '</body></html>';
        
        return $html;
    }

    protected function formataCNPJ($cnpj) {
        return preg_replace("/^(\d{2})(\d{3})(\d{3})(\d{4})(\d{2})$/", "$1.$2.$3/$4-$5", $cnpj);
    }

    protected function formataCPFCNPJ($doc) {
        $doc = preg_replace("/[^0-9]/", "", $doc);
        if(strlen($doc) === 11) {
            return preg_replace("/^(\d{3})(\d{3})(\d{3})(\d{2})$/", "$1.$2.$3-$4", $doc);
        }
        return $this->formataCNPJ($doc);
    }
}
